#include "testutils.h"

void
test_main(void)
{
  struct dsa_public_key pub;
  struct dsa_private_key key;
  struct dsa_signature signature;
  struct dsa_params *params = (struct dsa_params *) &pub;

  dsa_public_key_init(&pub);
  dsa_private_key_init(&key);
  dsa_signature_init(&signature);

  mpz_set_str(pub.p,
	      "83d9a7c2ce2a9179f43cdb3bffe7de0f0eef26dd5dfae44d"
	      "531bc0de45634d2c07cb929b0dbe10da580070e6abfbb841"
	      "5c44bff570b8ad779df653aad97dc7bdeb815d7e88103e61"
	      "606ed3d8a295fbfd340d2d49e220833ebace5511e22c4f02"
	      "97ed351e9948fa848e9c8fadb7b47bcc47def4255b5e1d5e"
	      "10215b3b55a0b85f", 16);
  mpz_set_str(pub.q,
	      "8266e0deaf46020ba48d410ca580f3a978629b5d", 16);
  mpz_set_str(pub.g,
	      "30d34bb9f376bec947154afe4076bc7d359c9d32f5471ddb"
	      "be8d6a941c47fa9dc4f32573151dbb4aa59eb989b74ac36b"
	      "b6310a5e8b580501655d91f393daa193ae1303049b87febb"
	      "093dc0404b53b4c5da2463300f9c5b156d788c4ace8ecbb9"
	      "dd00c18d99537f255ac025d074d894a607cbe3023a1276ef"
	      "556916a33f7de543", 16);
  mpz_set_str(pub.y,
	      "64402048b27f39f404a546a84909c9c0e9e2dd153a849946"
	      "1062892598d30af27ae3cefc2b700fb6d077390a83bdcad7"
	      "8a1299487c9623bb62af0c85a3df9ef1ee2c0d66658e1fd3"
	      "283b5407f6cd30ee7e6154fad41a6a8b0f5c86c5accc1127"
	      "bf7c9a5d6badcb012180cb62a55c5e17d6d3528cdbe002cc"
	      "ee131c1b86867f7a", 16);
  mpz_set_str(key.x,
	      "56c6efaf878d06eef21dc070fab71da6ec1e30a6", 16);

  test_dsa_key(params, pub.y, key.x, 160);

  mpz_set_str(signature.r, "180342f8d4fb5bd0311ebf205bdee6e556014eaf", 16);
  mpz_set_str(signature.s, "392dc6566b2735531a8460966171464ef7ddfe12", 16);

  test_dsa160(&pub, &key, &signature);

  mpz_set_str(pub.p,
	      "fda45d8f1df8f2b84fb3cf9ae69f93b087d98bea282f643e"
	      "23472c5b57605952010e4c846d711f2783e8ad4e1447698e"
	      "2e328fdb1d411ccb0f3caef5b8fc0b9dcecfadf022ecc7de"
	      "5c153c8f10fe88d63abf7d296ad485dfd6eead595fc1c36b"
	      "8bd42e8668b55b2bb0f1a6aecbe678df504880de2481a5e4"
	      "97d1b7d92ee48ffeb083a1833094a0418ec0d914409c720c"
	      "87ea63c164ec448c471b574a8f88073ebeb44dc6d6b98260"
	      "46126f03022ff04dcb6a2381a09b0a227d3c57cfbfd48e4a"
	      "19cbb0a35242c9e234ebe105ae26cab01ede40aa2869fad8"
	      "6bff57a19ec87b8de294ca03269c268c10813f18169beac5"
	      "ac97c0e748ccb244282c50c670e1bccb", 16);
  mpz_set_str(pub.q,
	      "bd612630da4d930779a32546dc413efd299111b443c7355d"
	      "65d991163cc3cd9d", 16);
  mpz_set_str(pub.g,
	      "050c56e14adb03e47d3902852f5b21c96c28a2aa89619c8b"
	      "78a98aa5083700994f99184588d2cefaf2a3ea213dd2d084"
	      "0e682a52357d5fefaef44520622f021855744d638e792f21"
	      "89543f9770aa1960da4d7b325a37a2922b035c8da3d71543"
	      "5d7a6ddefc62e84fe76fecbbf9667c6a1781a84aa434548b"
	      "bdc315f2fb0a420d65c1f72911845b148e994660138052a1"
	      "fce1c6f933be155b2af8c0177277cd3b75d9477ddbcb77bc"
	      "f5cccea915a2f3750ba41f337edd44f768cb3d24b17e299d"
	      "5cebe5e78cbaf5ad41e815edfc71df3131bd5359c653a224"
	      "bd3ac6a27bad7efff11b24fad0109ee26e4df76fc99e150d"
	      "666a9294bab8a03f113d228bfad349f4", 16);
  mpz_set_str(pub.y,
	      "da7f9abb0b554afaa926c9cffa897239bfdbc58ed9981748"
	      "edb1e38f42dea0560a407a48b509a5cb460bf31dee9057a0"
	      "b41d468698fa82ff03c47e8f3f6564c74d6f1daa5f84ad25"
	      "b937317f861fa68c19e20d6b855e85cd94d5af95b968416e"
	      "6d43711f24d5497f018b7627d2bed25dc793ddb897fdcc34"
	      "5d183e43a80205483dea7a12185be3b185a7d84d3385b962"
	      "4485882722d177ccd8f49c5b519fb96b9b59fcfc63422f25"
	      "88fb8ff00bce46acb7c80d105c31414ecf5be0a0dad975bd"
	      "dcd83d6f063f9bce562fdd5b68e18fc2159dbb2457adc7a7"
	      "ee5bc0796eff370908f866a41b9a8873f89e1904925141f8"
	      "e574df25bd869f43a184a804e8ce5fcc", 16);
  mpz_set_str(key.x,
	      "39f84f88569da55c6bee7e18175b539ea9b7ee24fabd85a7"
	      "1fa8c93b7181545b", 16);

  test_dsa_key(params, pub.y, key.x, 256);

  mpz_set_str(signature.r,
	      "03fe95c9dbbe1be019d7914e45c37c70"
	      "0f499f559312a59f3bc5037f51d3f74c", 16);
  mpz_set_str(signature.s,
	      "839dbee8d30e6b0cc349528f900f30ee"
	      "6d4ce9864332d07c87b5a98bd75dbdbb", 16);

  test_dsa256(&pub, &key, &signature);

  /* Some of the test vectors from
     http://csrc.nist.gov/groups/STM/cavp/documents/dss/186-3dsatestvectors.zip
  */
  /* L=1024, N=160, SHA-1 */
  mpz_set_str(pub.p,
	      "a8f9cd201e5e35d892f85f80e4db2599a5676a3b1d4f1903"
	      "30ed3256b26d0e80a0e49a8fffaaad2a24f472d2573241d4"
	      "d6d6c7480c80b4c67bb4479c15ada7ea8424d2502fa01472"
	      "e760241713dab025ae1b02e1703a1435f62ddf4ee4c1b664"
	      "066eb22f2e3bf28bb70a2a76e4fd5ebe2d1229681b5b0643"
	      "9ac9c7e9d8bde283", 16);
  mpz_set_str(pub.q, "f85f0f83ac4df7ea0cdf8f469bfeeaea14156495", 16);
  mpz_set_str(pub.g,
	      "2b3152ff6c62f14622b8f48e59f8af46883b38e79b8c74de"
	      "eae9df131f8b856e3ad6c8455dab87cc0da8ac973417ce4f"
	      "7878557d6cdf40b35b4a0ca3eb310c6a95d68ce284ad4e25"
	      "ea28591611ee08b8444bd64b25f3f7c572410ddfb39cc728"
	      "b9c936f85f419129869929cdb909a6a3a99bbe0892163681"
	      "71bd0ba81de4fe33", 16);
  mpz_set_str(pub.y,
	      "313fd9ebca91574e1c2eebe1517c57e0c21b0209872140c5"
	      "328761bbb2450b33f1b18b409ce9ab7c4cd8fda3391e8e34"
	      "868357c199e16a6b2eba06d6749def791d79e95d3a4d09b2"
	      "4c392ad89dbf100995ae19c01062056bb14bce005e8731ef"
	      "de175f95b975089bdcdaea562b32786d96f5a31aedf75364"
	      "008ad4fffebb970b", 16);
  mpz_set_str(signature.r, "50ed0e810e3f1c7cb6ac62332058448bd8b284c0", 16);
  mpz_set_str(signature.s, "c6aded17216b46b7e4b6f2a97c1ad7cc3da83fde", 16);

  test_dsa_verify(params, pub.y, &nettle_sha1,
		  SHEX("3b46736d559bd4e0c2c1b2553a33ad3c6cf23cac998d3d0c"
		       "0e8fa4b19bca06f2f386db2dcff9dca4f40ad8f561ffc308"
		       "b46c5f31a7735b5fa7e0f9e6cb512e63d7eea05538d66a75"
		       "cd0d4234b5ccf6c1715ccaaf9cdc0a2228135f716ee9bdee"
		       "7fc13ec27a03a6d11c5c5b3685f51900b1337153bc6c4e8f"
		       "52920c33fa37f4e7"),
		  &signature);

  mpz_set_str(pub.y,
	      "29bdd759aaa62d4bf16b4861c81cf42eac2e1637b9ecba51"
	      "2bdbc13ac12a80ae8de2526b899ae5e4a231aef884197c94"
	      "4c732693a634d7659abc6975a773f8d3cd5a361fe2492386"
	      "a3c09aaef12e4a7e73ad7dfc3637f7b093f2c40d6223a195"
	      "c136adf2ea3fbf8704a675aa7817aa7ec7f9adfb2854d4e0"
	      "5c3ce7f76560313b", 16);
  mpz_set_str(signature.r, "a26c00b5750a2d27fe7435b93476b35438b4d8ab", 16);
  mpz_set_str(signature.s, "61c9bfcb2938755afa7dad1d1e07c6288617bf70", 16);

  test_dsa_verify(params, pub.y, &nettle_sha1,
		  SHEX("d2bcb53b044b3e2e4b61ba2f91c0995fb83a6a97525e6644"
		       "1a3b489d9594238bc740bdeea0f718a769c977e2de003877"
		       "b5d7dc25b182ae533db33e78f2c3ff0645f2137abc137d4e"
		       "7d93ccf24f60b18a820bc07c7b4b5fe08b4f9e7d21b256c1"
		       "8f3b9d49acc4f93e2ce6f3754c7807757d2e1176042612cb"
		       "32fc3f4f70700e25"),
		  &signature);

  /* L=1024, N=160, SHA-224 */
  mpz_set_str(pub.p,
	      "8b9b32f5ba38faad5e0d506eb555540d0d7963195558ca30"
	      "8b7466228d92a17b3b14b8e0ab77a9f3b2959a09848aa69f"
	      "8df92cd9e9edef0adf792ce77bfceccadd9352700ca5faec"
	      "f181fa0c326db1d6e5d352458011e51bd3248f4e3bd7c820"
	      "d7e0a81932aca1eba390175e53eada197223674e3900263e"
	      "90f72d94e7447bff", 16);
  mpz_set_str(pub.q, "bc550e965647fb3a20f245ec8475624abbb26edd", 16);
  mpz_set_str(pub.g,
	      "11333a931fba503487777376859fdc12f7c687b0948ae889"
	      "d287f1b7a712ad220ae4f1ce379d0dbb5c9abf419621f005"
	      "fc123c327e5055d1850634c36d397e689e111d598c1c3636"
	      "b940c84f42f436846e8e7fcad9012ceda398720f32fffd1a"
	      "45ab6136ce417069207ac140675b8f86dd063915ae6f62b0"
	      "cec729fbd509ac17", 16);
  mpz_set_str(pub.y,
	      "7e339f3757450390160e02291559f30bed0b2d758c5ccc2d"
	      "8d456232bb435ae49de7e7957e3aad9bfdcf6fd5d9b6ee3b"
	      "521bc2229a8421dc2aa59b9952345a8fc1de49b348003a9b"
	      "18da642d7f6f56e3bc665131ae9762088a93786f7b4b72a4"
	      "bcc308c67e2532a3a5bf09652055cc26bf3b18833598cffd"
	      "7011f2285f794557", 16);
  mpz_set_str(signature.r, "afee719e7f848b54349ccc3b4fb26065833a4d8e", 16);
  mpz_set_str(signature.s, "734efe992256f31325e749bc32a24a1f957b3a1b", 16);
  test_dsa_verify(params, pub.y, &nettle_sha224,
		  SHEX("fb2128052509488cad0745ed3e6312850dd96ddaf791f1e6"
		       "24e22a6b9beaa65319c325c78ef59cacba0ccfa722259f24"
		       "f92c17b77a8f6d8e97c93d880d2d8dbbbedcf6acefa06b0e"
		       "476ca2013d0394bd90d56c10626ef43cea79d1ef0bc7ac45"
		       "2bf9b9acaef70325e055ac006d34024b32204abea4be5faa"
		       "e0a6d46d365ed0d9"),
		  &signature);

  /* L=1024, N=160, SHA-256 */
  mpz_set_str(pub.p,
	      "cba13e533637c37c0e80d9fcd052c1e41a88ac325c4ebe13"
	      "b7170088d54eef4881f3d35eae47c210385a8485d2423a64"
	      "da3ffda63a26f92cf5a304f39260384a9b7759d8ac1adc81"
	      "d3f8bfc5e6cb10efb4e0f75867f4e848d1a338586dd0648f"
	      "eeb163647ffe7176174370540ee8a8f588da8cc143d939f7"
	      "0b114a7f981b8483", 16);
  mpz_set_str(pub.q, "95031b8aa71f29d525b773ef8b7c6701ad8a5d99", 16);
  mpz_set_str(pub.g,
	      "45bcaa443d4cd1602d27aaf84126edc73bd773de6ece15e9"
	      "7e7fef46f13072b7adcaf7b0053cf4706944df8c4568f26c"
	      "997ee7753000fbe477a37766a4e970ff40008eb900b9de4b"
	      "5f9ae06e06db6106e78711f3a67feca74dd5bddcdf675ae4"
	      "014ee9489a42917fbee3bb9f2a24df67512c1c35c97bfbf2"
	      "308eaacd28368c5c", 16);
  mpz_set_str(pub.y,
	      "4cd6178637d0f0de1488515c3b12e203a3c0ca652f2fe30d"
	      "088dc7278a87affa634a727a721932d671994a958a0f8922"
	      "3c286c3a9b10a96560542e2626b72e0cd28e5133fb57dc23"
	      "8b7fab2de2a49863ecf998751861ae668bf7cad136e6933f"
	      "57dfdba544e3147ce0e7370fa6e8ff1de690c51b4aeedf04"
	      "85183889205591e8", 16);
  mpz_set_str(signature.r, "76683a085d6742eadf95a61af75f881276cfd26a", 16);
  mpz_set_str(signature.s, "3b9da7f9926eaaad0bebd4845c67fcdb64d12453", 16);
  test_dsa_verify(params, pub.y, &nettle_sha256,
		  SHEX("812172f09cbae62517804885754125fc6066e9a902f9db20"
		       "41eeddd7e8da67e4a2e65d0029c45ecacea6002f9540eb10"
		       "04c883a8f900fd84a98b5c449ac49c56f3a91d8bed3f08f4"
		       "27935fbe437ce46f75cd666a0707265c61a096698dc2f36b"
		       "28c65ec7b6e475c8b67ddfb444b2ee6a984e9d6d15233e25"
		       "e44bd8d7924d129d"),
		  &signature);

  /* L=2048, N=224, SHA-1 */
  mpz_set_str(pub.p,
	      "f2d39ed3062b13c916273600a0f2a029e86d7a4b9217b4f1"
	      "815bf2b24d9710a57ab33f997294b014585b8d0198dfdccb"
	      "cd75314da5ff85aa344b45adaeaa979b51a312a7bfa94472"
	      "fb633f1a6f156bb4458867dfd38403f06b851f00fe2d3484"
	      "077bded71ab7513d04a140220575fb693395480e4c8402b7"
	      "a46cec2d37a778c305accd1f13e9f62e865315f4b22cc467"
	      "c8986ec8e4961ddf810566b0c4ee369ac6aa15e43f474400"
	      "5826f5bde8071a19e30b6909aac4b3d174237270dad02799"
	      "d09b8a2cc5f22e66894b5422228b2c234f11f5a771c5b89c"
	      "f465a2acecbbeeaa1725fe8f9b59422be8991052cb556ddf"
	      "2c8ce8fa9206dbf39feadc194e00f8e5", 16);
  mpz_set_str(pub.q,
	      "8000000000000000c118f49835e4ef733c4d15800fcf059e884d31b1", 16);
  mpz_set_str(pub.g,
	      "e3a93c09da6f560e4d483a382a4c546f2335c36a4c35ac14"
	      "63c08a3e6dd415df56fdc537f25fd5372be63e4f5300780b"
	      "782f1acd01c8b4eb33414615fd0ea82573acba7ef83f5a94"
	      "3854151afc2d7dfe121fb8cd03335b065b549c5dcc606be9"
	      "052483bc284e12ac3c8dba09b426e08402030e70bc1cc2bf"
	      "8957c4ba0630f3f32ad689389ac47443176063f247d9e229"
	      "6b3ea5b5bc2335828ea1a080ed35918dee212fd031279d1b"
	      "894f01afec523833669eac031a420e540ba1320a59c424a3"
	      "e5849a460a56bcb001647885b1433c4f992971746bfe2977"
	      "ce7259c550b551a6c35761e4a41af764e8d92132fcc0a59d"
	      "1684eab90d863f29f41cf7578faa908c", 16);
  mpz_set_str(pub.y,
	      "289ff18c32a56bb0b8839370647683a38a5a7e291410b932"
	      "07212adc8088d30f93e9e4abc523f3d46936e7d5c90d8874"
	      "2b36afd37563408f15c8c1a4f7ac24bf05f01008ffee70c8"
	      "825d57c3a9308bad8a095af2b53b2dda3cbed846d95e301e"
	      "b9b84766415d11f6c33209a0d28571096ab04a79aa0dc465"
	      "997529686b68e887cd8a205c2dc8195aef0422eba9979f54"
	      "9ac85548e419413643b7244361153ada1480d238cd00dc16"
	      "527938955548dd5d027ded1029eeeb8ed6c61b4cd59341d8"
	      "b15466e9da890a989996f4d7691e6072de136af28b5874bf"
	      "08bd1f8a60cfb1c00888132909f515e04bce81b02951aa41"
	      "baac68ffdb8c5dc77a1d32d8f2c10dd7", 16);
  mpz_set_str(signature.r,
	      "45df2f423e94bf155dd4e1d9e63f315ea606dd38527d4cf6328738c8", 16);
  mpz_set_str(signature.s,
	      "59b3e8efa5bc0ccbf4a3cbb6515c4b9bf784cfacdcc101dc9f81d31f", 16);
  test_dsa_verify(params, pub.y, &nettle_sha1,
		  SHEX("edc6fd9b6c6e8a59f283016f7f29ee16deeaa609b5737927"
		       "162aef34fed985d0bcb550275637ba67831a2d4efccb3529"
		       "6dfe730f4a0b4f4728d1d7d1bb8f4a36238a5c94311fa113"
		       "4a93a6b4de39c085e9f60ae4e237c0416d58042bb36baa38"
		       "cba8c896295b745d5376fd8ce42eb6ee5a1b38f87716b265"
		       "b76e58cfb24a9170"),
		  &signature);
  /* L=2048, N=224, SHA-224 */
  mpz_set_str(pub.p,
	      "aa815c9db1c4d3d2773c7d0d4d1da75ecfc4a39e97d5fa19"
	      "1ffec8b1490a290ce335e5ce87ea620a8a17de0bb64714e2"
	      "ec840bf00e6ebdb4ffb4e324ca07c3c8717309af1410362a"
	      "772c9add838b2b0cae1e90ab448adabdacd2e5df59c4187a"
	      "32a23719d6c57e9400885383bf8f066f23b941920d54c35b"
	      "4f7cc5044f3b40f17046956307b748e840732844d00a9ce6"
	      "ec5714293b6265147f15c67f4be38b082b55fdeadb612468"
	      "9fb76f9d25cc28b8eaa98b562d5c1011e0dcf9b39923240d"
	      "332d89dc9603b7bddd0c70b83caa2905631b1c83cabbae6c"
	      "0c0c2efe8f58131ed8351bf93e875f6a73a93cbad470141a"
	      "2687fbacf2d71c8ddee971ad660729ad", 16);
  mpz_set_str(pub.q,
	      "ea347e90be7c2875d1fe1db622b4763837c5e27a6037310348c1aa11", 16);
  mpz_set_str(pub.g,
	      "2042094ccbc8b8723fc928c12fda671b83295e99c743576f"
	      "44504be1186323319b5002d24f173df909ea241d6ea52899"
	      "04ee4636204b2fbe94b068fe093f7962579549551d3af219"
	      "ad8ed19939eff86bcec834de2f2f78596e89e7cb52c524e1"
	      "77098a56c232eb1f563aa84bc6b026deee6ff51cb441e080"
	      "f2dafaea1ced86427d1c346be55c66803d4b76d133cd445b"
	      "4c3482fa415023463c9bf30f2f784223e26057d3aa0d7fbb"
	      "660630c52e49d4a0325c7389e072aa349f13c966e159752f"
	      "bb71e9336890f93243fa6e72d299365ee5b3fe266ebf1110"
	      "568fee4425c847b50210bd484b97431a42856adca3e7d1a9"
	      "c9c675c7e266918320dd5a78a48c48a9", 16);
  mpz_set_str(pub.y,
	      "1ae10c786ad0902c5c685dae5c7121418a377b888b5f2f2b"
	      "c76623570fd62bcb190b471ad5359c5f062f8819289e956d"
	      "8aa6f90d1f8cf1ee72d3a1bdfd56c478dc29a19c4569b5a6"
	      "0e3a8f34f60656eac5b25dde5514a5c67b675423204f6cca"
	      "f0990617cc7355b9d3ed868978a252020a769ed59a6edaa6"
	      "efe3377eef45f3f6f3e64179cc7db8b143fb835c5d71bfcf"
	      "a1e2a9049bccf7fe9ab57546220fe3f4b7521c861739d138"
	      "507e81a46a6993605441dcb90d6ee4afbc42cabe90a25444"
	      "4968109d7edd9694a023239f1d56175dd1fac115915e24fa"
	      "b563f4fc3f269bed2f300832d112596485a711417aa73bb4"
	      "ac72a651a1fa5baed3636c720d397008", 16);
  mpz_set_str(signature.r,
	      "65102e8f64ecb11f06017b1a0c0def3c29897c277c4a948b1f4da6b9", 16);
  mpz_set_str(signature.s,
	      "21ad0abb27bd3c21166cb96aef70c0dbd5f3079cab0dd543d4125bd1", 16);
  test_dsa_verify(params, pub.y, &nettle_sha224,
		  SHEX("e920fc1610718f2b0213d301c0092a51f3c6b0107bbbd824"
		       "3a9689c044e2d142f202d9d195a5faef4be5acadc9ff6f7d"
		       "2261e58b517139bcb9489b110423c2e59eb181294ffdae8a"
		       "ad0e624fab974c97f9f5e7dc19d678a9cb3429cf05ec5090"
		       "72856f5adfec6e29bafe8e5ba95593e612843e343111d88a"
		       "1eaff7dc0a2e277f"),
		  &signature);

  /* mod = L=2048, N=224, SHA-256 */
  mpz_set_str(pub.p,
	      "a4c7eaab42c4c73b757770916489f17cd50725cd0a4bc4e1"
	      "cf67f763b8c1de2d6dab9856baafb008f365b18a42e14dc5"
	      "1f350b88eca0209c5aa4fd71a7a96c765f5901c21e720570"
	      "d7837bec7c76d2e49344731ca39405d0a879b9e0dcd1a812"
	      "5fd130ec1e783e654b94e3002e6b629e904ab3877867720c"
	      "bd54b4270a9e15cd028c7cc796f06c272a660951928fdbeb"
	      "2dca061b41e932257305742ff16e2f429191d5e5f1a6ddf6"
	      "e78c5d7722cff80a9c0bd5c8d7aeba8c04438992b075e307"
	      "c1534c49ad380f477f5f7987dc172c161dca38dcaf3fb384"
	      "6c72c9119a5299adc748951b3dce0d00d4a9013800b20082"
	      "03b72465bc6a84ae059a30c4522dea57", 16);
  mpz_set_str(pub.q,
	      "ce89fe332b8e4eb3d1e8ddcea5d163a5bc13b63f16993755427aef43", 16);
  mpz_set_str(pub.g,
	      "8c465edf5a180730291e080dfc5385397a5006450dba2efe"
	      "0129264fbd897bb5579ca0eab19aa278220424724b4f2a6f"
	      "6ee6328432abf661380646097233505339c5519d357d7112"
	      "b6eec938b85d5aa75cc2e38092f0a530acb54e50fe82c4d5"
	      "62fb0f3036b80b30334023ebbe6637a0010b00c7db863711"
	      "68563671e1e0f028aedbd45d2d572621a609982a073e51aa"
	      "e27707afbeef29e2ecee84d7a6d5da382be3a35f42b6c668"
	      "49202ab19d025b869d08776476d1ab981475ad2ad2f3e6fd"
	      "07e30696d90a626816df60d6ca7afd7b482f942f83b45cc8"
	      "2933731f87faee320900f2aa3e70b1867e1430e40be67c07"
	      "f9290299ef067b8b24a7515b3f992c07", 16);
  mpz_set_str(pub.y,
	      "748a40237211a2d9852596e7a891f43d4eb0ee48826c9cfb"
	      "336bbb68dbe5a5e16b2e1271d4d13de03644bb85ef6be523"
	      "a4d4d88415bcd596ba8e0a3c4f6439e981ed013d7d9c7033"
	      "6febf7d420cfed02c267457bb3f3e7c82145d2af54830b94"
	      "2ec74a5d503e4226cd25dd75decd3f50f0a858155d7be799"
	      "410836ddc559ce99e1ae513808fdaeac34843dd7258f16f6"
	      "7f19205f6f139251a4186da8496d5e90d3fecf8ed10be6c2"
	      "5ff5eb33d960c9a8f4c581c8c724ca43b761e9fdb5af66bf"
	      "fb9d2ebb11a6b504a1fbe4f834ecb6ac254cab513e943b9a"
	      "953a7084b3305c661bfad434f6a835503c9ade7f4a57f5c9"
	      "65ec301ecde938ee31b4deb038af97b3", 16);
  mpz_set_str(signature.r,
	      "9c5fa46879ddaf5c14f07dfb5320715f67a6fec179e3ad53342fb6d1", 16);
  mpz_set_str(signature.s,
	      "c3e17e7b3c4d0ac8d49f4dd0f04c16a094f42da0afcc6c90f5f1bbc8", 16);
  test_dsa_verify(params, pub.y, &nettle_sha256,
		  SHEX("cec8d2843dee7cb5f9119b75562585e05c5ce2f4e6457e9b"
		       "cc3c1c781ccd2c0442b6282aea610f7161dcede176e77486"
		       "1f7d2691be6c894ac3ebf80c0fab21e52a3e63ae0b350257"
		       "62ccd6c9e1fecc7f9fe00aa55c0c3ae33ae88f66187f9598"
		       "eba9f863171f3f56484625bf39d883427349b8671d9bb7d3"
		       "96180694e5b546ae"),
		  &signature);
  
  /* L=2048, N=256, SHA-1 */
  mpz_set_str(pub.p,
	      "c1a59d215573949e0b20a974c2edf2e3137ff2463062f75f"
	      "1d13df12aba1076bb2d013402b60af6c187fb0fa362167c9"
	      "76c2617c726f9077f09e18c11b60f65008825bd6c02a1f57"
	      "d3eb0ad41cd547de43d87f2525f971d42b306506e7ca03be"
	      "63b35f4ada172d0a06924440a14250d7822ac2d5aeafed46"
	      "19e79d4158a7d5eb2d9f023db181a8f094b2c6cb87cb8535"
	      "416ac19813f07144660c557745f44a01c6b1029092c129b0"
	      "d27183e82c5a21a80177ee7476eb95c466fb472bd3d2dc28"
	      "6ce25847e93cbfa9ad39cc57035d0c7b64b926a9c7f5a7b2"
	      "bc5abcbfbdc0b0e3fede3c1e02c44afc8aefc7957da07a0e"
	      "5fd12339db8667616f62286df80d58ab", 16);
  mpz_set_str(pub.q,
	      "8000000000000000000000001bd62c65e8b87c89797f8f0c"
	      "bfa55e4a6810e2c7", 16);
  mpz_set_str(pub.g,
	      "aea5878740f1424d3c6ea9c6b4799615d2749298a17e2620"
	      "7f76cef340ddd390e1b1ad6b6c0010ad015a103342ddd452"
	      "cac024b36e42d9b8ed52fafae7a1d3ce9e4b21f910d1356e"
	      "b163a3e5a8184c781bf14492afa2e4b0a56d8884fd01a628"
	      "b9662739c42e5c5795ade2f5f27e6de1d963917ce8806fc4"
	      "0d021cd87aa3aa3a9e4f0c2c4c45d2959b2578b2fb1a2229"
	      "c37e181059b9d5e7b7862fa82e2377a49ed0f9dca820a581"
	      "4079dd6610714efaf8b0cc683d8e72e4c884e6f9d4946b3e"
	      "8d4cbb92adbbe7d4c47cc30be7f8c37ca81883a1aac68600"
	      "59ff4640a29ccae73de20b12e63b00a88b2ee9ba94b75eb4"
	      "0a656e15d9ec83731c85d0effcb9ef9f", 16);
  mpz_set_str(pub.y,
	      "880e17c4ae8141750609d8251c0bbd7acf6d0b460ed3688e"
	      "9a5f990e6c4b5b00875da750e0228a04102a35f57e74b8d2"
	      "f9b6950f0d1db8d302c5c90a5b8786a82c68ff5b17a57a75"
	      "8496c5f8053e4484a253d9942204d9a1109f4bd2a3ec311a"
	      "60cf69c685b586d986f565d33dbf5aab7091e31aa4102c4f"
	      "4b53fbf872d700156465b6c075e7f778471a23502dc0fee4"
	      "1b271c837a1c26691699f3550d060a331099f64837cddec6"
	      "9caebf51bf4ec9f36f2a220fe773cb4d3c02d0446ddd4613"
	      "3532ef1c3c69d432e303502bd05a75279a7809a742ac4a78"
	      "72b07f1908654049419350e37a95f2ef33361d8d8736d408"
	      "3dc14c0bb972e14d4c7b97f3ddfccaef", 16);
  mpz_set_str(signature.r,
	      "363e01c564f380a27d7d23b207af3f961d48fc0995487f60"
	      "052775d724ab3d10", 16);
  mpz_set_str(signature.s,
	      "4916d91b2927294e429d537c06dd2463d1845018cca2873e"
	      "90a6c837b445fdde", 16);
  test_dsa_verify(params, pub.y, &nettle_sha1,
		  SHEX("de3605dbefde353cbe05e0d6098647b6d041460dfd4c0003"
		       "12be1afe7551fd3b93fed76a9763c34e004564b8f7dcacbd"
		       "99e85030632c94e9b0a032046523b7aacdf934a2dbbdcfce"
		       "efe66b4e3d1cb29e994ff3a4648a8edd9d58ed71f12399d9"
		       "0624789c4e0eebb0fbd5080f7d730f875a1f290749334cb4"
		       "05e9fd2ae1b4ed65"),
		  &signature);

  /* L=2048, N=256, SHA-224 */
  mpz_set_str(pub.p,
	      "d02276ebf3c22ffd666983183a47ae94c9bccbcbf95ddcb4"
	      "91d1f7ce643549199992d37c79e7b032d26ed031b6ba4489"
	      "f3125826fafb2726a98333ebd9abdde592d8693d9859536d"
	      "9cc3841a1d24e044d35aced6136256fc6d6b615cf4f4163a"
	      "a381eb2b4c480825a8eccc56d8ddcf5fe637e38ad9b2974b"
	      "d2cf68bf271e0d067d2465a8b6b660524f0082598945ada5"
	      "8ea649b9804eb4753408c2c59768c46abb82e3295f3d9ca4"
	      "69f84cc187f572dc4b5a3b39346ec839dfad6f07d6d1f0e2"
	      "15209bb0ecc05c767cf2e7943ac9cfb02eee1e9ef5946e8c"
	      "e88316b5e15fdcf95a132ef2e4bb0817136528cfa5dd9653"
	      "2f9c3abe5c421620edb6bcbd52234ca9", 16);
  mpz_set_str(pub.q,
	      "8000000012997e8285e4089708f528070c6d7af8a0bd0140"
	      "9e7a079cdb6fc5bb", 16);
  mpz_set_str(pub.g,
	      "778453049ef262147fed7b59b0ee6764607c51e7b5b5fc6f"
	      "ea7a7a7b1dd6bb283f4a9ae98efd3964b1556758cb15b2a5"
	      "3af8619e74d85898bec77d3b3f382494ae5961a13ffc745d"
	      "a386182291519800f99dd710e00aeb15adee088e2798ee2e"
	      "46f598526cf0f4667055d1ba009750041dc5cdd2725ff1d9"
	      "7dd340c8518af7671b87d39d67aeced84b66f84e0701efc8"
	      "2a5c9ef954ee576d24c385b14d63037f0d866fd424b4975b"
	      "dd5485ed740cb932e843f906683f7c7b2c74775d901c361b"
	      "847b519c0da699638da40bd736b783d2710b2c2cc26ef912"
	      "71bf4e2c1929f876e902e2057164223bc78d6a2b9f6c0c7a"
	      "7cb85922f7d6c4287ae23861f8128848", 16);
  mpz_set_str(pub.y,
	      "7bb31e98c7a0437f978a73d5dcfbdfbb09cc2499dfaf1eb5"
	      "256bccd6358cabb5f67d04a42823463b7e957f2b9213f1fa"
	      "8e5a98d614484701abb8c7d67641fe6ed06fa4527b493dda"
	      "b2e74640fde3de70da693f1db2b8e26417040af0eea6cab4"
	      "51a795a52e187d2ee241b93f65c86c6d66f45834cce165ac"
	      "5eb670d4f0095c23ce9757e3bdc636f991ee0073d90a0920"
	      "2edb35cc3ea1cf9adca1617fa0bffd9c126229a604a1d3bf"
	      "4931ddf0b9942dfc8a2f8c09fcc97032564a79ae1ebe1e2c"
	      "e49ff57839e7c43fa60b1603d15a450898aa4e4a1ee80657"
	      "94126d64f013367096a83686b9f158c33b10f5f3b36cf1f6"
	      "358b3f34f84b101dc26d3db68bcc95c8", 16);
  mpz_set_str(signature.r,
	      "059bee9e708b7f20c3f791a640edee964e0aa672893c4847"
	      "99715817b3a8f6d4", 16);
  mpz_set_str(signature.s,
	      "4bd41c84a724cc86e4f0194ec0fbf379e654d0d7f6a1f08b"
	      "d468139422a5c353", 16);
  test_dsa_verify(params, pub.y, &nettle_sha224,
		  SHEX("39f2d8d503aae8cd17854456ecfad49a18900d4375412bc6"
		       "89181ed9c2ccafea98dca689a72dc75e5367d3d3abfc2169"
		       "700d5891cff70f69d9aca093b061b9f5057f94636bc27831"
		       "15254344fb12e33b167272e198838a8728e7744ea9a2e824"
		       "8e34d5906e298302472637b879de91c1a6f9f331a5cf98a5"
		       "af29132990d27416"),
		  &signature);
  
  /* L=2048, N=256, SHA-256 */
  mpz_set_str(pub.p,
	      "a8adb6c0b4cf9588012e5deff1a871d383e0e2a85b5e8e03"
	      "d814fe13a059705e663230a377bf7323a8fa117100200bfd"
	      "5adf857393b0bbd67906c081e585410e38480ead51684dac"
	      "3a38f7b64c9eb109f19739a4517cd7d5d6291e8af20a3fbf"
	      "17336c7bf80ee718ee087e322ee41047dabefbcc34d10b66"
	      "b644ddb3160a28c0639563d71993a26543eadb7718f317bf"
	      "5d9577a6156561b082a10029cd44012b18de6844509fe058"
	      "ba87980792285f2750969fe89c2cd6498db3545638d5379d"
	      "125dccf64e06c1af33a6190841d223da1513333a7c9d7846"
	      "2abaab31b9f96d5f34445ceb6309f2f6d2c8dde06441e879"
	      "80d303ef9a1ff007e8be2f0be06cc15f", 16);
  mpz_set_str(pub.q,
	      "e71f8567447f42e75f5ef85ca20fe557ab0343d37ed09edc"
	      "3f6e68604d6b9dfb", 16);
  mpz_set_str(pub.g,
	      "5ba24de9607b8998e66ce6c4f812a314c6935842f7ab54cd"
	      "82b19fa104abfb5d84579a623b2574b37d22ccae9b3e415e"
	      "48f5c0f9bcbdff8071d63b9bb956e547af3a8df99e5d3061"
	      "979652ff96b765cb3ee493643544c75dbe5bb39834531952"
	      "a0fb4b0378b3fcbb4c8b5800a5330392a2a04e700bb6ed7e"
	      "0b85795ea38b1b962741b3f33b9dde2f4ec1354f09e2eb78"
	      "e95f037a5804b6171659f88715ce1a9b0cc90c27f35ef2f1"
	      "0ff0c7c7a2bb0154d9b8ebe76a3d764aa879af372f4240de"
	      "8347937e5a90cec9f41ff2f26b8da9a94a225d1a913717d7"
	      "3f10397d2183f1ba3b7b45a68f1ff1893caf69a827802f7b"
	      "6a48d51da6fbefb64fd9a6c5b75c4561", 16);
  mpz_set_str(pub.y,
	      "5a55dceddd1134ee5f11ed85deb4d634a3643f5f36dc3a70"
	      "689256469a0b651ad22880f14ab85719434f9c0e407e60ea"
	      "420e2a0cd29422c4899c416359dbb1e592456f2b3cce2332"
	      "59c117542fd05f31ea25b015d9121c890b90e0bad033be13"
	      "68d229985aac7226d1c8c2eab325ef3b2cd59d3b9f7de7db"
	      "c94af1a9339eb430ca36c26c46ecfa6c5481711496f624e1"
	      "88ad7540ef5df26f8efacb820bd17a1f618acb50c9bc197d"
	      "4cb7ccac45d824a3bf795c234b556b06aeb9291734532520"
	      "84003f69fe98045fe74002ba658f93475622f76791d9b262"
	      "3d1b5fff2cc16844746efd2d30a6a8134bfc4c8cc80a4610"
	      "7901fb973c28fc553130f3286c1489da", 16);
  mpz_set_str(signature.r,
	      "633055e055f237c38999d81c397848c38cce80a55b649d9e"
	      "7905c298e2a51447", 16);
  mpz_set_str(signature.s,
	      "2bbf68317660ec1e4b154915027b0bc00ee19cfc0bf75d01"
	      "930504f2ce10a8b0", 16);
  test_dsa_verify(params, pub.y, &nettle_sha256,
		  SHEX("4e3a28bcf90d1d2e75f075d9fbe55b36c5529b17bc3a9cca"
		       "ba6935c9e20548255b3dfae0f91db030c12f2c344b3a29c4"
		       "151c5b209f5e319fdf1c23b190f64f1fe5b330cb7c8fa952"
		       "f9d90f13aff1cb11d63181da9efc6f7e15bfed4862d1a62c"
		       "7dcf3ba8bf1ff304b102b1ec3f1497dddf09712cf323f561"
		       "0a9d10c3d9132659"),
		  &signature);
  
  /* L=2048, N=256, SHA-384 */
  mpz_set_str(pub.p,
	      "a6167c16fff74e29342b8586aed3cd896f7b1635a2286ff1"
	      "6fdff41a06317ca6b05ca2ba7c060ad6db1561621ccb0c40"
	      "b86a03619bfff32e204cbd90b79dcb5f86ebb493e3bd1988"
	      "d8097fa23fa4d78fb3cddcb00c466423d8fa719873c37645"
	      "fe4eecc57171bbedfe56fa9474c96385b8ba378c79972d7a"
	      "aae69a2ba64cde8e5654f0f7b74550cd3447e7a472a33b40"
	      "37db468dde31c348aa25e82b7fc41b837f7fc226a6103966"
	      "ecd8f9d14c2d3149556d43829f137451b8d20f8520b0ce8e"
	      "3d705f74d0a57ea872c2bdee9714e0b63906cddfdc28b677"
	      "7d19325000f8ed5278ec5d912d102109319cba3b6469d467"
	      "2909b4f0dbeec0bbb634b551ba0cf213", 16);
  mpz_set_str(pub.q,
	      "8427529044d214c07574f7b359c2e01c23fd97701b328ac8"
	      "c1385b81c5373895", 16);
  mpz_set_str(pub.g,
	      "6fc232415c31200cf523af3483f8e26ace808d2f1c6a8b86"
	      "3ab042cc7f6b7144b2d39472c3cb4c7681d0732843503d8f"
	      "858cbe476e6740324aaa295950105978c335069b919ff9a6"
	      "ff4b410581b80712fe5d3e04ddb4dfd26d5e7fbca2b0c52d"
	      "8d404343d57b2f9b2a26daa7ece30ceab9e1789f9751aaa9"
	      "387049965af32650c6ca5b374a5ae70b3f98e053f51857d6"
	      "bbb17a670e6eaaf89844d641e1e13d5a1b24d053dc6b8fd1"
	      "01c624786951927e426310aba9498a0042b3dc7bbc59d705"
	      "f80d9b807de415f7e94c5cf9d789992d3bb8336d1d808cb8"
	      "6b56dde09d934bb527033922de14bf307376ab7d22fbcd61"
	      "6f9eda479ab214a17850bdd0802a871c", 16);
  mpz_set_str(pub.y,
	      "5ca7151bca0e457bbc46f59f71d81ab16688dc0eb7e4d17b"
	      "166c3326c5b12c5bdebb3613224d1a754023c50b83cb5ecc"
	      "139096cef28933b3b12ca31038e4089383597c59cc27b902"
	      "be5da62cae7da5f4af90e9410ed1604082e2e38e25eb0b78"
	      "dfac0aeb2ad3b19dc23539d2bcd755db1cc6c9805a7dd109"
	      "e1c98667a5b9d52b21c2772121b8d0d2b246e5fd3da80728"
	      "e85bbf0d7067d1c6baa64394a29e7fcbf80842bd4ab02b35"
	      "d83f59805a104e0bd69d0079a065f59e3e6f21573a00da99"
	      "0b72ea537fa98caaa0a58800a7e7a0623e263d4fca65ebb8"
	      "eded46efdfe7db92c9ebd38062d8f12534f015b186186ee2"
	      "361d62c24e4f22b3e95da0f9062ce04d", 16);
  mpz_set_str(signature.r,
	      "4fd8f25c059030027381d4167c3174b6be0088c15f0a573d"
	      "7ebd05960f5a1eb2", 16);
  mpz_set_str(signature.s,
	      "5f56869cee7bf64fec5d5d6ea15bb1fa1169003a87eccc16"
	      "21b90a1b892226f2", 16);
  test_dsa_verify(params, pub.y, &nettle_sha384,
		  SHEX("8c78cffdcf25d8230b835b30512684c9b252115870b603d1"
		       "b4ba2eb5d35b33f26d96b684126ec34fff67dfe5c8c856ac"
		       "fe3a9ff45ae11d415f30449bcdc3bf9a9fb5a7e48afeaba6"
		       "d0b0fc9bce0197eb2bf7a840249d4e550c5a25dc1c71370e"
		       "67933edad2362fae6fad1efba5c08dc1931ca2841b44b78c"
		       "0c63a1665ffac860"),
		  &signature);

  /* L=3072, N=256, SHA-1 */
  mpz_set_str(pub.p,
	      "fd5a6c56dd290f7dd84a29de17126eb4e4487b3eff0a44ab"
	      "e5c59792d2e1200b9c3db44d528b9f7d2248032e4ba0f7bf"
	      "c4fafc706be511db2276c0b7ecffd38da2e1c2f237a75390"
	      "c1e4d3239cba8e20e55840ecb05df5f01a1b6977ad1906f2"
	      "cb544ccfb93b901ad0966b1832ad2dab526244a3156c905c"
	      "01ac51cb73b9dcd9860d56175a425d846485d9b1f44a8a0c"
	      "2578e6cf61947bc1a1392fdd320b16a9d70455fe436f2d47"
	      "ded8e8e605f7486eb578ea7fc4ffd13c07f9996af159fd41"
	      "1e9451403278dd1141a8c926b35c96384bbd6bee09c46f44"
	      "c36b1ffc7197f5e925dbe0544a68e6ab8c18e426a466b392"
	      "f9c27dd79fefa9ca163cc5a375539a8559f277f657a535d1"
	      "964c6a5e91683ef5698ebaa01ef818dbf72cb04c3ff092d1"
	      "88866f25cd405108f566b087f73d2d5beb51fac6de84ae51"
	      "61a66af9602c7e4bfc146f4820bdfc092faeac69133e4a08"
	      "a5b202a12498a22e57bad54674ed4b510109d52b5f74e70e"
	      "1f6f82161718cd4cf00cc9f1958acc8bddcdfbd1fbe46cd1", 16);
  mpz_set_str(pub.q,
	      "800000000000000000000000334a26dd8f49c6811ce81bb1"
	      "342b06e980f64b75", 16);
  mpz_set_str(pub.g,
	      "99ab030a21a5c9818174872167641c81c1e03c9b274cfbc2"
	      "7bc472542927766de5fa0539b3b73f3f16ac866a9aec8b44"
	      "5ded97fbff08834ed98c77e7fc89e5dc657bef766ff7fbf8"
	      "e76873e17bee412762d56fe1141760ab4d25bafd4b6ef25b"
	      "49a3506632d1f8e10770930760ec1325932c5a4baf9e9015"
	      "4264ddf442ec5c41fed95d11525151dbcfb3758149bad81c"
	      "62b9cff7816b8f953b8b7c022590d1584e921dc955f5328a"
	      "c72983ed5cf0d04056fe0d531e62f8f6c9ab3c0fcd44e148"
	      "60b7311d2561c77c1d32f6c69dc8f77968c9d881ad9db5e0"
	      "c114fda8628bca0335eb7fb9e15e625aabab58fc01194c81"
	      "bf6fb2ce54077b82250e57c6a7b25deb6ee39d4b686a5c30"
	      "7a7612b2d85ee92512413dea297e44f317be7ceb70a3328a"
	      "f0b401001a418562b8ffe4e9771b4b4a8e0b40c791349d5d"
	      "4e459fe620a1a2fc72e2f6ca28567d4c2632bbde1b49864c"
	      "06bb12619f132c1da8f571ef613eac739f66ab3914cb3fa1"
	      "ab86e05e5082ebaa24ebeea4cf51beefc27df512fe3fee7d", 16);
  mpz_set_str(pub.y,
	      "e7c2ee18c3aa362c0182c6a56c2584628083c73e045beda8"
	      "d653690c9c2f6544edf9702c57c455273905336a5f517110"
	      "7a313cd7d0b0f50f8d3342c60219f22a9023394059d05f46"
	      "4c4496d55dab6eb0898527ff4cf5678e7b5bfb5e18d92c4a"
	      "9d73288cce14530fc4702f6d0397ec39a880c4a72d358730"
	      "c56633386ede028023c1791f3164d1574e7823c79b8a3ca1"
	      "343ea166ba6f02b7ff7e9ef2198db107f7cc159f3b6a1c00"
	      "a78c355c566deb0ac6fde3f633cb9177a1fbc6c1766ca021"
	      "d5fec470101abb440d2f06982181a8c92b7cdd765336b9a1"
	      "e1ab70283d6db0a963fb648c37c4e29a74c37577291049ab"
	      "47cdbc104c04db966681ea8ebb9f00cf4c4a546211737957"
	      "5fbda4b801979451fa94b19b4e93656705c0f734f3e0914b"
	      "b96c1e2b8a0fb68faf14296efdf3300ad95bcde8b67cc4b2"
	      "6e6488eef925cfaeac6f0d6567e8b41355f89d1c2b8fe687"
	      "bfa2df5e287e1305b89b8c388c26196090ac0351abc561aa"
	      "dc797da8ccea4146c3e96095ebce353e0da4c55019052caa", 16);
  mpz_set_str(signature.r,
	      "21ca148cdf44be4ae93b2f353b8e512d03ad96dafa80623f"
	      "de4922a95f032732", 16);
  mpz_set_str(signature.s,
	      "73e48b77a3aa44307483c2dd895cb51db2112177c185c59c"
	      "b1dcff32fda02a4f", 16);
  test_dsa_verify(params, pub.y, &nettle_sha1,
		  SHEX("ca84af5c9adbc0044db00d7acfb1b493aab0388ffbad47b3"
		       "8cd3e9e3111cfe2cda2a45f751c46862f05bdcec4b698adf"
		       "d2e1606e484c3be4ac0c379d4fbc7c2cda43e922811d7f6c"
		       "33040e8e65d5f317684b90e26387cf931fe7c2f515058d75"
		       "3b08137ff2c6b79c910de8283149e6872cb66f7e02e66f23"
		       "71785129569362f1"),
		  &signature);

  /* L=3072, N=256, SHA-256 */
  mpz_set_str(pub.p,
	      "c7b86d7044218e367453d210e76433e4e27a983db1c560bb"
	      "9755a8fb7d819912c56cfe002ab1ff3f72165b943c0b28ed"
	      "46039a07de507d7a29f738603decd1270380a41f971f2592"
	      "661a64ba2f351d9a69e51a888a05156b7fe1563c4b77ee93"
	      "a44949138438a2ab8bdcfc49b4e78d1cde766e5498476005"
	      "7d76cd740c94a4dd25a46aa77b18e9d707d6738497d4eac3"
	      "64f4792d9766a16a0e234807e96b8c64d404bbdb876e39b5"
	      "799ef53fe6cb9bab62ef19fdcc2bdd905beda13b9ef7ac35"
	      "f1f557cb0dc458c019e2bc19a9f5dfc1e4eca9e6d4665641"
	      "24304a31f038605a3e342da01be1c2b545610edd2c1397a3"
	      "c8396588c6329efeb4e165af5b368a39a88e4888e39f40bb"
	      "3de4eb1416672f999fead37aef1ca9643ff32cdbc0fcebe6"
	      "28d7e46d281a989d43dd21432151af68be3f6d56acfbdb6c"
	      "97d87fcb5e6291bf8b4ee1275ae0eb4383cc753903c8d29f"
	      "4adb6a547e405decdff288c5f6c7aa30dcb12f84d392493a"
	      "70933317c0f5e6552601fae18f17e6e5bb6bf396d32d8ab9", 16);
  mpz_set_str(pub.q,
	      "876fa09e1dc62b236ce1c3155ba48b0ccfda29f3ac5a97f7"
	      "ffa1bd87b68d2a4b", 16);
  mpz_set_str(pub.g,
	      "110afebb12c7f862b6de03d47fdbc3326e0d4d31b12a8ca9"
	      "5b2dee2123bcc667d4f72c1e7209767d2721f95fbd9a4d03"
	      "236d54174fbfaff2c4ff7deae4738b20d9f37bf0a1134c28"
	      "8b420af0b5792e47a92513c0413f346a4edbab2c45bdca13"
	      "f5341c2b55b8ba54932b9217b5a859e553f14bb8c120fbb9"
	      "d99909dff5ea68e14b379964fd3f3861e5ba5cc970c4a180"
	      "eef54428703961021e7bd68cb637927b8cbee6805fa27285"
	      "bfee4d1ef70e02c1a18a7cd78bef1dd9cdad45dde9cd6907"
	      "55050fc4662937ee1d6f4db12807ccc95bc435f11b71e708"
	      "6048b1dab5913c6055012de82e43a4e50cf93feff5dcab81"
	      "4abc224c5e0025bd868c3fc592041bba04747c10af513fc3"
	      "6e4d91c63ee5253422cf4063398d77c52fcb011427cbfcfa"
	      "67b1b2c2d1aa4a3da72645cb1c767036054e2f31f88665a5"
	      "4461c885fb3219d5ad8748a01158f6c7c0df5a8c908ba8c3"
	      "e536822428886c7b500bbc15b49df746b9de5a78fe3b4f69"
	      "91d0110c3cbff458039dc36261cf46af4bc2515368f4abb7", 16);
  mpz_set_str(pub.y,
	      "456a105c713566234838bc070b8a751a0b57767cb75e9911"
	      "4a1a46641e11da1fa9f22914d808ad7148612c1ea55d2530"
	      "1781e9ae0c9ae36a69d87ba039ec7cd864c3ad094873e6e5"
	      "6709fd10d966853d611b1cff15d37fdee424506c184d62c7"
	      "033358be78c2250943b6f6d043d63b317de56e5ad8d1fd97"
	      "dd355abe96452f8e435485fb3b907b51900aa3f24418df50"
	      "b4fcdafbf6137548c39373b8bc4ba3dabb4746ebd17b87fc"
	      "d6a2f197c107b18ec5b465e6e4cb430d9c0ce78da5988441"
	      "054a370792b730da9aba41a3169af26176f74e6f7c0c9c9b"
	      "55b62bbe7ce38d4695d48157e660c2acb63f482f55418150"
	      "e5fee43ace84c540c3ba7662ae80835c1a2d51890ea96ba2"
	      "06427c41ef8c38aa07d2a365e7e58380d8f4782e22ac2101"
	      "af732ee22758337b253637838e16f50f56d313d07981880d"
	      "685557f7d79a6db823c61f1bb3dbc5d50421a4843a6f2969"
	      "0e78aa0f0cff304231818b81fc4a243fc00f09a54c466d6a"
	      "8c73d32a55e1abd5ec8b4e1afa32a79b01df85a81f3f5cfe", 16);
  mpz_set_str(signature.r,
	      "53bae6c6f336e2eb311c1e92d95fc449a929444ef81ec427"
	      "9660b200d59433de", 16);
  mpz_set_str(signature.s,
	      "49f3a74e953e77a7941af3aefeef4ed499be209976a0edb3"
	      "fa5e7cb961b0c112", 16);
  test_dsa_verify(params, pub.y, &nettle_sha256,
		  SHEX("cb06e02234263c22b80e832d6dc5a1bee5ea8af3bc2da752"
		       "441c04027f176158bfe68372bd67f84d489c0d49b07d4025"
		       "962976be60437be1a2d01d3be0992afa5abe0980e26a9da4"
		       "ae72f827b423665195cc4eed6fe85c335b32d9c03c945a86"
		       "e7fa99373f0a30c6eca938b3afb6dff67adb8bece6f8cfec"
		       "4b6a12ea281e2323"),
		  &signature);

  /* L=3072, N=256, SHA-384 */
  mpz_set_str(pub.p,
	      "a410d23ed9ad9964d3e401cb9317a25213f75712acbc5c12"
	      "191abf3f1c0e723e2333b49eb1f95b0f9748d952f04a5ae3"
	      "58859d384403ce364aa3f58dd9769909b45048548c55872a"
	      "6afbb3b15c54882f96c20df1b2df164f0bac849ca17ad2df"
	      "63abd75c881922e79a5009f00b7d631622e90e7fa4e98061"
	      "8575e1d6bd1a72d5b6a50f4f6a68b793937c4af95fc11541"
	      "759a1736577d9448b87792dff07232415512e933755e1225"
	      "0d466e9cc8df150727d747e51fea7964158326b1365d580c"
	      "b190f4518291598221fdf36c6305c8b8a8ed05663dd7b006"
	      "e945f592abbecae460f77c71b6ec649d3fd5394202ed7bbb"
	      "d040f7b8fd57cb06a99be254fa25d71a3760734046c2a0db"
	      "383e02397913ae67ce65870d9f6c6f67a9d00497be1d763b"
	      "21937cf9cbf9a24ef97bbcaa07916f8894e5b7fb03258821"
	      "ac46140965b23c5409ca49026efb2bf95bce025c4183a5f6"
	      "59bf6aaeef56d7933bb29697d7d541348c871fa01f869678"
	      "b2e34506f6dc0a4c132b689a0ed27dc3c8d53702aa584877", 16);
  mpz_set_str(pub.q,
	      "abc67417725cf28fc7640d5de43825f416ebfa80e191c42e"
	      "e886303338f56045", 16);
  mpz_set_str(pub.g,
	      "867d5fb72f5936d1a14ed3b60499662f3124686ef108c5b3"
	      "da6663a0e86197ec2cc4c9460193a74ff16028ac9441b0c7"
	      "d27c2272d483ac7cd794d598416c4ff9099a61679d417d47"
	      "8ce5dd974bf349a14575afe74a88b12dd5f6d1cbd3f91ddd"
	      "597ed68e79eba402613130c224b94ac28714a1f1c552475a"
	      "5d29cfcdd8e08a6b1d65661e28ef313514d1408f5abd3e06"
	      "ebe3a7d814d1ede316bf495273ca1d574f42b482eea30db5"
	      "3466f454b51a175a0b89b3c05dda006e719a2e6371669080"
	      "d768cc038cdfb8098e9aad9b8d83d4b759f43ac9d22b353e"
	      "d88a33723550150de0361b7a376f37b45d437f71cb711f28"
	      "47de671ad1059516a1d45755224a15d37b4aeada3f58c69a"
	      "136daef0636fe38e3752064afe598433e80089fda24b144a"
	      "462734bef8f77638845b00e59ce7fa4f1daf487a2cada11e"
	      "aba72bb23e1df6b66a183edd226c440272dd9b06bec0e57f"
	      "1a0822d2e00212064b6dba64562085f5a75929afa5fe509e"
	      "0b78e630aaf12f91e4980c9b0d6f7e059a2ea3e23479d930", 16);
  mpz_set_str(pub.y,
	      "1f0a5c75e7985d6e70e4fbfda51a10b925f6accb600d7c65"
	      "10db90ec367b93bb069bd286e8f979b22ef0702f717a8755"
	      "c18309c87dae3fe82cc3dc8f4b7aa3d5f3876f4d4b3eb68b"
	      "fe910c43076d6cd0d39fc88dde78f09480db55234e6c8ca5"
	      "9fe2700efec04feee6b4e8ee2413721858be7190dbe905f4"
	      "56edcab55b2dc2916dc1e8731988d9ef8b619abcf8955aa9"
	      "60ef02b3f02a8dc649369222af50f1338ed28d667f3f10ca"
	      "e2a3c28a3c1d08df639c81ada13c8fd198c6dae3d62a3fe9"
	      "f04c985c65f610c06cb8faea68edb80de6cf07a8e89c0021"
	      "8185a952b23572e34df07ce5b4261e5de427eb503ee1baf5"
	      "992db6d438b47434c40c22657bc163e7953fa33eff39dc27"
	      "34607039aadd6ac27e4367131041f845ffa1a13f556bfba2"
	      "307a5c78f2ccf11298c762e08871968e48dc3d1569d09965"
	      "cd09da43cf0309a16af1e20fee7da3dc21b364c4615cd512"
	      "3fa5f9b23cfc4ffd9cfdcea670623840b062d4648d2eba78"
	      "6ad3f7ae337a4284324ace236f9f7174fbf442b99043002f", 16);
  mpz_set_str(signature.r,
	      "7695698a14755db4206e850b4f5f19c540b07d07e08aac59"
	      "1e20081646e6eedc", 16);
  mpz_set_str(signature.s,
	      "3dae01154ecff7b19007a953f185f0663ef7f2537f0b15e0"
	      "4fb343c961f36de2", 16);
  test_dsa_verify(params, pub.y, &nettle_sha384,
		  SHEX("ed9a64d3109ef8a9292956b946873ca4bd887ce624b81be8"
		       "1b82c69c67aaddf5655f70fe4768114db2834c71787f858e"
		       "5165da1a7fa961d855ad7e5bc4b7be31b97dbe770798ef79"
		       "66152b14b86ae35625a28aee5663b9ef3067cbdfbabd8719"
		       "7e5c842d3092eb88dca57c6c8ad4c00a19ddf2e1967b59bd"
		       "06ccaef933bc28e7"),
		  &signature);

  /* L=3072, N=256, SHA-512 */
  mpz_set_str(pub.p,
	      "c1d0a6d0b5ed615dee76ac5a60dd35ecb000a202063018b1"
	      "ba0a06fe7a00f765db1c59a680cecfe3ad41475badb5ad50"
	      "b6147e2596b88d34656052aca79486ea6f6ec90b23e363f3"
	      "ab8cdc8b93b62a070e02688ea877843a4685c2ba6db111e9"
	      "addbd7ca4bce65bb10c9ceb69bf806e2ebd7e54edeb7f996"
	      "a65c907b50efdf8e575bae462a219c302fef2ae81d73cee7"
	      "5274625b5fc29c6d60c057ed9e7b0d46ad2f57fe01f82323"
	      "0f31422722319ce0abf1f141f326c00fbc2be4cdb8944b6f"
	      "d050bd300bdb1c5f4da72537e553e01d51239c4d461860f1"
	      "fb4fd8fa79f5d5263ff62fed7008e2e0a2d36bf7b9062d0d"
	      "75db226c3464b67ba24101b085f2c670c0f87ae530d98ee6"
	      "0c5472f4aa15fb25041e19106354da06bc2b1d322d40ed97"
	      "b21fd1cdad3025c69da6ce9c7ddf3dcf1ea4d56577bfdec2"
	      "3071c1f05ee4077b5391e9a404eaffe12d1ea62d06acd6bf"
	      "19e91a158d2066b4cd20e4c4e52ffb1d5204cd022bc7108f"
	      "2c799fb468866ef1cb09bce09dfd49e4740ff8140497be61", 16);
  mpz_set_str(pub.q,
	      "bf65441c987b7737385eadec158dd01614da6f15386248e59f3cddbefc8e9dd1", 16);
  mpz_set_str(pub.g,
	      "c02ac85375fab80ba2a784b94e4d145b3be0f92090eba17b"
	      "d12358cf3e03f4379584f8742252f76b1ede3fc37281420e"
	      "74a963e4c088796ff2bab8db6e9a4530fc67d51f88b905ab"
	      "43995aab46364cb40c1256f0466f3dbce36203ef228b35e9"
	      "0247e95e5115e831b126b628ee984f349911d30ffb9d613b"
	      "50a84dfa1f042ba536b82d5101e711c629f9f2096dc834de"
	      "ec63b70f2a2315a6d27323b995aa20d3d0737075186f5049"
	      "af6f512a0c38a9da06817f4b619b94520edfac85c4a6e2e1"
	      "86225c95a04ec3c3422b8deb284e98d24b31465802008a09"
	      "7c25969e826c2baa59d2cba33d6c1d9f3962330c1fcda7cf"
	      "b18508fea7d0555e3a169daed353f3ee6f4bb30244319161"
	      "dff6438a37ca793b24bbb1b1bc2194fc6e6ef60278157899"
	      "cb03c5dd6fc91a836eb20a25c09945643d95f7bd50d20668"
	      "4d6ffc14d16d82d5f781225bff908392a5793b803f9b70b4"
	      "dfcb394f9ed81c18e391a09eb3f93a032d81ba670cabfd6f"
	      "64aa5e3374cb7c2029f45200e4f0bfd820c8bd58dc5eeb34", 16);
  mpz_set_str(pub.y,
	      "6da54f2b0ddb4dcce2da1edfa16ba84953d8429ce60cd111"
	      "a5c65edcf7ba5b8d9387ab6881c24880b2afbdb437e9ed7f"
	      "fb8e96beca7ea80d1d90f24d546112629df5c9e9661742cc"
	      "872fdb3d409bc77b75b17c7e6cfff86261071c4b5c9f9898"
	      "be1e9e27349b933c34fb345685f8fc6c12470d124cecf51b"
	      "5d5adbf5e7a2490f8d67aac53a82ed6a2110686cf631c348"
	      "bcbc4cf156f3a6980163e2feca72a45f6b3d68c10e5a2283"
	      "b470b7292674490383f75fa26ccf93c0e1c8d0628ca35f2f"
	      "3d9b6876505d118988957237a2fc8051cb47b410e8b7a619"
	      "e73b1350a9f6a260c5f16841e7c4db53d8eaa0b4708d62f9"
	      "5b2a72e2f04ca14647bca6b5e3ee707fcdf758b925eb8d4e"
	      "6ace4fc7443c9bc5819ff9e555be098aa055066828e21b81"
	      "8fedc3aac517a0ee8f9060bd86e0d4cce212ab6a3a243c5e"
	      "c0274563353ca7103af085e8f41be524fbb75cda88903907"
	      "df94bfd69373e288949bd0626d85c1398b3073a139d5c747"
	      "d24afdae7a3e745437335d0ee993eef36a3041c912f7eb58", 16);
  mpz_set_str(signature.r,
	      "a40a6c905654c55fc58e99c7d1a3feea2c5be64823d4086c"
	      "e811f334cfdc448d", 16);
  mpz_set_str(signature.s,
	      "6478050977ec585980454e0a2f26a03037b921ca588a78a4"
	      "daff7e84d49a8a6c", 16);
  test_dsa_verify(params, pub.y, &nettle_sha512,
		  SHEX("494180eed0951371bbaf0a850ef13679df49c1f13fe3770b"
		       "6c13285bf3ad93dc4ab018aab9139d74200808e9c55bf883"
		       "00324cc697efeaa641d37f3acf72d8c97bff0182a35b9401"
		       "50c98a03ef41a3e1487440c923a988e53ca3ce883a2fb532"
		       "bb7441c122f1dc2f9d0b0bc07f26ba29a35cdf0da846a9d8"
		       "eab405cbf8c8e77f"),
		  &signature);
  
  dsa_public_key_clear(&pub);
  dsa_private_key_clear(&key);
  dsa_signature_clear(&signature);
}
