/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file lib/exchange_api_reveal_melt.c
 * @brief Implementation of the /reveal-melt request
 * @author Özgür Kesim
 */
#include "platform.h"
#include <jansson.h>
#include <microhttpd.h> /* just for HTTP status codes */
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_json_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include "taler_json_lib.h"
#include "taler_exchange_service.h"
#include "exchange_api_common.h"
#include "exchange_api_handle.h"
#include "taler_signatures.h"
#include "exchange_api_curl_defaults.h"
#include "exchange_api_refresh_common.h"


/**
 * Handler for a running reveal-melt request
 */
struct TALER_EXCHANGE_RevealMeltHandle
{
  /**
   * The url for the request
   */
  char *request_url;

  /**
   * CURL handle for the request job.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Post Context
   */
  struct TALER_CURL_PostContext post_ctx;

  /**
   * Number of coins to expect
   */
  size_t num_expected_coins;

  /**
   * The input provided
   */
  const struct TALER_EXCHANGE_RevealMeltInput *reveal_input;

  /**
   * The melt data
   */
  struct MeltData_v27 md;

  /**
   * Callback to pass the result onto
   */
  TALER_EXCHANGE_RevealMeltCallback callback;

  /**
   * Closure for @e callback
   */
  void *callback_cls;

};

/**
 * We got a 200 OK response for the /reveal-melt operation.
 * Extract the signed blinded coins and return it to the caller.
 *
 * @param mrh operation handle
 * @param j_response reply from the exchange
 * @return #GNUNET_OK on success, #GNUNET_SYSERR on errors
 */
static enum GNUNET_GenericReturnValue
reveal_melt_ok (
  struct TALER_EXCHANGE_RevealMeltHandle *mrh,
  const json_t *j_response)
{
  struct TALER_EXCHANGE_RevealMeltResponse response = {
    .hr.reply = j_response,
    .hr.http_status = MHD_HTTP_OK,
  };
  struct TALER_BlindedDenominationSignature blind_sigs[mrh->num_expected_coins];
  struct GNUNET_JSON_Specification spec[] = {
    TALER_JSON_spec_array_of_blinded_denom_sigs ("ev_sigs",
                                                 mrh->num_expected_coins,
                                                 blind_sigs),
    GNUNET_JSON_spec_end ()
  };
  if (GNUNET_OK !=
      GNUNET_JSON_parse (j_response,
                         spec,
                         NULL, NULL))
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }

  {
    struct TALER_EXCHANGE_RevealedCoinInfo coins[mrh->num_expected_coins];

    /* Reconstruct the coins and unblind the signatures */
    for (unsigned int i = 0; i<mrh->num_expected_coins; i++)
    {
      struct TALER_EXCHANGE_RevealedCoinInfo *rci = &coins[i];
      const struct FreshCoinData *fcd = &mrh->md.fcds[i];
      const struct TALER_DenominationPublicKey *pk;
      struct TALER_CoinSpendPublicKeyP coin_pub;
      struct TALER_CoinPubHashP coin_hash;
      struct TALER_FreshCoin coin;
      union GNUNET_CRYPTO_BlindingSecretP bks;
      const struct TALER_AgeCommitmentHash *pah = NULL;

      rci->ps = fcd->ps[mrh->reveal_input->noreveal_index];
      rci->bks = fcd->bks[mrh->reveal_input->noreveal_index];
      rci->age_commitment_proof = NULL;
      pk = &fcd->fresh_pk;
      if (NULL != mrh->md.melted_coin.age_commitment_proof)
      {
        rci->age_commitment_proof
          = fcd->age_commitment_proofs[mrh->reveal_input->noreveal_index];
        TALER_age_commitment_hash (
          &rci->age_commitment_proof->commitment,
          &rci->h_age_commitment);
        pah = &rci->h_age_commitment;
      }

      TALER_planchet_setup_coin_priv (&rci->ps,
                                      &mrh->reveal_input->blinding_values[i],
                                      &rci->coin_priv);
      TALER_planchet_blinding_secret_create (&rci->ps,
                                             &mrh->reveal_input->blinding_values
                                             [i],
                                             &bks);
      /* needed to verify the signature, and we didn't store it earlier,
         hence recomputing it here... */
      GNUNET_CRYPTO_eddsa_key_get_public (&rci->coin_priv.eddsa_priv,
                                          &coin_pub.eddsa_pub);
      TALER_coin_pub_hash (&coin_pub,
                           pah,
                           &coin_hash);
      if (GNUNET_OK !=
          TALER_planchet_to_coin (pk,
                                  &blind_sigs[i],
                                  &bks,
                                  &rci->coin_priv,
                                  pah,
                                  &coin_hash,
                                  &mrh->reveal_input->blinding_values[i],
                                  &coin))
      {
        GNUNET_break_op (0);
        GNUNET_JSON_parse_free (spec);
        return GNUNET_SYSERR;
      }
      GNUNET_JSON_parse_free (spec);
      rci->sig = coin.sig;
    }

    response.details.ok.num_coins = mrh->num_expected_coins;
    response.details.ok.coins = coins;
    mrh->callback (mrh->callback_cls,
                   &response);
    /* Make sure the callback isn't called again */
    mrh->callback = NULL;
    /* Free resources */
    for (size_t i = 0; i < mrh->num_expected_coins; i++)
    {
      struct TALER_EXCHANGE_RevealedCoinInfo *rci = &coins[i];

      TALER_denom_sig_free (&rci->sig);
      TALER_blinded_denom_sig_free (&blind_sigs[i]);
    }
  }

  return GNUNET_OK;
}


/**
 * Function called when we're done processing the
 * HTTP /reveal-melt request.
 *
 * @param cls the `struct TALER_EXCHANGE_RevealMeltHandle`
 * @param response_code The HTTP response code
 * @param response response data
 */
static void
handle_reveal_melt_finished (
  void *cls,
  long response_code,
  const void *response)
{
  struct TALER_EXCHANGE_RevealMeltHandle *mrh = cls;
  const json_t *j_response = response;
  struct TALER_EXCHANGE_RevealMeltResponse awr = {
    .hr.reply = j_response,
    .hr.http_status = (unsigned int) response_code
  };

  mrh->job = NULL;
  switch (response_code)
  {
  case 0:
    awr.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
    break;
  case MHD_HTTP_OK:
    {
      enum GNUNET_GenericReturnValue ret;

      ret = reveal_melt_ok (mrh,
                            j_response);
      if (GNUNET_OK != ret)
      {
        GNUNET_break_op (0);
        awr.hr.http_status = 0;
        awr.hr.ec = TALER_EC_GENERIC_REPLY_MALFORMED;
        break;
      }
      GNUNET_assert (NULL == mrh->callback);
      TALER_EXCHANGE_reveal_melt_cancel (mrh);
      return;
    }
  case MHD_HTTP_BAD_REQUEST:
    /* This should never happen, either us or the exchange is buggy
       (or API version conflict); just pass JSON reply to the application */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  case MHD_HTTP_NOT_FOUND:
    /* Nothing really to verify, the exchange basically just says
       that it doesn't know this age-melt commitment. */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  case MHD_HTTP_CONFLICT:
    /* An age commitment for one of the coins did not fulfill
     * the required maximum age requirement of the corresponding
     * reserve.
     * Error code: TALER_EC_EXCHANGE_GENERIC_COIN_AGE_REQUIREMENT_FAILURE
     * or TALER_EC_EXCHANGE_AGE_WITHDRAW_REVEAL_INVALID_HASH.
     */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    /* Server had an internal issue; we should retry, but this API
       leaves this to the application */
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;
  default:
    /* unexpected response code */
    GNUNET_break_op (0);
    awr.hr.ec = TALER_JSON_get_error_code (j_response);
    awr.hr.hint = TALER_JSON_get_error_hint (j_response);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d for exchange melt\n",
                (unsigned int) response_code,
                (int) awr.hr.ec);
    break;
  }
  mrh->callback (mrh->callback_cls,
                 &awr);
  TALER_EXCHANGE_reveal_melt_cancel (mrh);
}


/**
 * Call /reveal-melt
 *
 * @param curl_ctx The context for CURL
 * @param mrh The handler
 */
static void
perform_protocol (
  struct GNUNET_CURL_Context *curl_ctx,
  struct TALER_EXCHANGE_RevealMeltHandle *mrh)
{
  CURL *curlh;
  json_t *j_array_of_signatures;


  j_array_of_signatures = json_array ();
  GNUNET_assert (NULL != j_array_of_signatures);

  for (uint8_t k = 0; k < TALER_CNC_KAPPA; k++)
  {
    if (mrh->reveal_input->noreveal_index == k)
      continue;

    GNUNET_assert (0 == json_array_append_new (
                     j_array_of_signatures,
                     GNUNET_JSON_from_data_auto (&mrh->md.signatures[k])));
  }
  {
    json_t *j_request_body;

    j_request_body = GNUNET_JSON_PACK (
      GNUNET_JSON_pack_data_auto ("rc",
                                  &mrh->md.rc),
      GNUNET_JSON_pack_array_steal ("signatures",
                                    j_array_of_signatures));
    GNUNET_assert (NULL != j_request_body);

    if (NULL != mrh->reveal_input->melt_input->melt_age_commitment_proof)
    {
      json_t *j_age = GNUNET_JSON_PACK (
        TALER_JSON_pack_age_commitment (
          "age_commitment",
          &mrh->reveal_input->melt_input->melt_age_commitment_proof->commitment)
        );
      GNUNET_assert (NULL != j_age);
      GNUNET_assert (0 ==
                     json_object_update_new (j_request_body,
                                             j_age));
    }
    curlh = TALER_EXCHANGE_curl_easy_get_ (mrh->request_url);
    GNUNET_assert (NULL != curlh);
    GNUNET_assert (GNUNET_OK ==
                   TALER_curl_easy_post (&mrh->post_ctx,
                                         curlh,
                                         j_request_body));
    json_decref (j_request_body);
  }

  mrh->job = GNUNET_CURL_job_add2 (
    curl_ctx,
    curlh,
    mrh->post_ctx.headers,
    &handle_reveal_melt_finished,
    mrh);
  if (NULL == mrh->job)
  {
    GNUNET_break (0);
    if (NULL != curlh)
      curl_easy_cleanup (curlh);
    TALER_EXCHANGE_reveal_melt_cancel (mrh);
  }
}


struct TALER_EXCHANGE_RevealMeltHandle *
TALER_EXCHANGE_reveal_melt (
  struct GNUNET_CURL_Context *curl_ctx,
  const char *exchange_url,
  const struct TALER_EXCHANGE_RevealMeltInput *reveal_melt_input,
  TALER_EXCHANGE_RevealMeltCallback reveal_cb,
  void *reveal_cb_cls)
{
  struct TALER_EXCHANGE_RevealMeltHandle *mrh =
    GNUNET_new (struct TALER_EXCHANGE_RevealMeltHandle);
  mrh->callback = reveal_cb;
  mrh->callback_cls = reveal_cb_cls;
  mrh->reveal_input = reveal_melt_input;
  mrh->num_expected_coins = reveal_melt_input->melt_input->num_fresh_denom_pubs;
  mrh->request_url = TALER_url_join (exchange_url,
                                     "reveal-melt",
                                     NULL);
  if (NULL == mrh->request_url)
  {
    GNUNET_break (0);
    GNUNET_free (mrh);
    return NULL;
  }
  if (reveal_melt_input->num_blinding_values !=
      reveal_melt_input->melt_input->num_fresh_denom_pubs)
  {
    GNUNET_break (0);
    GNUNET_free (mrh);
    return NULL;
  }
  TALER_EXCHANGE_get_melt_data_v27 (
    reveal_melt_input->rms,
    reveal_melt_input->melt_input,
    reveal_melt_input->blinding_seed,
    reveal_melt_input->blinding_values,
    &mrh->md);
  perform_protocol (curl_ctx,
                    mrh);
  return mrh;
}


void
TALER_EXCHANGE_reveal_melt_cancel (
  struct TALER_EXCHANGE_RevealMeltHandle *mrh)
{
  if (NULL != mrh->job)
  {
    GNUNET_CURL_job_cancel (mrh->job);
    mrh->job = NULL;
  }
  TALER_curl_easy_post_finished (&mrh->post_ctx);
  TALER_EXCHANGE_free_melt_data_v27 (&mrh->md);
  GNUNET_free (mrh->request_url);
  GNUNET_free (mrh);
}
