/*
    cobbler-enlist - Cobbler enlistment tool
    Copyright (C) 2011 Canonical Ltd.

    Authors: Dave Walker (Daviey) <DaveWalker@ubuntu.com>
             Carlos de-Avillez <carlos.de.avillez@ubuntu.com>
             Adam Gandelman <adamg@canonical.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, version 3 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Usage: ./cobbler-enlist --serverurl=http://cobbler.server/cobbler_api \
             --username=cobbler --password=cobbler --name=test \
             --profile=ubuntu-server-x86_64
            optionally, '--netif=<network I/F>' can also be passed, restricting
            registration to this I/F. If no --netif, then all I/Fs will be
            registered.

*/

#define NAME "Cobbler Enlistment Tool"
#define VERSION "1.0"

#define OPT_URL      0
#define OPT_USRNAME  1
#define OPT_PASWD    2
#define OPT_NAME     3
#define OPT_PROFILE  4
#define OPT_NETIF    5
#define OPT_HELP     6

#define NO_MACADDR "00:00:00:00:00:00"

static int debug =  0;

struct cobbler_client {
  xmlrpc_env env;
  xmlrpc_client *client;
  xmlrpc_value *resultP;
  const char *token;
  const char *system;
  const char *serverUrl;
  const char *username;
  const char *password;
  const char *name;
  const char *profile;
  const char *netif;
  const char *mgmt_class;
  const char *macaddr;
  struct netif *netifs;
};

struct netif {
  const char *interface;
  const char *macaddr;
  struct netif *next;
};

void get_netifs(struct cobbler_client *enlister);

int init_client(struct cobbler_client *enlister) {
  enlister->client = NULL;
  enlister->token = NULL;
  enlister->system = NULL;
  enlister->serverUrl = NULL;
  enlister->username = NULL;
  enlister->password = NULL;
  enlister->name = NULL;
  enlister->profile = NULL;
  enlister->netif = NULL;
  enlister->mgmt_class = NULL;
  enlister->macaddr = NULL;
  enlister->netifs = NULL;
  return 0;
}

void display_config(struct cobbler_client *enlister) {
  printf("[DEBUG] Enlisting system using the following config:\n");
  printf("\tserverUrl:  %s\n", enlister->serverUrl);
  printf("\tusername:   %s\n", enlister->username);
  printf("\tpassword:   %s\n", enlister->password);
  printf("\tname:       %s\n", enlister->name);
  printf("\tprofile:    %s\n", enlister->profile);
  if (enlister->netifs) {
    get_netifs(enlister);
  }
  if (enlister->mgmt_class)
    printf("\tmgmt class: %s\n", enlister->mgmt_class);
}

void get_netifs(struct cobbler_client *enlister) {
  struct netif *current = enlister->netifs;
  if (current == NULL) {
    printf("Interface list is empty\n");
    return;
  }
  int i = 0;
  while (current != NULL) {
    printf("\tinterface %d: %s - %s\n", i, current->interface, current->macaddr);
    i++;
    current = current->next;
  }
}

void die(char *msg) {
  fprintf(stderr, "[ERROR] %s\n", msg);
  exit(1);
}

/* append a new interface to the netifs list. macaddr remains empty
   until populated by get_macaddresses() */
int append_netif(struct cobbler_client *enlister, char *netif) {
  if (debug)
    printf("[DEBUG] Appending interface %s to enlister's netif list\n", netif);
  struct netif *new_netif, *current;
  new_netif = (struct netif *) malloc(sizeof(struct netif));
  if (new_netif == NULL)
    die("malloc() - append_netif()");
  memset(new_netif, 0, sizeof(struct netif));
  new_netif->interface = netif;
  new_netif->macaddr = NULL;
  new_netif->next = NULL;
  if (enlister->netifs == NULL) {
    enlister->netifs = new_netif;
    return 0;
  }
  int i;
  current = enlister->netifs;
  while (current->next != NULL) {
    current = current->next;
    i++;
  }
  current->next = new_netif;
  return 0;
}

/* delete an interface from the netifs list */
int remove_netif(struct cobbler_client *enlister, const char *interface) {
  if (debug)
    printf("[DEBUG] Removing from interface list: %s\n", interface);
  struct netif *current = enlister->netifs;
  int i = 0;
  if (strcasecmp(current->interface, interface) == 0) {
    enlister->netifs = current->next;
    return 0;
  }
  i++;
  while (current->next) {
    if (strcasecmp(current->next->interface, interface) == 0) {
      current->next = current->next->next;
      return 0;
    }
    current = current->next;
    i++;
  }
  printf("ERROR: NEtif: %s doesn't exist in enlister->netifs\n", interface);
  return 1;
}

// Overide ether_ntoa_r because stock decides the convention of leading zeros is silly.
char *
ether_ntoa_r (const struct ether_addr *addr, char *buf)
{
  snprintf (buf, 18, "%02x:%02x:%02x:%02x:%02x:%02x",
            addr->ether_addr_octet[0], addr->ether_addr_octet[1],
            addr->ether_addr_octet[2], addr->ether_addr_octet[3],
            addr->ether_addr_octet[4], addr->ether_addr_octet[5]);
  return buf;
}

int interface_exists(const char *interface) {
  if (debug)
    printf("[DEBUG] Checking for interface: %s\n", interface);
  struct if_nameindex *curif, *ifs = if_nameindex();
  for (curif = ifs; curif && curif->if_name; curif++) {
    if (strcasecmp(interface, curif->if_name) == 0)
     return 0;
  }
  return 1;
}

/* inspect each enlister->netifs.  Make sure the interface exists on the system
   Remove it from the list if it doesn't.
   If we've cleaned the entire list because none of them exist, return 1. 
   Can/should be extended to do more verification?
*/
int verify_netifs(struct cobbler_client *enlister) {
  struct netif *current = enlister->netifs;
  while (current) {
    if (interface_exists(current->interface) != 0) {
      printf("[WARNING] Interface %s doesn't exist, removing.\n", current->interface);
      remove_netif(enlister, current->interface);
    }
    current = current->next;
  }
  if (enlister->netifs == NULL)
    return 1;
  return 0;
}

/* utility function to test whether the netifs list contains at least
   one mac address 
*/
int contains_one_macaddr(struct cobbler_client *enlister) {
  struct netif *current = enlister->netifs;
  while (current) {
    if ((current->macaddr != NULL) &&
        (strcasecmp(current->macaddr, NO_MACADDR) != 0)) {
          return 0;
    }
  }
  return 1;
}

char *get_interface_macaddr(const char *interface) {
  if (debug)
    printf("[DEBUG] Getting mac addr for interface %s\n", interface);
  char mac_str[255];
  char *macaddr;
  int ioctl_data = 0;
  struct ifreq ifr;
  ioctl_data = socket(PF_INET, SOCK_STREAM, 0);
  memset(&ifr, 0, sizeof(ifr));
  strncpy(ifr.ifr_name, interface, sizeof(interface));
  ioctl(ioctl_data, SIOCGIFHWADDR, &ifr);
  ether_ntoa_r((const struct ether_addr *) &(ifr.ifr_hwaddr.sa_data), mac_str);
  macaddr = malloc(sizeof(mac_str));
  if (macaddr == NULL)
    die("malloc() - get_interface_macaddr()");
  strncpy(macaddr, mac_str, sizeof(mac_str));
  return macaddr;
}

/* iterate through enlister->netifs, populating macaddr for each.
   if we can't determine a valid mac addr, remove from list and continue
   returns non-zero if the resulting enlister->netifs does not contain at least
   one mac addr.
*/
int get_mac_addresses(struct cobbler_client *enlister) {
  if (debug)
    printf("[DEBUG] get_mac_address()\n");
  const char *addr;
  struct netif *current = enlister->netifs;
  while (current != NULL) {
    addr = get_interface_macaddr(current->interface);
    if (strcasecmp(addr, NO_MACADDR) == 0) {
      printf("[WARNING] Could not determine mac addr for %s, skipping.\n",
             current->interface);
      remove_netif(enlister, current->interface);
    } else {
      current->macaddr = addr;
    };
    current = current->next;
  }
  return contains_one_macaddr(enlister);
}

void help(char *progname)
{
  printf("Usage: %s --help  -- provides this help\n" , progname);
  printf("     : %s <parameters>\n", progname);
  printf("Parameters: -s, --serverurl   -- resolvable Cobbler server API URL\n");
  printf("            -u, --username    -- valid Cobbler user\n");
  printf("            -p, --password    -- password for above user\n");
  printf("            -n, --name        -- Cobbler name\n");
  printf("            -P, --profile     -- Cobbler profile\n");
  printf("            -i, --netif       -- single network I/F to be registered\n");
  printf("            -m, --manclass    -- Cobbler management class\n");
  printf("\n  --netif is optional; if not specified, all network I/Fs will be registered\n");
  printf("  all other parameters are required\n");
}
