// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef REMOTING_PROTOCOL_SDP_MESSAGE_H_
#define REMOTING_PROTOCOL_SDP_MESSAGE_H_

#include <string>
#include <vector>

#include "base/macros.h"

namespace remoting {
namespace protocol {

// SdpMessage is used to process session descriptions messages in SDP format
// generated by WebRTC (see RFC 4566). In particularly it allows configuring
// video and audio codecs.
//
// It also normalizes the SDP message to make sure the text used for HMAC
// signature verification is the same that was signed on the sending side.
// This is necessary because WebRTC generates SDP with CRLF line endings which
// are sometimes converted to LF after passing the signaling channel.
class SdpMessage {
 public:
  explicit SdpMessage(const std::string& sdp);
  ~SdpMessage();

  bool has_audio() const { return has_audio_; }
  bool has_video() const { return has_video_; }

  // Returns string representation of the SDP message. The result always has
  // line-endings normalized to LF and empty lines removed.
  std::string ToString() const;

  // Adds specified parameters for the |codec|. Returns false if the |codec| is
  // not listed anywhere in the SDP message.
  bool AddCodecParameter(const std::string& codec,
                         const std::string& parameters_to_add);

 private:
  // Finds the line of the form "a=rtpmap:<payload_type> <codec>/.." with the
  // specified |codec|. Sets |line_num| to line number and |payload_type| to the
  // payload type from that line. Returns false if the codec wasn't found.
  bool FindCodec(const std::string& codec,
                 int* line_num,
                 std::string* payload_type) const;

  std::vector<std::string> sdp_lines_;

  bool has_audio_ = false;
  bool has_video_ = false;

  DISALLOW_COPY_AND_ASSIGN(SdpMessage);
};

}  // namespace protocol
}  // namespace remoting

#endif  // REMOTING_PROTOCOL_SDP_MESSAGE_H_
