
// qt
#include <QKeyEvent>
#include <QFile>
#include <QTextStream>

// myth
#include "mythcorecontext.h"
#include "mythdialogbox.h"
#include "recordingrule.h"
#include "mythdb.h"
#include "mythuihelper.h"
#include "mythmainwindow.h"

#include "progdetails.h"
#include "mythdate.h"

#define LASTPAGE 2

ProgDetails::ProgDetails(MythScreenStack *parent, const ProgramInfo *progInfo) :
    MythScreenType (parent, "progdetails"),
    m_progInfo(*progInfo), m_infoList(*this)
{
}

bool ProgDetails::Create(void)
{
    bool foundtheme = false;

    // Load the theme for this screen
    foundtheme = LoadWindowFromXML("schedule-ui.xml", "programdetails", this);

    if (!foundtheme)
        return false;

    // Initialise details list
    if (!m_infoList.Create(true))
    {
        LOG(VB_GENERAL, LOG_ERR, "Cannot load 'Info buttonlist'");
        return false;
    }

    BuildFocusList();

    return true;
}

QString ProgDetails::getRatings(bool recorded, uint chanid, QDateTime startts)
{
    QString table = (recorded) ? "recordedrating" : "programrating";
    QString sel = QString(
        "SELECT system, rating FROM %1 "
        "WHERE chanid  = :CHANID "
        "AND starttime = :STARTTIME").arg(table);

    MSqlQuery query(MSqlQuery::InitCon());
    query.prepare(sel);
    query.bindValue(":CHANID", chanid);
    query.bindValue(":STARTTIME", startts);

    if (!query.exec() || !query.isActive())
    {
        MythDB::DBError("ProgDetails::getRatings", query);
        return "";
    }

    QMap<QString,QString> main_ratings;
    QString advisory;
    while (query.next())
    {
        if (query.value(0).toString().toLower() == "advisory")
        {
            advisory += query.value(1).toString() + ", ";
            continue;
        }
        main_ratings[query.value(0).toString()] = query.value(1).toString();
    }

    advisory = advisory.left(advisory.length() - 2);

    if (main_ratings.empty())
        return advisory;

    if (!advisory.isEmpty())
        advisory = ": " + advisory;

    if (main_ratings.size() == 1)
    {
        return *main_ratings.begin() + advisory;
    }

    QString ratings;
    QMap<QString,QString>::const_iterator it;
    for (it = main_ratings.begin(); it != main_ratings.end(); ++it)
    {
        ratings += it.key() + ": " + *it + ", ";
    }

    return ratings + "Advisory" + advisory;
}

ProgDetails::~ProgDetails(void)
{
    m_infoList.Hide();
}

void ProgDetails::Init(void)
{
    updatePage();
}

void ProgDetails::updatePage(void)
{
    if (m_data.isEmpty())
        loadPage();

    m_infoList.Display(m_data);
}

void ProgDetails::addItem(const QString &title, const QString &value,
                          ProgInfoList::VisibleLevel level)
{
    if (value.isEmpty())
        return;
    ProgInfoList::DataItem item = std::make_tuple(title, value, level);
    m_data.append(item);
}

bool ProgDetails::keyPressEvent(QKeyEvent *event)
{
    bool handled;
    QStringList actions;
    handled = GetMythMainWindow()->TranslateKeyPress("Global", event, actions);

    for (int i = 0; i < actions.size() && !handled; i++)
    {
        QString action = actions[i];
        handled = true;

        if (action == "INFO")
        {
            m_infoList.Toggle();
            updatePage();
        }
        else if (action == "DOWN")
        {
            m_infoList.PageDown();
        }
        else if (action == "UP")
        {
            m_infoList.PageUp();
        }
        else
            handled = false;
    }

    if (!handled && MythScreenType::keyPressEvent(event))
        handled = true;

    return handled;
}

void ProgDetails::loadPage(void)
{
    MSqlQuery query(MSqlQuery::InitCon());
    QString category_type, showtype, year, syndicatedEpisodeNum;
    QString rating, colorcode, title_pronounce;
    float stars = 0.0;
    int partnumber = 0, parttotal = 0;
    int audioprop = 0, videoprop = 0, subtype = 0, generic = 0;
    bool recorded = false;

    RecordingRule* record = NULL;
    if (m_progInfo.GetRecordingRuleID())
    {
        record = new RecordingRule();
        record->LoadByProgram(&m_progInfo);
    }

    if (m_progInfo.GetFilesize())
        recorded = true;

    if (m_progInfo.GetScheduledEndTime() != m_progInfo.GetScheduledStartTime())
    {
        QString ptable = "program";
        if (recorded)
            ptable = "recordedprogram";

        query.prepare(QString("SELECT category_type, airdate, stars,"
                      " partnumber, parttotal, audioprop+0, videoprop+0,"
                      " subtitletypes+0, syndicatedepisodenumber, generic,"
                      " showtype, colorcode, title_pronounce"
                      " FROM %1 WHERE chanid = :CHANID AND"
                      " starttime = :STARTTIME ;").arg(ptable));

        query.bindValue(":CHANID",    m_progInfo.GetChanID());
        query.bindValue(":STARTTIME", m_progInfo.GetScheduledStartTime());

        if (query.exec() && query.next())
        {
            category_type = query.value(0).toString();
            year = query.value(1).toString();
            stars = query.value(2).toFloat();
            partnumber = query.value(3).toInt();
            parttotal = query.value(4).toInt();
            audioprop = query.value(5).toInt();
            videoprop = query.value(6).toInt();
            subtype = query.value(7).toInt();
            syndicatedEpisodeNum = query.value(8).toString();
            generic = query.value(9).toInt();
            showtype = query.value(10).toString();
            colorcode = query.value(11).toString();
            title_pronounce = query.value(12).toString();
        }
        else if (!query.isActive())
            MythDB::DBError("ProgDetails", query);

        rating = getRatings(
            recorded, m_progInfo.GetChanID(),
            m_progInfo.GetScheduledStartTime());
    }

    if (category_type.isEmpty() && !m_progInfo.GetProgramID().isEmpty())
    {
        QString prefix = m_progInfo.GetProgramID().left(2);

        if (prefix == "MV")
           category_type = "movie";
        else if (prefix == "EP")
           category_type = "series";
        else if (prefix == "SP")
           category_type = "sports";
        else if (prefix == "SH")
           category_type = "tvshow";
    }

    addItem(tr("Title"),
            m_progInfo.toString(ProgramInfo::kTitleSubtitle, " - "),
            ProgInfoList::kLevel1);

    addItem(tr("Title Pronounce"), title_pronounce, ProgInfoList::kLevel2);

    QString s = m_progInfo.GetDescription();

    QString attr;

    if (partnumber > 0)
        attr += tr("Part %1 of %2, ").arg(partnumber).arg(parttotal);

    if (!rating.isEmpty() && rating != "NR")
        attr += rating + ", ";
    if (category_type == "movie")
    {
        if (!year.isEmpty())
            attr += year + ", ";

    /* see #7810, was hardcoded to 4 star system, when every theme
     * uses 10 stars / 5 stars with half stars
     */
        if (stars > 0.0)
            attr += tr("%n star(s)", "", roundf(stars * 10.0)) + ", ";
    }
    if (!colorcode.isEmpty())
        attr += colorcode + ", ";

    if (audioprop & AUD_MONO)
        attr += tr("Mono") + ", ";
    if (audioprop & AUD_STEREO)
        attr += tr("Stereo") + ", ";
    if (audioprop & AUD_SURROUND)
        attr += tr("Surround Sound") + ", ";
    if (audioprop & AUD_DOLBY)
        attr += tr("Dolby Sound") + ", ";
    if (audioprop & AUD_HARDHEAR)
        attr += tr("Audio for Hearing Impaired") + ", ";
    if (audioprop & AUD_VISUALIMPAIR)
        attr += tr("Audio for Visually Impaired") + ", ";

    if (videoprop & VID_HDTV)
        attr += tr("HDTV") + ", ";
    if  (videoprop & VID_WIDESCREEN)
        attr += tr("Widescreen") + ", ";
    if  (videoprop & VID_AVC)
        attr += tr("AVC/H.264") + ", ";
    if  (videoprop & VID_720)
        attr += tr("720p Resolution") + ", ";
    if  (videoprop & VID_1080)
        attr += tr("1080i/p Resolution") + ", ";

    if (subtype & SUB_HARDHEAR)
        attr += tr("CC","Closed Captioned") + ", ";
    if (subtype & SUB_NORMAL)
        attr += tr("Subtitles Available") + ", ";
    if (subtype & SUB_ONSCREEN)
        attr += tr("Subtitled") + ", ";
    if (subtype & SUB_SIGNED)
        attr += tr("Deaf Signing") + ", ";

    if (generic && category_type == "series")
        attr += tr("Unidentified Episode") + ", ";
    else if (m_progInfo.IsRepeat())
        attr += tr("Repeat") + ", ";

    if (!attr.isEmpty())
    {
        attr.truncate(attr.lastIndexOf(','));
        s += " (" + attr + ")";
    }

    addItem(tr("Description"), s, ProgInfoList::kLevel1);

    QString actors, directors, producers, execProducers;
    QString writers, guestStars, hosts, adapters;
    QString presenters, commentators, guests;

    if (m_progInfo.GetScheduledEndTime() != m_progInfo.GetScheduledStartTime())
    {
        if (recorded)
            query.prepare("SELECT role,people.name FROM recordedcredits"
                          " AS credits"
                          " LEFT JOIN people ON credits.person = people.person"
                          " WHERE credits.chanid = :CHANID"
                          " AND credits.starttime = :STARTTIME"
                          " ORDER BY role;");
        else
            query.prepare("SELECT role,people.name FROM credits"
                          " LEFT JOIN people ON credits.person = people.person"
                          " WHERE credits.chanid = :CHANID"
                          " AND credits.starttime = :STARTTIME"
                          " ORDER BY role;");
        query.bindValue(":CHANID",    m_progInfo.GetChanID());
        query.bindValue(":STARTTIME", m_progInfo.GetScheduledStartTime());

        if (query.exec() && query.size() > 0)
        {
            QStringList plist;
            QString rstr, role, pname;

            while(query.next())
            {
                role = query.value(0).toString();
                /* The people.name column uses utf8_bin collation.
                 * Qt-MySQL drivers use QVariant::ByteArray for string-type
                 * MySQL fields marked with the BINARY attribute (those using a
                 * *_bin collation) and QVariant::String for all others.
                 * Since QVariant::toString() uses QString::fromAscii()
                 * (through QVariant::convert()) when the QVariant's type is
                 * QVariant::ByteArray, we have to use QString::fromUtf8()
                 * explicitly to prevent corrupting characters.
                 * The following code should be changed to use the simpler
                 * toString() approach, as above, if we do a DB update to
                 * coalesce the people.name values that differ only in case and
                 * change the collation to utf8_general_ci, to match the
                 * majority of other columns, or we'll have the same problem in
                 * reverse.
                 */
                pname = QString::fromUtf8(query.value(1)
                                          .toByteArray().constData());

                if (rstr != role)
                {
                    if (rstr == "actor")
                        actors = plist.join(", ");
                    else if (rstr == "director")
                        directors = plist.join(", ");
                    else if (rstr == "producer")
                        producers = plist.join(", ");
                    else if (rstr == "executive_producer")
                        execProducers = plist.join(", ");
                    else if (rstr == "writer")
                        writers = plist.join(", ");
                    else if (rstr == "guest_star")
                        guestStars = plist.join(", ");
                    else if (rstr == "host")
                        hosts = plist.join(", ");
                    else if (rstr == "adapter")
                        adapters = plist.join(", ");
                    else if (rstr == "presenter")
                        presenters = plist.join(", ");
                    else if (rstr == "commentator")
                        commentators = plist.join(", ");
                    else if (rstr == "guest")
                        guests =  plist.join(", ");

                    rstr = role;
                    plist.clear();
                }

                plist.append(pname);
            }
            if (rstr == "actor")
                actors = plist.join(", ");
            else if (rstr == "director")
                directors = plist.join(", ");
            else if (rstr == "producer")
                producers = plist.join(", ");
            else if (rstr == "executive_producer")
                execProducers = plist.join(", ");
            else if (rstr == "writer")
                writers = plist.join(", ");
            else if (rstr == "guest_star")
                guestStars = plist.join(", ");
            else if (rstr == "host")
                hosts = plist.join(", ");
            else if (rstr == "adapter")
                adapters = plist.join(", ");
            else if (rstr == "presenter")
                presenters = plist.join(", ");
            else if (rstr == "commentator")
                commentators = plist.join(", ");
            else if (rstr == "guest")
                guests =  plist.join(", ");
        }
    }
    addItem(tr("Actors"), actors, ProgInfoList::kLevel1);
    addItem(tr("Guest Star"), guestStars, ProgInfoList::kLevel1);
    addItem(tr("Guest"), guests, ProgInfoList::kLevel1);
    addItem(tr("Host"), hosts, ProgInfoList::kLevel1);
    addItem(tr("Presenter"), presenters, ProgInfoList::kLevel1);
    addItem(tr("Commentator"), commentators, ProgInfoList::kLevel1);
    addItem(tr("Director"), directors, ProgInfoList::kLevel1);
    addItem(tr("Producer"), producers, ProgInfoList::kLevel2);
    addItem(tr("Executive Producer"), execProducers, ProgInfoList::kLevel2);
    addItem(tr("Writer"), writers, ProgInfoList::kLevel2);
    addItem(tr("Adapter"), adapters, ProgInfoList::kLevel2);

    addItem(tr("Category"), m_progInfo.GetCategory(), ProgInfoList::kLevel1);

    query.prepare("SELECT genre FROM programgenres "
                  "WHERE chanid = :CHANID AND starttime = :STARTTIME "
                  "AND relevance > 0 ORDER BY relevance;");
    query.bindValue(":CHANID",    m_progInfo.GetChanID());
    query.bindValue(":STARTTIME", m_progInfo.GetScheduledStartTime());

    if (query.exec())
    {
        s.clear();
        while (query.next())
        {
            if (!s.isEmpty())
                s += ", ";
            s += query.value(0).toString();
        }
        addItem(tr("Genre"), s, ProgInfoList::kLevel1);
    }

    s.clear();
    if (!category_type.isEmpty())
    {
        s = category_type;
        if (!m_progInfo.GetSeriesID().isEmpty())
            s += "  (" + m_progInfo.GetSeriesID() + ")";
        if (!showtype.isEmpty())
            s += "  " + showtype;
    }
    addItem(tr("Type", "category_type"), s, ProgInfoList::kLevel1);

    s.clear();
    if (m_progInfo.GetSeason() > 0)
        s = QString::number(m_progInfo.GetSeason());
    addItem(tr("Season"), s, ProgInfoList::kLevel1);

    s.clear();
    if (m_progInfo.GetEpisode() > 0)
    {
        if (m_progInfo.GetEpisodeTotal() > 0)
            s = tr("%1 of %2").arg(m_progInfo.GetEpisode())
                              .arg(m_progInfo.GetEpisodeTotal());
        else
            s = QString::number(m_progInfo.GetEpisode());

    }
    addItem(tr("Episode"), s, ProgInfoList::kLevel1);

    addItem(tr("Syndicated Episode Number"), syndicatedEpisodeNum,
            ProgInfoList::kLevel1);

    s.clear();
    if (m_progInfo.GetOriginalAirDate().isValid() &&
        category_type != "movie")
    {
        s = MythDate::toString(m_progInfo.GetOriginalAirDate(),
                               MythDate::kDateFull | MythDate::kAddYear);
    }
    addItem(tr("Original Airdate"), s, ProgInfoList::kLevel1);

    addItem(tr("Program ID"), m_progInfo.GetProgramID(), ProgInfoList::kLevel1);

    // Begin MythTV information not found in the listings info
    QDateTime statusDate;
    if (m_progInfo.GetRecordingStatus() == RecStatus::WillRecord ||
        m_progInfo.GetRecordingStatus() == RecStatus::Pending)
        statusDate = m_progInfo.GetScheduledStartTime();

    RecordingType rectype = kSingleRecord; // avoid kNotRecording
    RecStatus::Type recstatus = m_progInfo.GetRecordingStatus();

    if (recstatus == RecStatus::PreviousRecording ||
        recstatus == RecStatus::NeverRecord ||
        recstatus == RecStatus::Unknown)
    {
        query.prepare("SELECT recstatus, starttime "
                      "FROM oldrecorded WHERE duplicate > 0 AND "
                      "future = 0 AND "
                      "((programid <> '' AND programid = :PROGRAMID) OR "
                      " (title <> '' AND title = :TITLE AND "
                      "  subtitle <> '' AND subtitle = :SUBTITLE AND "
                      "  description <> '' AND description = :DECRIPTION));");

        query.bindValue(":PROGRAMID",  m_progInfo.GetProgramID());
        query.bindValue(":TITLE",      m_progInfo.GetTitle());
        query.bindValue(":SUBTITLE",   m_progInfo.GetSubtitle());
        query.bindValue(":DECRIPTION", m_progInfo.GetDescription());

        if (!query.exec())
        {
            MythDB::DBError("showDetails", query);
        }
        else if (query.next())
        {
            if (recstatus == RecStatus::Unknown)
                recstatus = RecStatus::Type(query.value(0).toInt());

            if (recstatus == RecStatus::PreviousRecording ||
                recstatus == RecStatus::NeverRecord ||
                recstatus == RecStatus::Recorded)
            {
                statusDate = MythDate::as_utc(query.value(1).toDateTime());
            }
        }
    }

    if (recstatus == RecStatus::Unknown)
    {
        if (recorded)
        {
            recstatus = RecStatus::Recorded;
            statusDate = m_progInfo.GetScheduledStartTime();
        }
        else
        {
            // re-enable "Not Recording" status text
            rectype = m_progInfo.GetRecordingRuleType();
        }
    }

    s = RecStatus::toString(recstatus, rectype);

    if (statusDate.isValid())
        s += " " + MythDate::toString(statusDate, MythDate::kDateFull |
                                      MythDate::kAddYear);

    addItem(tr("MythTV Status"), s, ProgInfoList::kLevel1);

    QString recordingRule;
    QString lastRecorded;
    QString nextRecording;
    QString averageTimeShift;
    QString watchListScore;
    QString watchListStatus;
    QString searchPhrase;

    if (m_progInfo.GetRecordingRuleID())
    {
        recordingRule = QString("%1, ").arg(m_progInfo.GetRecordingRuleID());
        if (m_progInfo.GetRecordingRuleType() != kNotRecording)
            recordingRule += toString(m_progInfo.GetRecordingRuleType());
        if (!(record->m_title.isEmpty()))
            recordingRule += QString(" \"%2\"").arg(record->m_title);

        query.prepare("SELECT last_record, next_record, avg_delay "
                      "FROM record WHERE recordid = :RECORDID");
        query.bindValue(":RECORDID", m_progInfo.GetRecordingRuleID());

        if (query.exec() && query.next())
        {
            if (query.value(0).toDateTime().isValid())
                lastRecorded = MythDate::toString(
                    MythDate::as_utc(query.value(0).toDateTime()),
                    MythDate::kDateFull | MythDate::kAddYear);
            if (query.value(1).toDateTime().isValid())
                nextRecording = MythDate::toString(
                    MythDate::as_utc(query.value(1).toDateTime()),
                    MythDate::kDateFull | MythDate::kAddYear);
            if (query.value(2).toInt() > 0)
                averageTimeShift = tr("%n hour(s)", "",
                                                query.value(2).toInt());
        }
        if (recorded)
        {
            if (m_progInfo.GetRecordingPriority2() > 0)
                watchListScore =
                    QString::number(m_progInfo.GetRecordingPriority2());

            if (m_progInfo.GetRecordingPriority2() < 0)
            {
                switch (m_progInfo.GetRecordingPriority2())
                {
                    case wlExpireOff:
                        watchListStatus = tr("Auto-expire off");
                        break;
                    case wlWatched:
                        watchListStatus = tr("Marked as 'watched'");
                        break;
                    case wlEarlier:
                        watchListStatus = tr("Not the earliest episode");
                        break;
                    case wlDeleted:
                        watchListStatus = tr("Recently deleted episode");
                        break;
                }
            }
        }
        if (record->m_searchType != kManualSearch &&
            record->m_description != m_progInfo.GetDescription())
            searchPhrase = record->m_description;
    }
    addItem(tr("Recording Rule"), recordingRule, ProgInfoList::kLevel1);
    addItem(tr("Search Phrase"), searchPhrase, ProgInfoList::kLevel1);

    s.clear();
    if (m_progInfo.GetFindID())
    {
        QDateTime fdate(QDate(1970, 1, 1),QTime(12,0,0));
        fdate = fdate.addDays((int)m_progInfo.GetFindID() - 719528);
        s = QString("%1 (%2)").arg(m_progInfo.GetFindID())
            .arg(MythDate::toString(
                     fdate, MythDate::kDateFull | MythDate::kAddYear));
    }
    addItem(tr("Find ID"), s, ProgInfoList::kLevel2);

    addItem(tr("Last Recorded"), lastRecorded, ProgInfoList::kLevel2);
    addItem(tr("Next Recording"), nextRecording, ProgInfoList::kLevel2);
    addItem(tr("Average Time Shift"), averageTimeShift, ProgInfoList::kLevel2);
    addItem(tr("Watch List Score"), watchListScore, ProgInfoList::kLevel2);
    addItem(tr("Watch List Status"), watchListStatus, ProgInfoList::kLevel2);

    QString recordingHost;
    QString recordingInput;
    QString recordedFilename;
    QString recordedFileSize;
    QString recordingGroup;
    QString storageGroup;
    QString playbackGroup;
    QString recordingProfile;

    recordingHost = m_progInfo.GetHostname();
    recordingInput = m_progInfo.GetInputName();

    if (recorded)
    {
        recordedFilename = m_progInfo.GetBasename();
        recordedFileSize = QString("%1 ")
            .arg(m_progInfo.GetFilesize()/((double)(1<<30)),0,'f',2);
        recordedFileSize += tr("GB", "GigaBytes");

        query.prepare("SELECT profile FROM recorded"
                      " WHERE chanid = :CHANID"
                      " AND starttime = :STARTTIME;");
        query.bindValue(":CHANID",    m_progInfo.GetChanID());
        query.bindValue(":STARTTIME", m_progInfo.GetRecordingStartTime());

        if (query.exec() && query.next())
        {
            recordingProfile = m_progInfo.i18n(query.value(0).toString());
        }
        recordingGroup = m_progInfo.i18n(m_progInfo.GetRecordingGroup());
        storageGroup   = m_progInfo.i18n(m_progInfo.GetStorageGroup());
        playbackGroup  = m_progInfo.i18n(m_progInfo.GetPlaybackGroup());
    }
    else if (m_progInfo.GetRecordingRuleID())
    {
        recordingProfile =  record->m_recProfile;
    }
    addItem(tr("Recording Host"), recordingHost, ProgInfoList::kLevel2);
    addItem(tr("Recording Input"), recordingInput, ProgInfoList::kLevel2);
    addItem(tr("Recorded File Name"), recordedFilename, ProgInfoList::kLevel1);
    addItem(tr("Recorded File Size"), recordedFileSize, ProgInfoList::kLevel1);
    addItem(tr("Recording Profile"), recordingProfile, ProgInfoList::kLevel2);
    addItem(tr("Recording Group"), recordingGroup, ProgInfoList::kLevel1);
    addItem(tr("Storage Group"), storageGroup, ProgInfoList::kLevel2);
    addItem(tr("Playback Group"),  playbackGroup, ProgInfoList::kLevel2);

    delete record;
}
