; Copyright (C) 2024, ForrestHunt, Inc.
; Written by J Moore
; License: A 3-clause BSD license.  See the LICENSE file distributed with ACL2.

; df-solver-v9:
; A Gaussian Elimination Solver using Double-float Arithmetic
; J Moore
; February, March, and April, 2024

; Note: Matt Kaufmann improved performance of this code by about 38% by
; experimenting with the defstobj options.  He finally settled on adding
; :inline t and :non-memoizable t to each defstobj.  He also considered
; :element-type t but found that it produced worse performance than leaving it
; out.

; This file is distributed in conjunction with the paper ``ACL2 Support for
; Floating-Point Calculations,'' by Kaufmann and Moore, (to appear, 2024).

; This is a work in progress for solving Ax = b, for sparse matrices A.  I am
; not happy with this solver's performance on sparse matrices of size larger
; than about 1000 x 1000: the basic algorithm is cubic in the dimension of A.
; But this code is a good way to test the new df arithmetic.  This is the 9th
; step in evolution of a Gaussian elimination solver using the df features
; introduced into ACL2 in early 2024.  The initial version of this code, v1,
; (February, 2024) used lists for basically everything; this version uses
; arrays in stobjs (and stobjs within stobjs) for everything.  The evolution
; was carried out step-by-step to assess the performance effects of each
; change.

; The supplemental file big-a-and-b.lsp contains a 1036 x 1036 matrix, called
; big-a, generated by the VWSIM circuit simulation tool.  This matrix was used
; to benchmark the solver, as reported in the paper cited above, ``ACL2 Support
; for Floating-Point Calculations.''  Instructions for running that test are
; included in the section named A Larger Example below.

; This file is divided into sections by lines of equal signs.  If you're
; exploring it for the first time I recommend reading everything down to the
; section headed: The Code, and then just reading the section headings.

; =================================================================
; Loading

; This is a book, but everything in it is in :program mode.  You may
; certify it with

; (certify "df-solver-v9")

; and then include it with

; (include-book "df-solver-v9")
; (set-check-invariant-risk nil)

; =================================================================
; General Scheme

; This is a Gaussian elimination solver optimized for sparse matrices.  It
; finds an approximate double-float solution to Ax = b, for sparse square
; matrix A and vector b.

; As with previous solvers produced in this project, this solver operates in
; two steps or ``phases.''  In phase1 the matrix A is put into triangular form.
; In phase2, the data structures produced by phase1 together with a vector b
; are used to find x such that Ax = b.  Thus, phase1 can be run once for a
; given A and then phase2 can be run for a series of vectors b.  Since each run
; of phase2 overwrites the array x containing the solution previously given b,
; you must use or extract and save the solution before moving on to the next b.

; All data is stored in a single stobj, dfst.  The input matrix A and vector b
; are presented as lists in the sparse representation (SR) format used in
; previous solvers.  (We define SR format below.)  Two ``load'' functions
; (named in the guide below) transfer the data into the dfst stobj.  If you
; wish to avoid creating lists to specify your A and b, inspect those loaders
; and their uses in the functions phase1 and phase2, to see how I configure
; dfst.  But in order to make this package stand-alone, I use SR format for
; input here.

; =================================================================
; SR Format

; In SR format, a vector (or a row of A) is a list of non-zero df rationals,
; each optionally preceded by a singleton lists containing a positive natural.
; The singleton lists represent contiguous sequences of n zeros, called
; ``gaps,'' where n is the listed natural.

; For example, the vector of length 10:

; (#d0.0 #d0.0 #d1.0 #d2.0 #d0.0 #d0.0 #d3.0 #d0.0 #d0.0 #d0.0)

; in the SR format is

; ((2) #d1.0 #d2.0 (2) #d3.0)

; It is important that the last gap is implicit.  The SR list above only
; specifies the first 7 entries.  But since the dimension of the problem, e.g.,
; 10 in this example, is always known in advance, there is no need to specify
; the last gap.  Indeed, it is illegal.

; You might be tempted to write

; ((2) #d1.0 #d2.0 (2) #d3.0 (3)) ; <-- This is illegal; omit the last gap!

; Also, the df values must be non-zero.  You might be tempted to write

; (#d0.0 #d0.0 #d1.0 #d2.0 (2) #d3.0) ; <-- This is illegal; omit df zeros!

; If you want one zero between two non-zero dfs, you must use a gap of 1, i.e.,
; (1).

; An n x n matrix A is just a list of n SR rows (none of length greater than
; n).

; =================================================================
; Extracted Solution

; After running phase2 the solution can be extracted as a list of df rationals
; in one of three styles of displaying df values.  The styles are determined
; by a flag:

; nil means return a list of dfp rationals
; t   means return a list of strings like "#d1.234E-4"
; cltl means return a list of strings like "1.234E-4"

; We show examples below.  But note that the list of values is not in SR format
; but is a list of length dim specifying every value.  (Solutions are not
; generally very sparse.)  This is generally useful for small tests where you
; want to be able to read the values.

; =================================================================
; Examples

; We first show a call of the most-top-level solver.  It checks the inputs for
; well-formedness, solves the problem (if possible), and extracts the solution
; as a list of df values displayed in one of three styles.

; (include-book "df-solver-v8")
; (set-check-invariant-risk nil)

; (defconst *classic-a*
;    '((#d5.0D0 #d1.0D0 #d2.0D0)  ; matrix ``A''
;      (#d7.0D0 #d9.0D0 #d-3.0D0)
;      ((1)     #d1.0D0 #d-4.0D0)))

; (defconst *classic-b* '(#d13.0D0 #d16.0D0 #d-10.0D0)) ; vector ``b''

; The solution to Ax = b for the above choice of the 3x3 matrix A and the
; vector b is the vector (1 2 3).  We extract the solution in each of the
; three styles: nil, t, cltl.

; (solver! nil   *classic-a* *classic-b* dfst)
; ==>
; ((2251799813685249/2251799813685248 9007199254740991/4503599627370496 3)
;  <dfst>)

; (solver! t     *classic-a* *classic-b* dfst)
; ==>
; (("#d1.0000000000000004" "#d1.9999999999999998" "#d3.0")
;  <dfst>)

; (solver! 'cltl *classic-a* *classic-b* dfst)
; ==>
; (("1.0000000000000004" "1.9999999999999998" "3.0")
;  <dfst>)

; Of these, only the nil style is readable back into ACL2 as a list of dfp
; (rational) objects.  Note that the other two ``vectors'' contain strings, not
; numbers.  However, if you print the strings, as with the ~s directive of fmt,
; the t style can be read as a list of df objects by ACL2 and the cltl style
; can be read as a list of double-floats in raw Common Lisp.

; Solver!, called above checks well-formedness of the input and extracts the
; solution (when successful).  We could use

; (solver *classic-a* *classic-b* 3 dfst)
; ==>
; (T <dfst>)

; The T value means success.

; Unlike solver!, solver doesn't bother to check well-formedness or extract the
; solution.  Note that we must supply the dimension, 3, of this problem as well
; as the matrix and vector.

; After calling solver we might access the elements of the solution directly:

; (xi 0 dfst)
; ==>
; #d1.0000000000000004

; (xi 1 dfst)
; ==>
; #d1.9999999999999998

; (xi 2 dfst)
; ==>
; #d3.0

; (The numbers are printed in the #d style because the signature of xi says the
; value is a :df.  But unfortunately we can't print a list of :dfs.  Instead, we
; must print strings.)

; (extract-solution t 2 dfst nil)
; ==>
; ("#d1.0000000000000004" "#d1.9999999999999998" "#d3.0")

; (The 2 in the call above is one less than the dimension.  Extract-solution
; recovers (xi 2 dfst), then (xi 1 dfst), and then (xi 0 dfst), accumulating
; the answers into the last argument, initially nil, above.)

; More useful than the ``total packages'' provided by solver! and solver might
; be the two phases.  Using them you can reduce A to triangular form and then
; solve Ax = b for various choices of b without re-processing A.

; (phase1 *classic-A* 3 dfst) ; note dimension supplied
; ==>
; (T <dfst>)

; (phase2 *classic-b* 3 dfst)
; ==>
; <dfst>

; (extract-solution t 2 dfst nil)
; ==>
; ("#d1.0000000000000004" "#d1.9999999999999998" "#d3.0")

; But we can also solve for a different b.  (In this example
; we just double the values in *classic-b*, expecting to get a
; solution that is double the values in the previous one.)

; (phase2 '(#d26.0 #d32.0 #d-20.0) 3 dfst)
; ==>
; <dfst>

; (extract-solution t 2 dfst nil)
; ==>
; ("#d2.000000000000001" "#d3.9999999999999996" "#d6.0")

; =================================================================
; A Larger Example

; To run the 1036 x 1036 ``big-a'' problem, after this file has been loaded,
; and invariant risk checking has been disabled (as instructed above),

; (ld "big-a-and-b.lsp")
; (time$ (solver! t (@ big-a) (@ big-b) dfst))

; Or to see how accurate the answer is, try

; (Ax=b? (@ big-a) (@ big-b) dfst)

; But our benchmarking convention has been to time only phase1 of the solver,
; in which A is put in triangular form (where 99% of the time goes).
; Furthermore, we typically do that in raw Lisp where the time is measured more
; accurately and where the ACL2 logic's read-eval-print machinery is avoided.
; To do so:

; (value :q) ; drop into raw Lisp
; (mv-let (okp dfst)
;         (time (phase1 (@ big-a) 1036 (cdr (assoc-eq 'dfst *user-stobj-alist*))))
;         (let ((dfst (phase2 (@ big-b) 1036 dfst)))
;           (extract-solution t (- 1036 1) dfst nil)))

; On a MacBook Pro running ACL2 in SBCL, the time command above printed

; Evaluation took:
;   0.000 seconds of real time
;   0.000928 seconds of total run time (0.000854 user, 0.000074 system)
;   100.00% CPU
;   2,222,552 processor cycles
;   327,680 bytes consed

; and the value of the mv-let form above is the same solution printed by the
; solver! call above.  Of course, times vary depending on CPU usage and storage
; allocation and garbage collection times.

; =================================================================
; Guide to Important Functions Defun'd Below

; (SOLVER! flg A b dfst) -- checks A and b for proper format, solves Ax = b (if
;   possible), and extracts the answer x into a list in which the dfs are shown
;   as rationals or strings as per flg.  Returns (mv extracted-x dfst), where
;   if the extracted-x is NIL if Ax = b is unsolvable.

; (SOLVER A b dfst) -- like solver! but does just assumes the input is
;   well-formed and does not extract the answer, it just leaves the answer in
;   the x array of dfst.

; (PHASE1 A dim dfst) -- loads A (which is assumed to be well-formed) and of
;   dimension dim x dim, into dfst and puts A into triangular form.  Returns
;   (mv okp dfst), where okp = t means the reduction was successful and okp =
;   nil means reduction to triangular form failed (e.g., matrix A was
;   singular).

; (PHASE2 b dim dfst) -- loads b (which is assumed to be well-formed) and
;   computes an df approximate to x such that Ax = b.  It is assumed phase1 has
;   been run on A with an okp value of t, and that the data in dfst thus
;   produced hasn't been changed.  Phase2 does not change the data produced by
;   phase1.  The answer is left in the x array of dfst.  Dfst is returned.

; Experienced ACL2 users should have no trouble figuring out the use of certain
; utilities by inspecting the functions above.  For example,

; SOLVER! uses
;   CHECK-WELL-FORMED-INPUT to check the input, and
;   EXTRACT-SOLUTION to extract the solution into a list.
; PHASE1 uses
;   INITIALIZE-DFST to properly size all the arrays in dfst
;   LOAD-ALL-ROWS to load the given A into dfst
;   BLOCKIFY to organize (permute) the rows according to number
;     of leading zeros
;   CANCEL-ALL to perform row operations and permutations to
;     reduce to triangular form.
; PHASE2 uses
;   LOAD-B-INTO-DFST to load the given b into dfst.
;   RUN-B-PROGRAM to transform b in accordance with the
;    row operations and permutations of phase1
;   SOLVE-FOR-ALL-VARS to compute and store the solution into x

; At the very end there is a section named Checking the Answers
; which provides a function named AX=B?.

; If you've run the solver (or solver! or a non-erroneous run of phase1
; followed by phase2) you can call Ax=b? on that A and b and dfst to see the
; absolute Euclidean distance between Ax and b and also the distance relative
; to the length of the vector b.

; =================================================================
; The Code

; Preliminaries, the stobjs within the main stobj dfst, and some basic
; recognizers for SR format.

(in-package "ACL2")

(program)
(set-verify-guards-eagerness 0)

(include-book "projects/apply/top" :dir :system)

(defstobj row
  (coef :type (array double-float (1)) :initially #d1.0D0 :resizable t
        )
  (gap :type (array integer (2)) :initially -1 :resizable t
       )
  :inline t :non-memoizable t)

; Note that the gap array must always be at least one longer than the coef
; array.  Otherwise, we could never use the last slot of the coef array.

(defstobj row1
  (row1-coef :type (array double-float (1)) :initially #d1.0D0 :resizable t
             )
  (row1-gapi :type (array integer (2)) :initially -1 :resizable t
             )
  :congruent-to row
  :inline t :non-memoizable t)

(defstobj row2
  (row2-coef :type (array double-float (1)) :initially #d1.0D0 :resizable t
             )
  (row2-gap :type (array integer (2)) :initially -1 :resizable t
            )
  :congruent-to row
  :inline t :non-memoizable t)

(defstobj xrow ; ``extra row''
  (xcoef :type (array double-float (1)) :initially #d1.0D0 :resizable t
         )
  (xgap :type (array integer (2)) :initially -1 :resizable t
        )
  :congruent-to row
  :inline t :non-memoizable t)

; The initial field values below are reset by blockify initialize-dfst,
; load-all-rows, blockify, load-b-into-dfst, and (of course) the solver itself.

(defstobj dfst
  (dim :type (unsigned-byte 30) :initially 1)
  (a :type (array row (1)) :resizable t
     )
  (scratch :type row)
  (pivot :type (array integer (1)) ; dimension will be set by blockify
         :initially -1
         :resizable t
         )
  (nx :type (array integer (1)) ; dimension will be set by blockify
      :initially -1
      :resizable t
      )
  (b :type (array double-float (1))
     :initially #d0.0
     :resizable t
     )
  (x :type (array double-float (1))
     :initially #d0.0
     :resizable t
     )
  (b-prog-target-index :type (array integer (1))
                       :initially -1
                       :resizable t
                       )
  (b-prog-pivot-index  :type (array integer (1))
                       :initially -1
                       :resizable t
                       )
  (b-prog-pivot-scalar :type (array double-float (1))
                       :initially #d0.0D0
                       :resizable t
                       )
  (b-prog-free-ptr :type (unsigned-byte 30) :initially 0)
  :inline t :non-memoizable t)

(defun sr-dfp-vectorp (x)
  (cond ((atom x) (null x))
        ((consp (car x))
         (and (posp (car (car x))) ; non-zero number of zeros
              (null (cdr (car x)))
              (not (atom (cdr x)))        ; not the last element
              (not (consp (car (cdr x)))) ; not adjacent to another gap
              (sr-dfp-vectorp (cdr x))))
        ((dfp (car x))
         (and (not (eql (car x) 0))
              (sr-dfp-vectorp (cdr x))))
        (t nil)))

(defun sr-dfp-vlen (x)
; The ``virtual len'' of a row is the sum of the lengths of the gap
; plus the number of coefficients.
  (cond ((endp x) 0)
        ((consp (car x))
         (+ (car (car x)) (sr-dfp-vlen (cdr x))))
        (t (+ 1 (sr-dfp-vlen (cdr x))))))

(defun sr-dfp-matrixp (m)
  (cond ((atom m) (null m))
        (t (and (sr-dfp-vectorp (car m))
                (sr-dfp-matrixp (cdr m))))))

(defun non-zeros-in-row (lst)
  (cond ((endp lst) 0)
        ((consp (car lst)) (non-zeros-in-row (cdr lst)))
        (t (+ 1 (non-zeros-in-row (cdr lst))))))

(defun max-non-zeros-in-row (m max)
  (cond ((endp m) max)
        (t (let ((nzcnt (non-zeros-in-row (car m))))
             (max-non-zeros-in-row (cdr m) (max nzcnt max))))))

(defun max-sr-dfp-vlen (m max)
  (cond ((endp m) max)
        (t (let ((vlen (sr-dfp-vlen (car m))))
             (max-sr-dfp-vlen (cdr m) (max vlen max))))))

(defun square-sr-dfp-matrixp (m)
  (and (sr-dfp-matrixp m)
       (<= (max-sr-dfp-vlen m 0) (length m))))

; =================================================================
; Resizing Things

(defun resize-row (row-allocation row)

; Resize the coef and gap arrays of row to be at least row-allocation,
; while maintaining the invariant that the gap array is always exactly
; one longer than the coef array.

  (declare (xargs :stobjs (row)))
  (let ((clen (coef-length row))
        (glen (gap-length row)))
    (cond
     ((= glen (+ 1 clen)) ; invariant holds initially
      (cond
       ((>= clen row-allocation)
        row)
       (t (let* ((row (resize-gap (+ 1 row-allocation) row))
                 (row (resize-coef row-allocation row)))
            row))))
     (t (prog2$
         (er hard? 'resize-row
             "The invariant that the length of the gap array in a row is ~
              always one greater than the length of the coef array in that ~
              row is violated in the row where (gap-length row) = ~x0 but ~
              (coef-length row) = ~x1."
             (gap-length row)
             (coef-length row))
         row)))))

(defbadge ai)
(defbadge update-ai)
(defbadge resize-row)

(defun resize-all-rows (row-allocation i dim dfst)

; Resize every row in dfst so that the gap arrays each have length at least
; row-allocation and the coef arrays each have length at least row-allocation.
; If one of these arrays is already longer than row-allocation, we leave it as
; is.

  (declare (xargs :stobjs dfst))
  (loop$ with i of-type (integer 0 *) = i
         do
         :measure (- dim i)
         :values (dfst)
         (cond
          ((>= i dim) (return dfst))
          (t (progn
               (setq dfst
                     (stobj-let ((row (ai i dfst)))
                        (row)
                        (resize-row row-allocation row)
                        dfst))
               (setq i (+ 1 i)))))))

; The only time we resize most rows is when we initialize dfst for a new
; matrix.  But we will dynamically grow rows during the basic operation of
; add-mult-rows.  At that time the row we're resizing is the extra row stored
; in (scratch dfst), but rows from A cycle in and out of scratch as they become
; targets.  Below are the two functions used to update the gap and coef
; elements of the extra row, which handle dynamic resizing of rows.

(defun update-xgapi! (xi val xrow dim)
  (declare (type (integer 0 *) xi)
           (type (integer -1 *) val)
           (xargs :stobjs (xrow)))
  (let ((gmax (gap-length xrow)))
    (cond
     ((< xi gmax)
      (update-gapi xi val xrow))
     (t ; xi = gmax = (coef-length xrow)+1
      (let* ((row-allocation (- gmax 1))
             (xrow (resize-row (min (* 2 row-allocation) dim) xrow)))
        (update-xgapi xi val xrow))))))

(defun update-xcoefi! (xi val xrow dim)
  (declare (type (integer 0 *) xi)
           (type double-float val)
           (xargs :stobjs (xrow)))
  (let ((cmax (coef-length xrow)))
    (cond
     ((< xi cmax)
      (update-xcoefi xi val xrow))
     (t ; xi = cmax
      (let ((xrow (resize-row (min (* 2 cmax) dim) xrow)))
        (update-xcoefi xi val xrow))))))

(defun resize-b-prog-arrays (k dfst)
  (declare (xargs :stobjs dfst))
  (if (< (b-prog-target-index-length dfst) k)
      (let* ((dfst (resize-b-prog-target-index k dfst))
             (dfst (resize-b-prog-pivot-index k dfst))
             (dfst (resize-b-prog-pivot-scalar k dfst)))
        dfst)
      dfst))

(defun add-b-prog-instr (target-index scalar pivot-index dfst)

; Abstractly, the b-program is a sequence of instructions of the form:

; bi[target-index] <-- (+ (* scalar bi[pivot-index]) bi[target-index])

; but it is coded by three arrays in dfst (plus a free pointer), one array each
; for holding the target indices, the pivot scalars and the pivot indices.
; This function adds one new instruction.

  (declare (type (integer 0 *) target-index pivot-index)
           (type double-float scalar)
           (xargs :stobjs (dfst)))
  (let* ((free-ptr (b-prog-free-ptr dfst))
         (dfst (if (>= free-ptr
                       (b-prog-target-index-length dfst))
                   (resize-b-prog-arrays (* 2 (b-prog-target-index-length dfst))
                                         dfst)
                   dfst))
         (dfst (update-b-prog-target-indexi free-ptr target-index dfst))
         (dfst (update-b-prog-pivot-scalari free-ptr scalar dfst))
         (dfst (update-b-prog-pivot-indexi  free-ptr pivot-index dfst))
         (dfst (update-b-prog-free-ptr (+ 1 free-ptr) dfst)))
    dfst))

(defbadge update-pivoti)
(defbadge update-nxi)

(defun clear-pivot-and-nx-arrays (dfst)

; When we initialize-dfst we clear the pivot and nx arrays of whatever
; information was left in them.  If we don't, we're in danger of creating a
; circular chain.

  (declare (xargs :stobjs (dfst)))
  (loop$ with i of-type (integer 0 *) = (dim dfst)
         do
; Note that i is ``one off.''  We write to position i-1 and stop when i is 0.
         :measure (nfix i)
         :values (dfst)
         (cond
          ((= i 0) (return dfst))
          (t (progn

               (setq dfst (update-pivoti (- i 1) -1 dfst))
               (setq dfst (update-nxi (- i 1) -1 dfst))
               (setq i (- i 1)))))))

; On the Initial Allocation of Spaces for the gaps and
; coefficients in A.

; Initially, no row needs to be longer than the maximum number of non-zero
; coefficients in any row of m, i.e., (max-non-zeros-in-row m 0).  But as soon
; as we perform the basic operation (add-mult-rows) on two rows we might have
; to resize the target row to hold all the non-zeros.  In the worst case, the
; basic operation might transform two rows with k and n non-zeros respectively
; into a row with k+n-1 non-zeros.  It is possible that the top-most row in the
; eventual LU triangle has dim non-zeros.  So we will have to allocate extra
; space to some rows dynamically.  But we'd like to delay that and not have to
; resize every row on every basic operation.  So we will initially allocate a
; fraction of the dimension to each row.  That fraction is (will be)
; empirically determined and stored in *initial-row-dim-ratio*.  For the moment
; assume it is 1/8.

; So if the dimension of the matrix is 1000 each row will initially be
; allocated one eighth of that, or 125 slots.  This accommodates 125 gaps and
; 125 coefficients.  It will then grow dynamically as needed.

; However, we must be able to store every non-zero coefficient in each row of
; the initial matrix because we don't want to have to resize a row while
; loading the arrays.  (For example, in the ``classic'' 3x3 A that I use for
; the first test of the solver, the first row has 3 non-zeros and so the
; fractional allocation just won't work.)  So in initialization we will check
; that the envisioned fractional allocation is adequate.  If not, we'll
; allocate each row the maximal number number of non-zeros in any row.  This is
; overkill but we don't really expect it to happen on actual data.

(defconst *initial-row-dim-ratio* 1/8)

(defun initialize-dfst (m dim dfst)

; This function assumes that m is a well-formed square matrix in SR format in
; which the coefficients are all dfps.  That could be checked by the
; conjunction in the :guard below.

  (declare (xargs :stobjs (dfst)
                  :guard (and (square-sr-dfp-matrixp m)
                              (equal dim (length m)))))
  (let* ((max-nz (max-non-zeros-in-row m 0))
         (temp (ceiling (* dim *initial-row-dim-ratio*) 1))
         (row-allocation (if (<= max-nz temp)
                             temp
                             max-nz)))

; The allocation above guarantees that we can load m into the rows of dfst.

    (cond
     ((and (equal dim (dim dfst))
           (<= max-nz temp))

; We assume that (dim dfst) reflects the current size of all the arrays in dfst.
; If max-nz (for the m we're dealing with now) is less than temp we know that
; all the rows in this m will fit and we need do nothing except clear the
; pivot and nx chains.

      (let* ((dfst (clear-pivot-and-nx-arrays dfst))
             (dfst (update-b-prog-free-ptr 0 dfst)))
        dfst))
     (t (let* ((dfst (resize-a dim dfst))
               (dfst (resize-all-rows row-allocation 0 dim dfst))
               (dfst (stobj-let ((row (scratch dfst)))
                                (row)
                                (resize-row row-allocation row)
                                dfst))
               (dfst (if (>= (pivot-length dfst) dim)
                         dfst
                         (resize-pivot dim dfst)))
               (dfst (if (>= (nx-length dfst) dim)
                         dfst
                         (resize-nx dim dfst)))
               (dfst (if (>= (b-length dfst) dim)
                         dfst
                         (resize-b dim dfst)))
               (dfst (if (>= (x-length dfst) dim)
                         dfst
                         (resize-x dim dfst)))

; The 5 below is completely arbitrary!  It would be good to collect data on the
; final values of (b-prog-free-ptr dfst) after several runs on matrices of
; different sizes.

               (dfst (resize-b-prog-arrays (* 5 dim) dfst))
               (dfst (update-b-prog-free-ptr 0 dfst))
               (dfst (update-dim dim dfst))
; Note: Clearing the pivot and nx chains depends on the proper setting of (dim
; dfst), so we waited until here to do it.
               (dfst (clear-pivot-and-nx-arrays dfst)))
          dfst)))))

; =================================================================
; Loading A into dfst

(defun load-row (lst i row)

; We copy the gap and coefficients of the SR-dfp-vector lst into the gap and
; coef arrays of stobj row, starting at index i (initially 0).  We know that
; initialize-dfst has ensured that each gap and coef array of row is long
; enough to accept every assignment below.  When we reach the end of lst we
; write a -1 into gap to mark the end of the (active part of the) row.

  (declare (xargs :stobjs (row)
                  :guard (and (sr-dfp-vectorp lst)
                              (natp i)
                              (< i (gap-length row))
                              (equal (gap-length row)
                                     (+ 1 (coef-length row))))))
  (cond
   ((endp lst) (update-gapi i -1 row))
   ((consp (car lst))
    (let* ((row (update-gapi i (car (car lst)) row))
           (row (update-coefi i (to-df (cadr lst)) row)))
      (load-row (cddr lst) (+ 1 i) row)))
   (t (let* ((row (update-gapi i 0 row))
             (row (update-coefi i (to-df (car lst)) row)))
        (load-row (cdr lst) (+ 1 i) row)))))

(defbadge load-row)

(defun load-all-rows (m dfst)

; We load all rows of the square SR dfp matrix m into the rows of dfst.
; Warning: Dfst must have been properly initialized with initialize-dfst before
; this operation!

  (declare (xargs :stobjs (dfst)
                  :guard (square-sr-dfp-matrixp m)))
  (loop$ with i of-type (integer 0 *) = 0
         with m of-type (satisfies true-listp) = m
         do
         :measure (acl2-count m)
         :values (dfst)
         (cond
          ((endp m) (return dfst))
          (t (progn
               (setq dfst (stobj-let ((row (ai i dfst)))
                                     (row)
                                     (load-row (car m) 0 row)
                                     dfst))
               (setq m (cdr m))
               (setq i (+ 1 i)))))))

(defun leading-coef (row)
  (declare (xargs :stobjs (row)))
  (coefi 0 row))

(defun zcnt (row)
  (declare (xargs :stobjs (row)))
  (the (integer 0 *)
       (gapi 0 row)))

; =================================================================
; Block Structure

; After loading all the rows of the matrix into the row arrays, we organize
; them into blocks by the number of leading zeros.  We think of a block as
; being composed of rows, but actually they are composed of row indices, where
; each index is the original position of the row in A.  All the rows in block n
; have n leading zeros.  The first row in the block is the pivot row, which is
; the row of the block with the maximum absolute value of the leading non-zero
; coefficient.  The (index of the pivot) row of block n is (pivoti n dfst).  If
; i is the index of a row in a block, then (nxi i dfst) is the index of the
; successor row.  Thus, (nxi (pivoti n dfst) dfst) is the index of the row
; right after the pivot row of block n.  We chase down the nxi chain until we
; reach an ``index'' of -1 which serves as the terminal marker of the sequence
; of rows in a block.

; Each row occupies a slot in A and the index of that slot is the index of the row.
; We now insert the index of each row into the block structure.

(defun insert-row-index-into-blk (i new-zcnt coef dfst)

; We want to insert row (ai i dfst) into the appropriate block -- which really
; means we want to insert i into the block for (zcnt (ai i row)), making it the
; pivot if appropriate.  It is critical that i not be in any chain, i.e., no
; pivoti and no nxi contains i before this operation.

; This function is called in two circumstances: The first is when we set up the
; blocks (in which case the block chain starts completely empty and each i
; is new).  The second is when cancel-blk1 has canceled the row after the
; pivot of a block -- producing a new row which has a zcnt greater than
; that of the block of the pivot.  Cancel-blk1 uses this function to insert the
; modified row into its proper block.  But before doing so it removes it from
; the chain!

  (declare (type (integer 0 *) i new-zcnt)
           (type double-float coef)
           (xargs :stobjs (dfst)))
  (let ((index-of-zcnt-pivot (pivoti new-zcnt dfst)))
    (cond
     ((= index-of-zcnt-pivot -1)
      (let ((dfst (update-pivoti new-zcnt i dfst)))
        (update-nxi i -1 dfst)))
     ((df< (df-abs (stobj-let ((row (ai index-of-zcnt-pivot dfst)))
                              (val)
                              (leading-coef row)
                              val))
           (df-abs coef))
      (let ((dfst (update-pivoti new-zcnt i dfst)))
        (update-nxi i index-of-zcnt-pivot dfst)))
     (t (let ((dfst (update-nxi i (nxi index-of-zcnt-pivot dfst) dfst)))
          (update-nxi index-of-zcnt-pivot i dfst))))))

(defbadge insert-row-index-into-blk)
(defbadge zcnt)
(defbadge leading-coef)

(defun blockify (dim dfst)

; We map over lst, which initially is the input matrix of rows, pair each row
; with its original position in the matrix, and then insert it into the block
; to which it belongs according to its zcnt.  Insertion always keeps the irow
; with the largest (absolute value) leading coef as the pivot of each block.

  (declare (type (unsigned-byte 30) dim)
           (xargs :stobjs (dfst)))
  (loop$ with i of-type (integer 0 *) = 0
         do
         :measure (- dim i)
         :values (dfst)
         (cond
          ((>= i dim) (return dfst))
          (t (mv-let (zcnt coef)
               (stobj-let ((row (ai i dfst)))
                          (val1 val2)
                          (mv (zcnt row) (leading-coef row))
                          (mv val1 val2))
               (declare (type (integer 0 *) zcnt)
                        (type double-float coef))

; Note: Initially I tried to define insert-row-index-into-blk: (i row dfst) ==>
; dfst.  That failed because row was being extracted from dfst and then passed
; to function that changed dfst, raising the possibility that row itself was
; inserted into a second place in dfst, destroying single-threadedness.  (Row
; is not inserted: i is inserted.  But we don't analyze that deeply!)  So now
; the stobj-let above extracts row from dfst, then digs out the two things
; insert-row-index-into-blk needs from it (its zcnt and leading-coef), and then
; passes those two non-stobjs into insert-row-index-into-blk.

               (progn
                 (setq dfst (insert-row-index-into-blk i zcnt coef dfst))
                 (setq i (+ 1 i))))))))

; =================================================================
; Debugging Tools

; None of this need concern the user!  Skip to the next section.

(defun prettyify-row1 (i max row)
  (declare (xargs :stobjs (row)))
  (cond ((>= i max) nil)
        (t (let ((d (gapi i row))) ; ``d'' for ``delta''
             (cond
              ((= d -1) nil)
              ((= d 0)
               (cons (df-string (coefi i row))
                     (prettyify-row1 (+ 1 i) max row)))
              (t (cons (list d)
                       (cons (df-string (coefi i row))
                             (prettyify-row1 (+ 1 i) max row)))))))))

(defun prettyify-row (row)
  (declare (xargs :stobjs (row)))
  (prettyify-row1 0 (gap-length row) row))

(defun prettyify-ai (i dfst)
  (declare (xargs :stobjs (dfst)))
  (stobj-let ((row (ai i dfst)))
             (val)
             (prettyify-row row)
             val))

(defun prettyify-coef-array1 (i max row)
  (declare (xargs :stobjs (row)))
  (cond ((>= i max) nil)
        (t (cons (df-string (coefi i row))
                 (prettyify-coef-array1 (+ 1 i) max row)))))

(defun prettyify-gap-array1 (i max row)
  (declare (xargs :stobjs (row)))
  (cond ((>= i max) nil)
        (t (cons (gapi i row)
                 (prettyify-gap-array1 (+ 1 i) max row)))))

(defun prettyify-row-arrays (row)
  (declare (xargs :stobjs (row)))
  `((gap ,(prettyify-gap-array1 0 (gap-length row) row))
    (coef ,(prettyify-coef-array1 0 (coef-length row) row))))

(defun prettyify-irow (i dfst)

; Unlike df-solver-v6, this implementation does not have ``irows,'' which were
; of the form (i . row) where row was the SR list with index i in the original
; matrix A.  But we keep the function name here for debugging purposes.

  (declare (xargs :stobjs (dfst)))
  (stobj-let ((row (ai i dfst)))
             (val)
             `(:orig-row-no ,i
                            :current-row ,(prettyify-row row))
             val))

(defun prettyify-successors (i dfst clk)
  (declare (xargs :stobjs (dfst)))
  (cond
   ((zp clk) (er hard? 'prettyify-block "Circular block chain!"))
   ((= (nxi i dfst) -1) nil)
   (t (cons (prettyify-irow (nxi i dfst) dfst)
            (prettyify-successors (nxi i dfst) dfst (- clk 1))))))

(defun prettyify-blocks1 (zcnt max dfst)
  (declare (xargs :stobjs (dfst)
                  :measure (nfix (- (nfix max) (nfix zcnt)))))
  (let ((zcnt (nfix zcnt))
        (max (nfix max)))
    (cond
     ((< zcnt max)
      (let ((index-of-zcnt-pivot (pivoti zcnt dfst)))
        (cond
         ((= index-of-zcnt-pivot -1)
          (cons `(blk ,zcnt nil)
                (prettyify-blocks1 (+ 1 zcnt) max dfst)))
         (t (cons `(blk ,zcnt
                        :pivot ,(prettyify-irow index-of-zcnt-pivot dfst)
                        :successors
                        ,(prettyify-successors index-of-zcnt-pivot dfst max))
                  (prettyify-blocks1 (+ 1 zcnt) max dfst))))))
     (t nil))))

(defun prettyify-blocks (dfst)
  (declare (xargs :stobjs (dfst)))
  (prettyify-blocks1 0 (a-length dfst) dfst))

(defun prettyify-b-prog1 (i dfst)
  (declare (xargs :stobjs (dfst)))
  (cond
   ((>= i (b-prog-free-ptr dfst)) nil)
   (t (cons `((bi ,(b-prog-target-indexi i dfst))
              =+
              ,(df-string (b-prog-pivot-scalari i dfst)) * (bi ,(b-prog-pivot-indexi i dfst)))
            (prettyify-b-prog1 (+ 1 i) dfst)))))

(defun prettyify-b-prog (dfst)
  (declare (xargs :stobjs (dfst)))
  (prettyify-b-prog1 0 dfst))

; -----------------------------------------------------------------
; Little Test 1:

; (defconst *classic-a*
;   '((#d5.0D0 #d1.0D0 #d2.0D0)  ; *A*
;     (#d7.0D0 #d9.0D0 #d-3.0D0)
;     ((1)     #d1.0D0 #d-4.0D0)))

; (defconst *classic-a-dim* 3)

; (defconst *classic-b* '(#d13.0D0 #d16.0D0 #d-10.0D0))

; (initialize-dfst *classic-a* *classic-a-dim* dfst)

; (load-all-rows *classic-a* dfst)

; (blockify *classic-a-dim* dfst)

; (prettyify-blocks dfst)
; ; ==>
; ((BLK 0
;       :PIVOT (:ORIG-ROW-NO 1
;                            :CURRENT-ROW ("7.0" "9.0" "-3.0"))
;       :SUCCESSORS ((:ORIG-ROW-NO 0
;                                  :CURRENT-ROW ("5.0" "1.0" "2.0"))))
;  (BLK 1
;       :PIVOT (:ORIG-ROW-NO 2
;                            :CURRENT-ROW ((1) "1.0" "-4.0"))
;       :SUCCESSORS NIL)
;  (BLK 2 NIL))

; =================================================================
; The Basic Row Operation - Cancelling One Row Against Another

; Add-mult-rows, below, is the basic row operation in which we multiply a
; scalar times one row and add the result to another row.  We store the result
; in a third row.  We denote it

; xrow[i] <- (scalar * row1[i]) + row2[i],
; or just
; xrow <-- (scalar * row1) + row2.

; We pad the resultant row on the left with pad zeros.  That feature is really
; just a generalization allowing the algorithm to collect all the leading zeros
; into a single gap.

; Refresher on the Stobj ROW: there are two arrays, gap and coef.  Think of
; corresponding elements as paired: gap[1] coef[1] gap[2] coef[2] ...  The
; sequence is terminated by some gap[k] = -1, otherwise all gap[i] are naturals
; telling us how long each gap is.  A gap[i] may be 0.  Each coef[i] is a
; non-zero double-float.

; In exact rational arithmetic the product of two non-zeros is non-zero, so if
; scalar is non-zero and row is well-formed and non-empty, it has a non-zero
; coefficient in it, so the scalar product is non-empty.  But fp underflow
; happens, meaning a product of two non-zeros might be zero.  We have to
; program quite defensively to preserve the invariant that no coef[i] is zero.

; We will only be writing to xrow.  And it may need to be resized dynamically.
; So we will use the following two updaters (to gaps and coefs) and always
; maintain the invariant that the gap array is one longer than the coef array,
; regardless of which update provokes the resizing.  We also assume that when
; the index xi is no longer less than the length of its respective array the
; index is equal to that length!  That is, write at the successive indices, not
; jumping further ahead.  When we dynamically grow our arrays we double their
; size, up to dim (the dimension of the matrix).

; About All These Pads: Abstractly, the basic row operation, add-mult-rows,
; takes a scalar and three rows.  It multiplies the first row by the scalar,
; adds it to the second row, and stores it in the third row.  It does this by
; scanning across the rows, skipping gaps and ``imagining'' 0 coefficient in
; the middle of a gap if a non-zero coefficient of the other row falls into
; that gap.

; However, each of the three rows is coded with three formals: a pad, an index,
; and a row stobj (containing coef and gap arrays).  The index records where we
; are in the scan across the arrays in the row.  But what is the purpose of the
; ``pad''?  That's what this comment is about.

; Consider adding two rows (here we forget about the scalar) and that we've
; scanned down the two rows so that the remaining vectors look like this in the
; traditional notation.  (The ``...'' is coded in the two indices which we
; ignore here, treating the arrays as list-like sequences.)

; row1:  ... 0.0  0.0  0.0  1.0  0.0  0.0  2.0  3.0  0.0  0.0
; row2:  ... 0.0  0.0  0.0  0.0  0.0  0.0  4.0  5.0  0.0  7.0

; row1:
;  coef: ... 1.0  2.0  3.0
;  gap : ... 3    2    0    -1

; row2:
;  coef: ... 4.0  5.0  7.0
;  gap : ... 6    0    1    -1

; Note that the current non-zero coefficient (1.0) of row1 falls in the
; current gap of row2.  It would be nice to re-represent row2 this way:

; row2:
;  coef: ... 0.0  4.0  5.0  7.0
;  gap : ... 3    2    0    1    -1

; to make its current gap the same as row1's.  But that would be inefficient.
; Instead, we ``imagine'' the 0.0, add the ``two'' coefficients, advance the
; index in row1, do not advance the index in row2, but make a ``note'' that the
; current gap of row2 is now 2.  (Duh: The current gap of 3 in row2 is matched
; with the gap of 6 in row2 by imagining that gap to be a gap of 3, a
; coefficient of 0.0, and another gap of 2: 6=3+1+2.)

; The ``note'' just mentioned is coded as the ``pad'' associated with each row.
; When the pad is non-nil, it is the value we're supposed to use instead of the
; stored gap.

; row2:
;  coef: ... 4.0  5.0  7.0
;  gap : ... 6    0    1    -1
;  pad :     2

; We we do not wish to override the stored gap, the pad is nil.

; Xpad is always a number -- never nil -- because it is the number of
; consecutive zeros we've seen before writing a non-zero to xrow.

; By the way, I considered using -1 instead of NIL as the signal that there is
; no pad so that I could declare pad to be type INTEGER instead of type (OR
; INTEGER NULL).  (I'm not even sure if (OR INTEGER NULL) is a type that the
; CLTL compilers could exploit!)  But in repeated tests of 100M iterations of
; (foo pad) where foo was

; (defun foo (pad)
;   (declare (type integer pad))
;   (if (= pad -1) (gapi 10 row) pad))

; versus

; (defun foo (pad)
;   (or pad (gapi 10 row)))

; where pad was -1 or 23 in the first case and nil or 23 in the second, the nil
; pad won, with average time per 100M calls 0.040 seconds (for the integer type
; pads) versus 0.029 seconds (for the undeclared type pads).

(defun copy-row1 (scalar pad i row xpad xi xrow dim)

; copy-row1 is used in add-mult-rows when row1 continues but row2 is exhausted
; of non-zeros.  We must copy (scalar * row1) to xrow.  This function writes a
; -1 into xrow when the process is complete.

; WARNING: Termination is an issue.  Ideally, I will need the guard that there
; is a j less than the length of the gap array in row but greater than or equal
; to i such that gapi[j] = -1.  ALTERNATIVELY: I could test i against
; (gap-length row)!  But I'm going to code it to be as fast as I can imagine
; and deal with termination later.  Note that the writes to xrow are
; exceptional in that they dynamically extend the arrays as necessary.

  (declare (type (integer 0 *) xpad i xi)
           (type double-float scalar)
           (xargs :stobjs (row xrow)))
  (let ((d (the (integer -1 *)
                (or pad (gapi i row)))))
    (cond
     ((= d -1)
      (update-xgapi! xi -1 xrow dim))
     (t
      (let ((nz (df* scalar (coefi i row))))
        (if (df= nz (to-df #d0.0D0))
            (copy-row1 scalar nil (+ 1 i) row (+ xpad d 1) xi xrow dim)
            (let* ((xrow (update-xgapi! xi (+ xpad d) xrow dim))
                   (xrow (update-xcoefi! xi nz xrow dim)))
              (copy-row1 scalar nil (+ 1 i) row 0 (+ 1 xi) xrow dim))))))))

(defun copy-row2 (pad i row xpad xi xrow dim)

; Copy-row is used in add-mult-rows when row1 is exhausted but row2 continues.
; This function writes a -1 into xrow when the copy is complete.

; WARNING: Termination/guard is an issue, as above.

  (declare (type (integer 0 *) xpad i xi)
           (xargs :stobjs (row xrow)))
  (let ((d (the (integer -1 *)
                (or pad (gapi i row)))))
    (cond
     ((= d -1)
      (update-xgapi! xi -1 xrow dim))
     (t
      (let* ((xrow (update-xgapi! xi (+ xpad d) xrow dim))
             (xrow (update-xcoefi! xi (coefi i row) xrow dim)))
        (copy-row2 nil (+ 1 i) row 0 (+ 1 xi) xrow dim))))))

(defun add-mult-rows (scalar pad1 i1 row1 pad2 i2 row2
                             xpad xi xrow dim first-flg)

; We compute (scalar x row1) + row2, where both all rows are in padded SR
; format and are positionally aligned.  First-flg is t only for the first
; non-zero coefficients we encounter.  We guarantee the scaled sum there is 0.0
; regardless of any rounding errors introduced by floating point.

  (declare (type (unsigned-byte 30) xpad i1 i2 xi)
           (type double-float scalar)
           (xargs :stobjs (row1 row2 xrow)))
  (let ((d1 (the (integer -1 *)
                 (or pad1 (gapi i1 row1)))))
    (cond
     ((= d1 -1)
      (copy-row2 pad2 i2 row2 xpad xi xrow dim))
     (t
      (let ((d2 (the (integer -1 *)
                     (or pad2 (gapi i2 row2)))))
        (cond
         ((= d2 -1)
          (copy-row1 scalar pad1 i1 row1 xpad xi xrow dim))
         ((= d1 d2)
          (let ((nz (if first-flg
                        (to-df #d0.0D0)
                        (df+ (df* scalar (coefi i1 row1))
                             (coefi i2 row2)))))
            (cond
             ((df= nz #d0.0D0)
              (add-mult-rows scalar
                             nil
                             (+ 1 i1)
                             row1
                             nil
                             (+ 1 i2)
                             row2
                             (+ xpad d1 1)
                             xi
                             xrow
                             dim
                             nil))
             (t
              (let* ((xrow (update-xgapi! xi (+ xpad d1) xrow dim))
                     (xrow (update-xcoefi! xi nz xrow dim)))
                (add-mult-rows scalar
                               nil
                               (+ 1 i1)
                               row1
                               nil
                               (+ 1 i2)
                               row2
                               0
                               (+ 1 xi)
                               xrow
                               dim
                               nil))))))

; We do not have to test first-flg unless d1 = d2 because we know that first-flg
; is only true when the numbers of leading zeros are equal.

         ((< d1 d2)
          (let ((nz (df* scalar (coefi i1 row1)))) ; imagine a 0.0 in row2!
            (cond
             ((df= nz #d0.0D0)
              (add-mult-rows scalar
                             nil
                             (+ 1 i1)
                             row1
                             (- d2 (+ 1 d1))
                             i2
                             row2
                             (+ xpad d1 1)
                             xi
                             xrow
                             dim
                             nil))
             (t
              (let* ((xrow (update-xgapi! xi (+ xpad d1) xrow dim))
                     (xrow (update-xcoefi! xi nz xrow dim)))
                (add-mult-rows scalar
                               nil
                               (+ 1 i1)
                               row1
                               (- d2 (+ 1 d1))
                               i2
                               row2
                               0
                               (+ 1 xi)
                               xrow
                               dim
                               nil))))))
         (t                               ; (< d2 d1)
          (let* ((nz (coefi i2 row2))     ; imagine a 0.0 in row1!
                 (xrow (update-xgapi! xi (+ xpad d2) xrow dim))
                 (xrow (update-xcoefi! xi nz xrow dim)))
            (add-mult-rows scalar
                           (- d1 (+ 1 d2))
                           i1
                           row1
                           nil
                           (+ 1 i2)
                           row2
                           0
                           (+ 1 xi)
                           xrow
                           dim
                           nil)))))))))

; -----------------------------------------------------------------
; Little Test 2:

; This little test works only if you've never run these functions in this image
; before!  For example, if you've resized row to be 100, resizing it to 15, as
; below, has no effect and the stored prettyifications in the equalities below
; will be wrong.  Similarly, if you've run other copy-rows or add-mult-rows you
; might have overwritten the initial values exhibited in the prettyifications.

; (resize-row 15 row)
; (resize-row 15 row1)
; (resize-row 15 row2)
; (resize-row 20 xrow)
; (load-row '((3) #d0.5 #d7.0 (2) #d33.0) 0 row)
; (load-row '(#d0.25 #d0.35) 0 xrow)
; (equal (prettyify-row row)
;        '((3) "0.5" "7.0" (2) "33.0"))
; (equal (prettyify-row-arrays row)
;        '((GAP (3 0 2
;                  -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1))
;          (COEF ("0.5" "7.0" "33.0"
;                 "1.0" "1.0" "1.0" "1.0" "1.0" "1.0"
;                 "1.0" "1.0" "1.0" "1.0" "1.0" "1.0"))))
; (equal (prettyify-row xrow)
;        '("0.25" "0.35"))
; (equal (prettyify-row-arrays xrow)
;        '((GAP (0 0 -1 -1 -1 -1 -1 -1
;                  -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1))
;          (COEF ("0.25" "0.35"
;                 "1.0" "1.0" "1.0" "1.0" "1.0" "1.0"
;                 "1.0" "1.0" "1.0" "1.0" "1.0" "1.0"
;                 "1.0" "1.0" "1.0" "1.0" "1.0" "1.0"))))
; (copy-row1 (to-df #d2.0) nil 0 row 0 2 xrow 15)
; (equal (prettyify-row xrow)
;        '("0.25" "0.35" (3) "1.0" "14.0" (2) "66.0"))
; (load-row '(#d1.0 #d2.0 #d3.0 #d4.0 #d5.0
;                   #d6.0 #d7.0 #d8.0 #d9.0 #d10.0
;                   #d11.0 #d12.0 #d13.0 #d14.0 #d15.0)
;           0 row)
; (load-row '(#d0.15 #d0.25 #d0.35 #d0.45 #d0.55
;                    #d0.65 #d0.75 #d0.85 #d0.95)
;           0 xrow)
; (equal (prettyify-row row)
;        '("1.0" "2.0" "3.0" "4.0" "5.0"
;          "6.0" "7.0" "8.0" "9.0" "10.0"
;          "11.0" "12.0" "13.0" "14.0" "15.0"))
; (equal (prettyify-row xrow)
;        '("0.15" "0.25" "0.35" "0.45" "0.55" "0.65" "0.75" "0.85" "0.95"))
; (xcoef-length xrow)
; (copy-row1 (to-df #d2.0) nil 0 row 0 7 xrow 15)
; (equal (xcoef-length xrow) 40)
; (equal (prettyify-row xrow)
;        '("0.15" "0.25" "0.35" "0.45" "0.55" "0.65"
;          "0.75" "2.0" "4.0" "6.0" "8.0" "10.0"
;          "12.0" "14.0" "16.0" "18.0" "20.0"
;          "22.0" "24.0" "26.0" "28.0" "30.0"))
; (load-row '(#d1.0 (2) #d2.0) 0 row1)                         ; [1 0 0 2]
; (load-row '((1) #d3.0 #d4.0 #d7.0) 0 row2)                   ; [0 3 4 7]
; (add-mult-rows (to-df 1) nil 0 row1 nil 0 row2 0 0 xrow 20 nil) ;----------
; (equal (prettyify-row xrow)                                  ; [1 3 4 9]
;        '("1.0" "3.0" "4.0" "9.0"))
; (load-row '((2) #d0.5 #d7.0 (2) #d33.0) 0 row1)
; (load-row '((2) #d2.0 (3)       #d7.0 (2) #d33.0) 0 row2)
; (add-mult-rows (to-df #d-4.0) nil 0 row1 nil 0 row2 0 0 xrow 20 t)
; ;   [0.0  0.0  0.5  7.0  0.0  0.0  33.0]                 x -4
; ; + [0.0  0.0  2.0  0.0  0.0  0.0   7.0  0.0  0.0  33.0]
; ; -----------------------------------------------------------------
; ;   [0.0  0.0  0.0 -28.0 0.0  0.0 -125.0 0.0  0.0  33.0]
; (equal (prettyify-row xrow)
;        '((3) "-28.0" (2) "-125.0" (2) "33.0"))
; (load-row '((2) #d0.5 (2) #d7.0 (2) #d33.0) 0 row1)
; (load-row '((2) #d2.0 (4) #d7.0 (2) #d33.0) 0 row2)
; (add-mult-rows (to-df #d-4.0) nil 0 row1 nil 0 row2 0 0 xrow 20 t)
; ;   [0.0  0.0  0.5  0.0  0.0  7.0  0.0  0.0  33.0]                 x -4
; ; + [0.0  0.0  2.0  0.0  0.0  0.0  0.0  7.0  0.0  0.0  33.0]
; ; -----------------------------------------------------------------
; ;   [0.0  0.0  0.0  0.0  0.0 -28.0 0.0  7.0 -132.0  0.0 33.0]   ((5) -28.0 (1) 7.0 -132.0 (1) 33.0)
; (equal (prettyify-row xrow)
;        '((5) "-28.0" (1) "7.0" "-132.0" (1) "33.0"))
; -----------------------------------------------------------------

(defun cancel-row (row1             ; row1 = pivot-row
                   row2             ; row2 = target-row
                   xrow             ; result row
                   dim)

; xrow <-- (scalar * row1) + row2, where scalar is computed to cancel the first
; nonzero coefficient in row2.  The result is (mv okp scalar xrow), where okp =
; t if the xrow is non-empty (i.e., has at least one nonzero coefficient.  If
; okp = nil, the row consists of all zeros and this matrix does not allow a
; unique solution.  The scalar returned should be used by the caller to
; add the following instruction to the b-program:

;  b[target-index] <-- (scalar * b[pivot-index]) + b[target-index].

  (declare (xargs :stobjs (row1 row2 xrow)))
  (let* ((scalar (df/ (df- (leading-coef row2))
                      (leading-coef row1)))
         (xrow
          (add-mult-rows scalar
                         nil       ; pad1
                         0         ; i1
                         row1      ; pivot-row
                         nil       ; pad2
                         0         ; i2
                         row2      ; target-row
                         0         ; xpad
                         0         ; xi
                         xrow      ; xrow
                         dim
                         t)))      ; first-flg

; The first result below tests whether xrow is non-empty, i.e., has at least
; one non-zero coefficient: t if non-empty, nil if empty.  If the row is empty,
; the matrix is singular.  The last result is the double-float scalar used.
; This is needed by the caller to add a new instruction to the b-program,
; namely the instruction:

    (mv (not (= (gapi 0 xrow) -1))
        scalar
        xrow)))

; -----------------------------------------------------------------
; Little Test 3

; This test was a way to confirm that I'm using swap-stobjs appropriately.

; (initialize-dfst '((15 10 5)
;                    ( 3  3 2)
;                    (17 18 19))
;                  3
;                  dfst)

; (load-all-rows '((15 10 5)
;                  ( 3  3 2)
;                  (17 18 19))
;                dfst)

; (prettyify-ai 0 dfst) ; ==>  ("15.0" "10.0" "5.0")
; (prettyify-ai 1 dfst) ; ==>  ( "3.0"  "3.0" "2.0")

; ; Note that if you cancel row 1 and row 2 you should get result:
; ;  (-3/15 * (15 10 5))
; ;  (-3  -2  -1)
; ; +( 3   3   2)
; ; -------------
; ;  ( 0   1   1) = ((1) 1 1)

; (defun prettyify-scratch (dfst)
;   (declare (xargs :stobjs (dfst)))
;   (stobj-let ((xrow (scratch dfst)))
;              (val)
;              (prettyify-row xrow)
;              val))

; (prettyify-scratch dfst) ; Output varies according to what else you've done!

; (defun tester (dfst)
;   (declare (xargs :stobjs (dfst)))
;   (stobj-let ((row1 (ai 0 dfst))
;               (row2 (ai 1 dfst))
;               (xrow (scratch dfst)))
;              (okp scalar-string row2 xrow)
;              (mv-let (okp scalar xrow)
;                (cancel-row row1 row2 xrow (dim dfst))
;                (mv-let (row2 xrow)
;                  (swap-stobjs row2 xrow)
;                  (mv okp
;                      (df-string scalar) ; just curious to see scalar
;                      row2
;                      xrow)))
;             (mv okp scalar-string dfst)))

; (prettyify-ai 0 dfst) ; ==>  ("15.0" "10.0" "5.0")
; (prettyify-ai 1 dfst) ; ==>  ( "3.0"  "3.0" "2.0")
; (tester dfst)
; (prettyify-ai 1 dfst) ; ==>  ( (1)    "1.0" "1.0")    ; new (ai 1 dfst)
; (prettyify-scratch dfst) ; ==>  ( "3.0"  "3.0" "2.0") ; the old (ai 1 dfst)
; -----------------------------------------------------------------

(defun cancel-blk1 (pivot-index target-index dim dfst)

; Pivot-index is the index in A of the pivot row of a block, and target-index
; is the index of the first non-pivot row of that block.  We cancel the the
; leading coef of each of the rows in the block, starting at target-index.  As
; they're canceled, remove target-index from the current block and put the
; modified target-index into its proper block (it has more leading zeros now).

; Note: Dim will be used here for two reasons.  First, when update-xgapi! and
; -xcoefi!  resize a row, they use dim as the maximum size the row can grow
; (well, dim+1 for gaps).  Second, when we address termination we will either
; have to prove the nxi chain is acyclic or use dim as a clock to limit it.

  (declare (type (integer 0 *) pivot-index dim)
           (type (integer -1 *) target-index)
           (xargs :stobjs (dfst)))
  (cond
   ((= target-index -1) (mv t dfst))
   (t (mv-let (okp scalar new-zcnt new-coef dfst)
        (stobj-let ((row1 (ai pivot-index dfst))
                    (row2 (ai target-index dfst))
                    (xrow (scratch dfst)))
                   (okp scalar new-zcnt new-coef row2 xrow)
                   (mv-let (okp scalar xrow)
                     (cancel-row row1 row2 xrow dim)
                     (let ((new-zcnt (gapi 0 xrow))
                           (new-coef (coefi 0 xrow)))
                       (mv-let (row2 xrow)
                         (swap-stobjs row2 xrow)
                         (mv okp scalar new-zcnt new-coef row2 xrow))))
                   (mv okp scalar new-zcnt new-coef dfst))
        (if (not okp)
            (mv nil dfst)
            (let* ((successor-index (nxi target-index dfst))
; Remove target-index from where it was, splicing in its successor,
                   (dfst (update-nxi pivot-index
                                     successor-index
                                     dfst))
; and put the target-index where it belongs.
                   (dfst (insert-row-index-into-blk target-index
                                                    new-zcnt
                                                    new-coef
                                                    dfst))
                   (dfst (add-b-prog-instr target-index
                                           scalar
                                           pivot-index
                                           dfst)))
              (cancel-blk1 pivot-index successor-index dim dfst)))))))

(defun cancel-blk (zcnt dim dfst)
  (declare (type (integer 0 *) zcnt)
           (xargs :stobjs (dfst)))
  (let ((pivot-index (pivoti zcnt dfst)))
    (cond
     ((= pivot-index -1) ; empty block! Singular matrix
      (mv nil dfst))
     (t
      (cancel-blk1 pivot-index
                   (nxi pivot-index dfst) ; = target-index
                   dim
                   dfst)))))

(defbadge cancel-blk)

(defun cancel-all (dim dfst)
  (declare (xargs :stobjs (dfst)))
  (loop$ with zcnt of-type (integer 0 *) = 0
         with okp = t
         do
         :measure (- dim zcnt)
         :values (nil dfst)
         (cond
          ((>= zcnt dim)
           (return (mv t dfst)))
          (t (progn
               (mv-setq (okp dfst)
                        (cancel-blk zcnt dim dfst))
               (if (not okp)
                   (return (mv nil dfst))
                   (setq zcnt (+ 1 zcnt))))))))

; =================================================================
; Phase1

(defun phase1 (m dim dfst)

; This function returns (mv okp b-program dfst), where okp is
; t if all went well and nil if a singularity or other problem was discovered,
; dfst is in LU form and b-program is the final program to transform the
; still unknown vector b.

  (declare (type (integer 0 *) dim)
           (type (satisfies true-listp) m) ; SR-dfp-matrixp
           (xargs :stobjs (dfst)))
  (let* ((dfst (initialize-dfst m dim dfst))
         (dfst (load-all-rows m dfst))
         (dfst (blockify dim dfst)))
    (mv-let (okp dfst)
      (cancel-all dim dfst)
      (mv okp dfst))))

; -----------------------------------------------------------------
; Little Test 4:
; (defconst *A* '((#d5.0D0 #d1.0D0 #d2.0D0)
;                 (#d7.0D0 #d9.0D0 #d-3.0D0)
;                 ((1)     #d1.0D0 #d-4.0D0)))
; (defconst *b* '(#d13.0D0 #d16.0D0 #d-10.0D0))
; (phase1 *A* (dim dfst) dfst)
; (prettyify-blocks dfst)

; =================================================================
; Loading b and Running the b-Program

(defun load-gap-into-bi-dfst (i gap dfst)

; We do not anticipate the b array having many zeros and so we represent it in
; the traditional way: one array element per row element (and no gap array).
; So this function writes gap zeros starting at index i of the b array.

; Dfst must be sized correctly before this function can be run, but the b array
; was resized by initialize-dfst in phase1.

  (declare (type (integer 0 *) i gap)
           (xargs :stobjs (dfst)))
  (cond
   ((zp gap) dfst)
   (t (let ((dfst (update-bi i (to-df #d0.0) dfst)))
        (load-gap-into-bi-dfst (+ 1 i) (- gap 1) dfst)))))

(defun load-b-into-dfst (i SR-dfp-b dim dfst)

; The b vector will initially contain a lot of zeros (in the first VWSIM
; example, where b was a vector of length 1036, it had 909 zeros in it.  So it
; is presented to us in SR dfp format.  But the first step in the solve for it,
; after reducing A to LU form, is to transform b by carrying out the
; instructions in the b-program, which will usually transform the b vector into
; one with very few zeros.  So we do not represent b internally as a row (with
; coefficients and gaps).  Instead, we represent it as a traditional array,
; devoting one slot to each element of the vector (including any initial
; zeros).  Here is how we load an SR dfp formatted vector into the b array.

  (declare (type (integer 0 *) i dim)
           (type (satisfies true-listp) sr-dfp-b) ; sr-dfp-vectorp
           (xargs :stobjs (dfst)))
  (cond
   ((endp SR-dfp-b)
    (load-gap-into-bi-dfst i (- dim i) dfst))
   ((consp (car SR-dfp-b))
    (let ((dfst (load-gap-into-bi-dfst i (car (car SR-dfp-b)) dfst)))
      (load-b-into-dfst (+ i (car (car SR-dfp-b)))
                        (cdr SR-dfp-b)
                        dim dfst)))
   (t (let ((dfst (update-bi i (to-df (car SR-dfp-b)) dfst)))
        (load-b-into-dfst (+ i 1) (cdr SR-dfp-b) dim dfst)))))

(defwarrant bi)
(defwarrant update-bi)
(defwarrant b-prog-target-indexi)
(defwarrant b-prog-pivot-scalari)
(defwarrant b-prog-pivot-indexi)

(defun run-b-program (dfst)
  (declare (xargs :stobjs (dfst)
                  :guard (and (<= (b-prog-free-ptr dfst)
                                  (b-prog-target-index-length dfst)))))
  (loop$ with i of-type (integer 0 *) = 0
         with free-ptr of-type (integer 0 *) = (b-prog-free-ptr dfst)
         do
         :guard (and (natp i)
                     (<= i free-ptr)
                     (dfstp dfst))
         :measure (- free-ptr i)
         :values (dfst)
         (cond
          ((>= i free-ptr) (return dfst))
          (t (let* ((target-index (b-prog-target-indexi i dfst))
                    (scalar (b-prog-pivot-scalari i dfst))
                    (pivot-index (b-prog-pivot-indexi i dfst)))
               (progn
                 (setq dfst (update-bi target-index
                                       (df+ (df* scalar (bi pivot-index dfst))
                                            (bi target-index dfst))
                                       dfst))
                 (setq i (+ i 1))))))))

; =================================================================
; Solving

; X will be in the dfst double-float array below (because (because in the
; big-SR-A example, the computed solution has only 6 zeros out of 1036
; entries).

; If there are k variables, numbered x0, x1, ..., xk-1, then when we're solving
; for variable j, we'll have already solved for (xj+1 xj+2 ... xk-1).  We'll
; fetch the row indicated by the block for j.  Suppose block contains the index
; pivot-index.  Then (ai pivot-index dfst), which we call row below, has j
; leading zeros.  Let the leading coefficient of that row be coefj.  That is
; the coefficient for variable xj.  Suppose the remaining coefficients in row,
; after that for xj, are cj+1, cj+2, ... ck-1.  We wish to compute (* cj+1
; xj+1) (* cj+2 xj+2) ... (* ck-1 xk-1)) = sop (``sum of products'').  The
; solution, then, for xj is thus (df/ (df- (bi pivot-index dfst) sop) coefj).

(defbadge coefi)
(defbadge gapi)
(defbadge xi)

(defun sum-of-products (row j dim dfst)

; We compute ``sop'' as described above.  However, row is a sparse row: (coefi
; i row) does not correspond to (xi i dfst)!  So we scan across row using index
; variable i, stepping it by 1 to visit every (non-zero) coefficient, we
; compute the corresponding index, j, in x by accounting for the gaps, and then
; we add the product to the running answer.  Of course, the xi that fall into
; the gaps contribute nothing to the products and sum, which is why we can skip
; them.

  (declare (type (integer 0 *) j)
           (xargs :stobjs (row dfst)))
  (loop$ with i of-type (integer 0 *) = 1 ; start after leading coef
         with j of-type (integer 0 *) = j ; caller pointed to starting place
         with ans of-type double-float = (to-df #d0.0)
         with d of-type (integer -1 *) = (gapi 1 row)
         do
         :measure (- dim j)
         :values (:df)
         (cond
          ((or (>= j dim)
               (= d -1))
           (return ans))
          (t (progn
               (setq j (+ j d))
               (setq ans (df+ ans (df* (coefi i row) (xi j dfst))))
               (setq i (+ 1 i))
               (setq j (+ 1 j))
               (setq d (gapi i row)))))))

(defun solve-for-leading-var (j dim dfst)

; Suppose tail-ll-ptr-x lists ptrs to the solutions for the variables from j+1
; up.  Let bj be the jth (virtual) element of b.  Then we extend tail-ll-ptr-x
; with the solution for xj.

  (declare (type (integer 0 *) j dim) ; 0 <= j < dim
           (xargs :stobjs (dfst)))
  (let ((pivot-index (pivoti j dfst)))
    (mv-let (coefj sop)
      (stobj-let ((row (ai pivot-index dfst)))
                 (coefj sop)
                 (mv (leading-coef row)
                     (sum-of-products row (+ j 1) dim dfst))
                 (mv coefj sop))
      (update-xi j (df/ (df- (bi pivot-index dfst) sop) coefj) dfst))))

(defun solve-for-all-vars (j dim dfst)
  (declare (type (integer -1 *) j)  ; -1 <= j < dim
           (type (integer 0 *) dim)
           (xargs :stobjs (dfst)
                  :measure (if (< (ifix j) -1)
                               0
                               (+ 1 (ifix j)))))
  (cond
   ((not (integerp j)) ; not mbt when guarded
    (prog2$ (er hard? 'solve-all "Dim must be posp!")
            dfst))
   ((< j 0)
    dfst)
   (t (let ((dfst
             (solve-for-leading-var j dim dfst)))
        (solve-for-all-vars (- j 1) dim dfst)))))

(defun phase2 (SR-dfp-b dim dfst)

; We assume phase1 has completed without error.  We also assume both the b and
; x arrays of dfst are properly sized (as they will be by the initialize-dfst
; call in phase1, if size of b is consistent with the the dimensions of the
; given matrix).  Note that this is the first time we've seen SR-dfp-b.  We
; assume it is a well-formed SR dfp vector.

; We load SR-dfp-b into the dfst, run b-program on it so it looks like it would
; have had we done standard Gaussian elimination with an augemented matrix Ab,
; and then solve for all the variables.  We compute the solution vector and
; store it in the double-float array x of dfst.

  (declare (type (integer 0 *) dim)
           (type (satisfies true-listp) sr-dfp-b)  ; sr-dfp-vectorp
           (xargs :stobjs (dfst)))
  (let* ((dfst (load-b-into-dfst 0 SR-dfp-b dim dfst))
         (dfst (run-b-program dfst))
         (dfst (solve-for-all-vars (- dim 1) dim dfst)))
    dfst))

; =================================================================
; Extracting the Answer

; The end user of the solver will presumably access the solution directly from
; the x array of dfst, where he or she will find double-floats.  But we wish to
; be able to print solutions for debugging purposes.  So we provide three ways
; to present :df objects, as rationals (e.g., 1/2), in #d notation (#d0.5), and
; in raw Lisp notation (0.5).

(defun dfify-dfp (e)

; Convert a dfp rational to a #d string.

  (declare (xargs :guard (dfp e)))
  (concatenate 'string "#d" (df-string (to-df e))))

(defun dfp-listp (lst)

; Recognize a list of dfp rationals.

  (cond
   ((atom lst) (eq lst nil))
   (t (and (dfp (car lst))
           (dfp-listp (cdr lst))))))

(defun dfify-dfp-list (lst)

; Convert a list of dfp rationals to a list of #d strings
  (declare (xargs :guard (dfp-listp lst)))
  (cond ((endp lst) nil)
        (t (cons (dfify-dfp (car lst))
                 (dfify-dfp-list (cdr lst))))))

(defun extract-solution (flg j dfst ans)

; If phase1 completes without error, and phase2 has been run, then dfst
; contains the solution.  The elements of the solution are accessed by (xi i
; dfst), for i from 0 to (- dim 1).  Your application can just access the
; solution directly from dfst.  But if you want to view the solution as a list,
; you can ``extract'' it.  However, since we can't put :df objects into lists
; we have to present it differently.  We can show you the solution in one of
; three formats determined by flg:

; flg = nil means return a list of dfp rationals
; flg = t means return a list of strings like "#d1.234E-4"
; flg = cltl means return a list of strings like "1.234E-4"

; Initially, j should be one less than the dimension of the problem, i.e., the
; index of the last variable.  Ans should initially be nil.

  (declare (type (unsigned-byte 30) j)
           (xargs :stobjs (dfst)))
  (cond
   ((zp j)
    (cons (cond
           ((eq flg t)
            (dfify-dfp (from-df (xi 0 dfst))))
           ((eq flg nil)
            (from-df (xi 0 dfst)))
           (t (df-string (xi 0 dfst))))
          ans))
   (t (extract-solution flg
                        (- j 1)
                        dfst
                        (cons (cond
                               ((eq flg t)
                                (dfify-dfp (from-df (xi j dfst))))
                               ((eq flg nil)
                                (from-df (xi j dfst)))
                               (t (df-string (xi j dfst))))
                              ans)))))

; =================================================================
; The Top-Level

; For the moment, I'm defining a convenient interface that takes SR-dfp-A and
; SR-dfp-b does everything except check for well-formedness and extract the
; solution.  However, if you want to avoid the well-formed checks because you
; KNOW your input is well-formed and you want to use the same A with a series
; of different b vectors, you can just run phase1 on A and then repeatedly run
; phase2 on b, reading the solution directly from x after each run of phase2.

(defun check-well-formed-input (SR-dfp-A SR-dfp-b)

; This function either causes an error or returns the dimension dim of this
; problem.

  (cond
   ((or (not (true-listp SR-dfp-A))
        (eq SR-dfp-A nil))
    (prog2$ (er hard? 'solve "Input matrix A must be a non-nil true-list!")
            nil))
   (t (let ((dim (length SR-dfp-A)))
        (cond
         ((not (SR-dfp-matrixp SR-dfp-A))
          (prog2$ (er hard? 'solve
                      "Input matrix A must satisfy SR-dfp-matrixp!")
                  nil))
         ((not (and (SR-dfp-vectorp SR-dfp-b)
                    (<= (sr-dfp-vlen SR-dfp-b) dim)))
          (prog2$ (er hard? 'solve
                      "Input vector b must statisfy SR-dfp-vectorp with ~
                       sr-dfp-vlen no greater than ~x0!"
                      dim)
                  nil))
         (t dim))))))

(defun solver (SR-dfp-A SR-dfp-b dim dfst)

; This is the crux of the solver.  We better know, before calling this
; function, that SR-dfp-A and SR-dfp-b are well-formed.

  (declare (type (satisfies true-listp) SR-dfp-A SR-dfp-b)
; Really they should satisfy sr-dfp-matrixp and sr-dfp-vectorp!
           (type (unsigned-byte 30) dim)
           (xargs :stobjs (dfst)))
  (mv-let (okp dfst)
    (phase1 SR-dfp-A dim dfst)

; Phase1 initializes dfst for A and b (just sizing arrays appropriately without
; loading them), loads SR-dfp-A into dfst, and then puts it into LU form,
; returning dfst.  It returns okp = t if everything worked and okp = nil if the
; matrix is singular or some other error was detected.  Note that we have not
; used SR-dfp-b yet.

; Phase2 loads SR-dfp-b into dfst, runs the b-program that phase1 left in dfst
; to transform b into the vector it would be had we augmented A with b before
; phase1, and then solves for all the variables, leaving the result in the
; double-float array x.

    (if okp
        (let ((dfst (phase2 SR-dfp-b dim dfst)))
          (mv t dfst))
        (mv nil dfst))))

(defun solver! (flg SR-dfp-A SR-dfp-b dfst)

; This is a ``careful'' interface to the solver meant primarily for debugging.
; It checks the well-formedness of the inputs and extracts the solution
; in the format specified by flg:

; flg = nil means return a list of dfp rationals
; flg = t means return a list of strings like "#d1.234E-4"
; flg = cltl means return a list of strings like "1.234E-4" 

; In any case, the solution is left in the double-float array x of dfst.

  (declare (xargs :stobjs (dfst)))
  (let* ((dim (check-well-formed-input SR-dfp-A SR-dfp-b)))

; The check above either causes an error or returns the dim, having checked
; that both SR-dfp-A and SR-dfp-b are well-formed for dimension dim.  Then we do the
; actual work with solver-wrapper.  We return (mv solution dfst) or
; cause an error.

    (mv-let (okp dfst)
      (solver SR-dfp-A SR-dfp-b dim dfst)
      (if okp
          (mv (extract-solution flg (- dim 1) dfst nil)
              dfst)
          (prog2$
           (er hard? 'solver! "Unsolvable!")
           (mv nil dfst))))))

; =================================================================
; Checking the Answers

; The code below checks the computed answer.  If we solved Ax = b and now have
; the original A and b and the alleged solution x, we can multiply the matrix A
; by x and see how close it is to b.  The metric we use is the Euclidean
; distance between the vector Ax and the vector b, divided by the Euclidean
; length of the vector b.

; This code is not part of our solution and is not meant to be run.  So I'm not
; interested in efficiency.  Our input A and b are assumed to be in SR dfp format.
; We convert them to LL format -- expanding the gaps with explicit 0s -- to
; make it easy to do dot-product.  We use exact rational arithmetic in our
; matrix-vector multiplication, but we use df arithmetic to compute Euclidean
; distance.

(defun dot-product (u v)
  (cond
   ((endp u) 0)
   (t (+ (* (car u) (car v))
         (dot-product (cdr u) (cdr v))))))

(defun LL-matrix-vector-mult (A x)

; A is a list-of-list style n x p matrix and x is a vector of length p.  We
; multiply A times x and return a vector of length n.

  (cond ((endp A) nil)
        (t (cons (dot-product (car A) x)
                 (LL-matrix-vector-mult (cdr A) x)))))

(defun sr-to-ll-vector (m row zero)
; Convert an SR row to an LL row of length m.
  (cond ((endp row) (make-list m :initial-element zero))
        ((consp (car row))
         (append (make-list (car (car row)) :initial-element zero)
                 (sr-to-ll-vector (- m (car (car row))) (cdr row) zero)))
        (t (cons (car row)
                 (sr-to-ll-vector (- m 1) (cdr row) zero)))))

(defun SR-to-LL-matrix1 (m rows zero)
  (cond ((endp rows) nil)
        (t (cons (sr-to-ll-vector m (car rows) zero)
                 (SR-to-LL-matrix1 m (cdr rows) zero)))))

(defun SR-to-LL-matrix (matrix zero)
; Convert an SR matrix to an LL matrix.  Zero should be either 0 or 0.0D0
; depending on the type of data in m.
  (SR-to-LL-matrix1 (length matrix) matrix zero))


(defun euclidean-distance (v1 v2)
  (declare (xargs :guard (and (dfp-listp v1)
                              (dfp-listp v2))
                  :verify-guards nil))
  (loop$ with lst1 = v1
         with lst2 = v2
         with ans of-type double-float = (to-df #d0.0)
         do
         :measure (acl2-count lst1)
         :guard (and (dfp-listp lst1)
                     (dfp-listp lst2))
         :values (:df)
         (cond
          ((endp lst1)
           (return (df-sqrt ans)))
          (t (let ((diff (df- (to-df (car lst1))
                              (to-df (car lst2)))))
               (declare (type double-float diff))
               (progn
                 (setq ans (df+ (df* diff diff) ; (df-expt diff 2)
                                ans))
                 (setq lst1 (cdr lst1))
                 (setq lst2 (cdr lst2))))))))

(defun euclidean-length (v2)
  (declare (xargs :guard (dfp-listp v2)
                  :verify-guards nil))
  (loop$ with lst2 = v2
         with ans of-type double-float = (to-df #d0.0)
         do
         :measure (acl2-count lst2)
         :guard (dfp-listp lst2)
         :values (:df)
         (cond
          ((endp lst2)
           (return (df-sqrt ans)))
          (t (let ((e (to-df (car lst2))))
               (declare (type double-float e))
               (progn
                 (setq ans (df+ (df* e e) ; (df-expt diff 2)
                                ans))
                 (setq lst2 (cdr lst2))))))))

(defun Ax=b? (SR-dfp-A SR-dfp-b dfst)

; After running the solver (or solver! or a non-erroneous run of phase1
; followed by phase2) you can call Ax=b? to see the absolute Euclidean distance
; between Ax and b and also the distance relative to the length of the vector
; b.

  (declare (xargs :stobjs (dfst)))
  (let* ((dim (dim dfst))
         (A (SR-to-LL-matrix SR-dfp-A 0))
         (x (extract-solution nil (- dim 1) dfst nil))
         (b (SR-to-LL-vector dim SR-dfp-b 0))
         (edistance (euclidean-distance
                     (LL-matrix-vector-mult A x)
                     b))
         (elength (euclidean-length b)))
    `((absolute-error ,(df-string edistance))
      (relative-error ,(df-string (df/ edistance elength))))))


