//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/MaterialItem.h
//! @brief     Defines class MaterialItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_SAMPLE_MATERIALITEM_H
#define BORNAGAIN_GUI_MODEL_SAMPLE_MATERIALITEM_H

#include "GUI/Model/Descriptor/VectorProperty.h"
#include <QColor>
#include <QObject>
#include <QXmlStreamReader>
#include <heinz/Complex.h>
#include <heinz/Vectors3D.h>
#include <memory>
#include <variant>

class Material;

class MaterialItem : public QObject {
    Q_OBJECT

public:
    MaterialItem();

    //! Creates a complete copy, also the identifier is the same.
    //!
    //! Note that the base QObject is freshly initialized, not copied from \a other. Therefore
    //! object hierarchy, connections, properties, object name or similar things are not copied.
    //! This is of no harm since this class is only derived from QObject to provide data change
    //! signaling.
    MaterialItem(const MaterialItem& other);

    //! Turns material into refractive index material.
    //!
    //! Set refractive index as in 1 - delta + i * beta
    void setRefractiveIndex(double delta, double beta);

    //! Turns material into SLD based material.
    void setScatteringLengthDensity(complex_t sld);

    DoubleProperty& delta();
    const DoubleProperty& delta() const;

    DoubleProperty& beta();
    const DoubleProperty& beta() const;

    DoubleProperty& sldRe();
    const DoubleProperty& sldRe() const;

    DoubleProperty& sldIm();
    const DoubleProperty& sldIm() const;

    VectorProperty& magnetization() { return m_magnetization; }
    const VectorProperty& magnetization() const { return m_magnetization; }
    void setMagnetization(const R3& magnetization);

    /// \return true if refractive index was given, otherwise SLD was given
    bool hasRefractiveIndex() const;

    QString matItemName() const;
    void setMatItemName(const QString& name);

    QString identifier() const;
    void createNewIdentifier();

    QColor color() const;
    void setColor(const QColor& color);

    std::unique_ptr<Material> createMaterial() const;

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    //! Updates content from the other material.
    //!
    //! Does NOT change the identifier.
    //! emits dataChanged, if differences exist.
    void updateFrom(const MaterialItem& other);

private:
    //! Compares all contents. The inactive contents (e.g. SLD in case of refractive) are not taken
    //! into account. Only used by updateFrom.
    bool operator==(const MaterialItem& other) const;

    QString m_name;
    QString m_id;
    QColor m_color;
    VectorProperty m_magnetization;
    bool m_useRefractiveIndex = true;
    DoubleProperty m_delta;
    DoubleProperty m_beta;
    DoubleProperty m_sldRe;
    DoubleProperty m_sldIm;

signals:
    void dataChanged() const;
};

#endif // BORNAGAIN_GUI_MODEL_SAMPLE_MATERIALITEM_H
