//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Tool/ItemDelegateForHTML.cpp
//! @brief     Implements class ItemDelegateForHTML
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Tool/ItemDelegateForHTML.h"
#include <QAbstractItemView>
#include <QAbstractTextDocumentLayout>
#include <QApplication>
#include <QPainter>
#include <QTextDocument>

namespace {

bool hasHtml(const QString& t)
{
    return t.contains("<") && t.contains(">");
}

QSize mySizeHint(const QString& text)
{
    QTextDocument doc;
    doc.setHtml(text);
    doc.setTextWidth(10000 /*options.rect.width()*/);
    QSize size = QSize(doc.idealWidth(), doc.size().height());
    return size;
}

} // namespace


ItemDelegateForHTML::ItemDelegateForHTML(QObject* parent)
    : QStyledItemDelegate(parent)
{
}

void ItemDelegateForHTML::paint(QPainter* painter, const QStyleOptionViewItem& option,
                                const QModelIndex& index) const
{
    QStyleOptionViewItem options = option;
    initStyleOption(&options, index);
    if (!hasHtml(options.text)) {
        QStyledItemDelegate::paint(painter, option, index);
        return;
    }

    painter->save();
    QTextDocument doc;
    doc.setHtml(options.text);

    options.text = "";

    const QWidget* widget = option.widget;
    QStyle* style = widget ? widget->style() : QApplication::style();
    style->drawControl(QStyle::CE_ItemViewItem, &options, painter, widget);

    // shift text right to make icon visible
    QSize iconSize = options.icon.actualSize(options.rect.size());
    painter->translate(options.rect.left() + iconSize.width(), options.rect.top());
    QRect clip(0, 0, options.rect.width() + iconSize.width(), options.rect.height());

    painter->setClipRect(clip);
    QAbstractTextDocumentLayout::PaintContext ctx;

    // set text color (see qcommonstyle.cpp, QCommonStyle::drawControl, case CE_ItemViewItem)
    QPalette::ColorGroup cg =
        options.state & QStyle::State_Enabled ? QPalette::Normal : QPalette::Disabled;
    if (cg == QPalette::Normal && !(options.state & QStyle::State_Active))
        cg = QPalette::Inactive;
    ctx.palette.setColor(QPalette::Text, option.palette.color(cg, QPalette::Text));

    ctx.clip = clip;
    doc.documentLayout()->draw(painter, ctx);
    painter->restore();
}

QSize ItemDelegateForHTML::sizeHint(const QStyleOptionViewItem& option,
                                    const QModelIndex& index) const
{
    QSize s = QStyledItemDelegate::sizeHint(option, index);

    // get size of parent; this is the minimum size
    const int h = QStyledItemDelegate::sizeHint(option, index).height();
    s.setHeight(std::max(s.height(), h));

    QStyleOptionViewItem options = option;
    initStyleOption(&options, index);

    auto s2 = mySizeHint(options.text);
    s.setHeight(std::max(s.height(), s2.height() + 10));
    s.setWidth(s2.width() + h); // +h: icon

    return s;
}

QString ItemDelegateForHTML::anchorAtGlobalPos(QAbstractItemView* view, const QModelIndex& index,
                                               const QPoint& globalPos) const
{
    QString text = index.model()->data(index, Qt::DisplayRole).toString();

    QTextDocument doc;
    doc.setHtml(text);

    QRect r = view->visualRect(index);
    QPoint P = view->viewport()->mapFromGlobal(globalPos);
    QPoint P2 = P - r.topLeft();

    QAbstractTextDocumentLayout::PaintContext ctx;
    return doc.documentLayout()->anchorAt(P2);
}
