#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <sys/stat.h>

#include "canlxx.h"
#include "opensslutil.h"
#include "fileutil.h"

#include "cache.h"

namespace AuthN {
namespace Utils {

  BDB::BDB(const std::string& db_home, const std::string& db_name, 
      AuthN::Context* context) : env_(NULL), db_(NULL), cursor_(NULL), 
      db_home_(db_home), db_name_(db_name), context_(context) {     

    env_ = new DbEnv(0);

    if(!AuthN::Utils::File::dir_exist(db_home)) {
      if (mkdir(db_home.c_str(), 0700) != 0) {
        context_->LogFormat(AuthN::Context::LogError, "Cannot create directory: %s", db_home.c_str());
        return;
      }
    }

    bool is_existing = false;
    std::string db_file = db_home + "/" + db_name;
    if(AuthN::Utils::File::file_exist(db_file)) is_existing = true;

    try {
      // setup internal deadlock detection mechanizm
      env_->set_lk_detect(DB_LOCK_DEFAULT);
      if(is_existing)
        env_->open(db_home.c_str(), DB_INIT_MPOOL | DB_INIT_LOCK | DB_INIT_TXN | DB_THREAD, 0);
      else
        env_->open(db_home.c_str(), DB_CREATE | DB_INIT_MPOOL | DB_INIT_LOCK | DB_INIT_TXN | DB_THREAD, 0);
    } catch (DbException &e) {
      context_->LogFormat(Context::LogError, "Cannot create db env: %s", e.what());
    }

    db_ = new Db(env_, 0);
    DbTxn *tid = NULL;
    try {
#if (DB_VERSION_MAJOR < 4) || (DB_VERSION_MAJOR == 4 && DB_VERSION_MINOR == 0)
      if(is_existing)
        db_->open(db_name.empty() ? NULL : db_name.c_str(), NULL, DB_BTREE, DB_THREAD, 0);
      else 
        db_->open(db_name.empty() ? NULL : db_name.c_str(), NULL, DB_BTREE, DB_CREATE | DB_THREAD, 0);
#else
      env_->txn_begin(NULL, &tid, 0);
      if(is_existing)
        db_->open(tid, db_name.empty() ? NULL : db_name.c_str(), NULL, DB_BTREE, DB_THREAD, 0);
      else
        db_->open(tid, db_name.empty() ? NULL : db_name.c_str(), NULL, DB_BTREE, DB_CREATE | DB_THREAD, 0);
      tid->commit(0);
#endif
    } catch (DbException &e) {
      try {
        tid->abort();
        context_->Log(Context::LogError, "Cannot open database");
      } catch (DbException &e) {
        context_->LogFormat(Context::LogError, "Cannot abort transaction %s", e.what());
      }
      db_ = NULL;
      return;
    }
  }

  BDB::~BDB() {
    if (db_ != NULL) {
      db_->close(0);
      //db_->remove(db_name_.c_str(), NULL, 0);
      delete db_;
      db_ = NULL;
    }
    if (env_ != NULL) {
      env_->close(0);
      //env_->remove(db_home_.c_str(), 0);
      delete env_;
      env_ = NULL;
    }
  }

  bool BDB::put(const std::string& name, const std::string& content) {
    void *k = (void *)name.c_str();
    void *v = (void *)content.c_str();
    Dbt key(k, name.size() + 1);
    Dbt value(v, content.size() + 1);
    DbTxn *tid = NULL;
    while (true) {
      try {
        env_->txn_begin(NULL, &tid, 0);
        db_->put(tid, &key, &value, 0);
        tid->commit(0);
        return true;
#ifdef HAVE_DBDEADLOCKEXCEPTION
      } catch (DbDeadlockException &e) {
        try {
          tid->abort();
          context_->Log(AuthN::Context::LogInfo, "put: deadlock handling: try again");
        } catch (DbException &e) {
          context_->LogFormat(AuthN::Context::LogError, "put: cannot abort transaction: %s", e.what());
          return false;
        }
#endif
      } catch (DbException &e) {
        context_->LogFormat(AuthN::Context::LogError, "put: %s", e.what());
        try {
          tid->abort();
        } catch (DbException &e) {
          context_->LogFormat(AuthN::Context::LogError, "put: cannot abort transaction: %s", e.what());
        }
        return false;
      }
    }
  }

  std::string BDB::get(const std::string& name) {
    void *k = (void *)name.c_str();
    Dbt key(k, name.size() + 1);
    Dbt value;
    value.set_flags(DB_DBT_MALLOC);
    DbTxn *tid = NULL;
    while (true) {
      try {
        env_->txn_begin(NULL, &tid, 0);
        if (db_->get(tid, &key, &value, 0) != DB_NOTFOUND) {
          std::string ret;
          ret.resize(value.get_size());
          memcpy((char*)(ret.c_str()), value.get_data(), value.get_size());
          free(value.get_data());
          tid->commit(0);
          return ret;
        }
        tid->commit(0);
        return "";
#ifdef HAVE_DBDEADLOCKEXCEPTION
      } catch (DbDeadlockException &e) {
        try {
          tid->abort();
          context_->Log(AuthN::Context::LogInfo, "get: deadlock handling, try again");
        } catch (DbException &e) {
          context_->LogFormat(AuthN::Context::LogError, "get: cannot abort transaction: %s", e.what());
          return "";
        }
#endif
      } catch (DbException &e) {
        context_->LogFormat(AuthN::Context::LogError, "get: %s", e.what());
        try {
          tid->abort();
        } catch (DbException &e) {
          context_->LogFormat(AuthN::Context::LogError, "get: cannot abort transaction: %s", e.what());
        }
        return "";
      }
    }
  }

  void BDB::del(const std::string &name) {
    void *k = (void *)name.c_str();
    Dbt key(k, name.size() + 1);
    DbTxn *tid = NULL;
    try {
      env_->txn_begin(NULL, &tid, 0);
      db_->del(tid, &key, 0);
      tid->commit(0);
#ifdef HAVE_DBDEADLOCKEXCEPTION
    } catch (DbDeadlockException &e) {
      try {
        tid->abort();
        context_->Log(Context::LogInfo, "del: deadlock handling, try again");
      } catch (DbException &e) {
        context_->LogFormat(AuthN::Context::LogError, "del: cannot abort transaction: %s", e.what());
        return;
      }
#endif
    } catch (DbException &e) {
      context_->LogFormat(AuthN::Context::LogError, "del: %s", e.what());
      try {
        tid->abort();
      } catch (DbException &e) {
        context_->LogFormat(AuthN::Context::LogError, "del: cannot abort transaction: %s", e.what());
      }
      return;
    }
    return;
  }

  void BDB::del_with_cb(cursor_delete_cb* cb) {
    Dbc* cursor;
    Dbt key;
    Dbt data;
    int result;

    DbTxn *tid = NULL;
    try {
      env_->txn_begin(NULL, &tid, 0);
      // Get a cursor
      db_->cursor(tid, &cursor, 0);

      // Iterator over the database, and retrieve each record
      while((result = cursor->get(&key, &data, DB_NEXT)) == 0) {
        std::string id, value;
        id.resize(key.get_size());
        memcpy((char*)(id.c_str()), key.get_data(), key.get_size());
        value.resize(data.get_size());
        memcpy((char*)(value.c_str()), data.get_data(), data.get_size());
        if(cb(id, value)) result = cursor->del(0);
      }
      cursor->close();

      tid->commit(0);
#ifdef HAVE_DBDEADLOCKEXCEPTION
    } catch (DbDeadlockException &e) {
      try {
        tid->abort();
        context_->Log(Context::LogInfo, "del: deadlock handling, try again");
      } catch (DbException &e) {
        context_->LogFormat(AuthN::Context::LogError, "del: cannot abort transaction: %s", e.what());
        return;
      }
#endif
    } catch (DbException &e) {
      context_->LogFormat(AuthN::Context::LogError, "del: %s", e.what());
      try {
        tid->abort();
      } catch (DbException &e) {
        context_->LogFormat(AuthN::Context::LogError, "del: cannot abort transaction: %s", e.what());
      }
      return;
    }
    return;
  }

  bool BDB::cursor_walk(std::string& id, std::string& value) {
    int ret = true;;
    if(cursor_ == NULL) return false;
    try {
      // Set up our DBTs
      Dbt data;
      Dbt key;
      int n;
      if ((n = cursor_->get(&key, &data, DB_NEXT)) == 0) {
        id.resize(key.get_size());
        memcpy((char*)(id.c_str()), key.get_data(), key.get_size());
        value.resize(data.get_size());
        memcpy((char*)(value.c_str()), data.get_data(), data.get_size());
      }
    } catch(DbException &e) {
      context_->LogFormat(AuthN::Context::LogError, "Error when to retrieve cursor: %s", e.what());
      ret = false;
    } 
    return ret;
  }    

  bool BDB::cursor_del() {
    try {
      if(cursor_ != NULL) cursor_->del(0);
    } catch (DbException &e) {
      context_->LogFormat(AuthN::Context::LogError, "Error when delete the item at cursor: %s", e.what());
      return false;
    }
    return true;
  }

  void BDB::cursor_init() {
    try {
      // Get the cursor
      db_->cursor(NULL, &cursor_, 0);
    } catch (DbException &e) {
      context_->LogFormat(AuthN::Context::LogError, "Error when to get cursor: %s", e.what());
    }
  }

  void BDB::cursor_close() {
    if (cursor_ != NULL) {
      cursor_->close();
      cursor_ = NULL;
    }
  }

  Cache::Cache(AuthN::Context* ctx) : cache_db(NULL), 
      last_cached_check(-1), cache_check_interval(300), context(ctx) {
    std::string db_home = "/tmp";
    cache_db = new BDB(db_home, "", context);
    if(!cache_db) return;
  }

  static void parse_dir_and_file(const std::string& path, std::string& dir, std::string& file) {
    std::string::size_type pos;
    pos = path.rfind("/");
    if(pos!=std::string::npos) {
      dir = path.substr(0, pos);
      file = path.substr(pos+1);
    }
  }

  Cache::Cache(const std::string& cache_path, AuthN::Context* ctx) :
      cache_db(NULL), last_cached_check(-1), cache_check_interval(300), context(ctx) {
    std::string db_home, db_name;
    if(AuthN::Utils::File::dir_exist(cache_path)) db_home = cache_path;
    else if(AuthN::Utils::File::file_exist(cache_path)) {
      parse_dir_and_file(cache_path, db_home, db_name);
    }
    else { 
      //If the path is neither an existing dir, nor 
      //an existing file, the directory of the path shoud be parsed.
      parse_dir_and_file(cache_path, db_home, db_name);
      if(!AuthN::Utils::File::dir_exist(db_home)) return;
    }

    if(!db_home.empty()) {
      cache_db = new BDB(db_home, db_name, context);
      if(!cache_db) return;
    }    
  }

  Cache::Cache(const std::string& db_home, const std::string& db_name, AuthN::Context* ctx) : 
      cache_db(NULL), last_cached_check(-1), cache_check_interval(300), context(ctx) {
    //5 minutes as default cache check interval
    if(!db_home.empty() && AuthN::Utils::File::dir_exist(db_home)) {
      cache_db = new BDB(db_home, db_name, context);
      if(!cache_db) return;
    }
  }

  void Cache::StoreItem(const std::string& id, const std::string& data, long till) {
    if(!cache_db) return;

    // create db item value, the length of value to put into db is 
    // the length of the input data, plus the length of the expiration time
    std::string val;
    int size = data.size() + sizeof(long long);
    val.resize(size);
    memcpy((char*)(val.c_str()), &till, sizeof(long long));    
    memcpy((char*)(val.c_str()) + sizeof(long long), data.c_str(), data.size());
    
    cache_db->put(id, val);

    expiration_check();

/*
    cache_db->cursor_init();
    for(;;) {
      std::string name;
      std::string value;
      cache_db->cursor_walk(name, value);
      if(name.empty()) break; // not find any more
      std::cout<<"id: "<<name<<" || value:"<<std::endl;
    }
    cache_db->cursor_close();
*/
  }

  std::string Cache::RetrieveItem(const std::string& id) {
    std::string data;
    std::string val;
    val = cache_db->get(id);

    if(val.empty()) return std::string();

    int data_size = val.size() - sizeof(long long);
    data.resize(data_size);

    long long till;
    memcpy(&till, val.c_str(), sizeof(long long));
    memcpy((char*)(data.c_str()), val.c_str() + sizeof(long long), data_size);

    // check if the retrieved item it expired
    long long now = AuthN::Utils::Time().GetTime();
    if(till <= now) {
      cache_db->del(id);
      return std::string();
    }
    return data;
  }

  bool Cache::record_del_cb(const std::string& key, const std::string& value) {
    long long now = AuthN::Utils::Time().GetTime();
    long long till;
    bool expired = false;

    if(key.empty()) return false; // not find any more

    if(value.size() <= sizeof(long long) || value.empty()) expired = true;
    else {
      memcpy(&till, value.c_str(), sizeof(long long));
      if(till <= now) expired = true;
    }
    return expired;
  }

  void Cache::expiration_check() {
    long long now = AuthN::Utils::Time().GetTime();

    if((last_cached_check != -1) && (now < last_cached_check + cache_check_interval)) return;
    last_cached_check = now;
    
    cache_db->del_with_cb(record_del_cb);
  }

/*
  void Cache::expiration_check() {
    long long now = AuthN::Utils::Time().GetTime();
    long long till;

    if((last_cached_check != -1) && (now < last_cached_check + cache_check_interval)) return;
    last_cached_check = now;
    cache_db->cursor_init();
    bool expired;
    for(;;) {
      std::string name;
      std::string value;
      expired = false;
      cache_db->cursor_walk(name, value);
      if(name.empty()) break; // not find any more

      if(value.size() <= sizeof(long long) || value.empty()) expired = true;
      else {
        memcpy(&till, value.c_str(), sizeof(long long));
        if(till <= now) expired = true;
      }
      if(expired) {
        cache_db->cursor_del();
      }
    }
    cache_db->cursor_close();

  }
*/

}
}
