{
  Copyright 2008-2018 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{$ifdef read_interface}
  { Base type for cubic environment map textures. }
  TAbstractEnvironmentTextureNode = class(TAbstractSingleTextureNode)
  public
    procedure CreateNode; override;

    {$I auto_generated_node_helpers/x3dnodes_x3denvironmenttexturenode.inc}
  end;

  { Cube environment map texture defined as a six individual 2D texture nodes. }
  TComposedCubeMapTextureNode = class(TAbstractEnvironmentTextureNode)
  strict private
    FAlphaChannelData: TAlphaChannel;
  strict protected
    function AlphaChannelData: TAlphaChannel; override;
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdBack: TSFNode;
    public property FdBack: TSFNode read FFdBack;

    strict private FFdBottom: TSFNode;
    public property FdBottom: TSFNode read FFdBottom;

    strict private FFdFront: TSFNode;
    public property FdFront: TSFNode read FFdFront;

    strict private FFdLeft: TSFNode;
    public property FdLeft: TSFNode read FFdLeft;

    strict private FFdRight: TSFNode;
    public property FdRight: TSFNode read FFdRight;

    strict private FFdTop: TSFNode;
    public property FdTop: TSFNode read FFdTop;

    strict private FFdTextureProperties: TSFNode;
    public property FdTextureProperties: TSFNode read FFdTextureProperties;

    { Make sure all 6 sides are loaded.
      Make sure that all 6 fields for each cube map side are assigned,
      are TAbstractTexture2DNode instance (X3D spec requires it),
      load them, and check are they TCastleImage (we cannot suport
      TGPUCompressedImage when loading cubemaps this way,
      as we have to rotate images from ComposedCubeMapTextureNode
      when passing them to GPU, which is only possible for TCastleImage).

      Also calculate AlphaChannel for the whole cube map.
      Our AlphaChannel method will reflect the state from
      last call of this. }
    function LoadSides: boolean;

    {$I auto_generated_node_helpers/x3dnodes_composedcubemaptexture.inc}
  end;

  { Cube environment map texture generated by rendering the 3D world,
    useful for real-time mirrors. }
  TGeneratedCubeMapTextureNode = class(TAbstractEnvironmentTextureNode)
  strict private
    FGeneratedTextureHandler: TGeneratedTextureHandler;
    function GetUpdate: TTextureUpdate;
    procedure SetUpdate(const Value: TTextureUpdate);
  public
    procedure CreateNode; override;
    destructor Destroy; override;

    class function ClassX3DType: string; override;

    strict private FFdUpdate: TSFTextureUpdate;
    public property FdUpdate: TSFTextureUpdate read FFdUpdate;

    { When to update the texture.
      See the X3D specification of the "update" field of the GeneratedCubeMapTexture.

      You should usually use this property instead of accessing @link(FdUpdate),
      it will change FdUpdate.Value and call FdUpdate.Changed. }
    public property Update: TTextureUpdate read GetUpdate write SetUpdate;

    strict private FFdSize: TSFInt32;
    public property FdSize: TSFInt32 read FFdSize;

    strict private FFdTextureProperties: TSFNode;
    public property FdTextureProperties: TSFNode read FFdTextureProperties;

    { Bias from shape center to the point from which we make GeneratedCubeMapTexture.
      TODO: for now in world space, should be in shape space. }
    strict private FFdBias: TSFVec3f;
    public property FdBias: TSFVec3f read FFdBias;

    property GeneratedTextureHandler: TGeneratedTextureHandler
      read FGeneratedTextureHandler;

    {$I auto_generated_node_helpers/x3dnodes_generatedcubemaptexture.inc}
  end;

  { Cube environment map texture loaded from a single file, like DDS. }
  TImageCubeMapTextureNode = class(TAbstractEnvironmentTextureNode, IAbstractUrlObject)
  strict private
    FAlphaChannelData: TAlphaChannel;
  strict protected
    function AlphaChannelData: TAlphaChannel; override;
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdUrl: TMFString;
    public property FdUrl: TMFString read FFdUrl;

    strict private FFdTextureProperties: TSFNode;
    public property FdTextureProperties: TSFNode read FFdTextureProperties;

    { Load cube environment map from a composite (DDS, KTX...) image.

      In case of problems, will make WritelnWarning.
      This includes situations when url cannot be loaded for whatever reason.
      Also problems when url contains valid Composite image,
      but not describing cube map with all 6 sides.

      If all URLs failed, will return @nil.

      Although the loaded image is not saved here, we do save the AlphaChannel
      type. Our AlphaChannel method will reflect last loaded image. }
    function LoadImage: TCompositeImage;

    {$I auto_generated_node_helpers/x3dnodes_imagecubemaptexture.inc}
  end;

{$endif read_interface}

{$ifdef read_implementation}
procedure TAbstractEnvironmentTextureNode.CreateNode;
begin
  inherited;
end;

procedure TComposedCubeMapTextureNode.CreateNode;
begin
  inherited;

  { TODO: changes to nodes inside will not be properly caught,
    they should result in chTextureRendererProperties on us. }

  FFdBack := TSFNode.Create(Self, true, 'back', [TAbstractX3DTexture2DNode]);
   FdBack.ChangeAlways := chTextureRendererProperties;
  AddField(FFdBack);

  FFdBottom := TSFNode.Create(Self, true, 'bottom', [TAbstractX3DTexture2DNode]);
   FdBottom.ChangeAlways := chTextureRendererProperties;
  AddField(FFdBottom);

  FFdFront := TSFNode.Create(Self, true, 'front', [TAbstractX3DTexture2DNode]);
   FdFront.ChangeAlways := chTextureRendererProperties;
  AddField(FFdFront);

  FFdLeft := TSFNode.Create(Self, true, 'left', [TAbstractX3DTexture2DNode]);
   FdLeft.ChangeAlways := chTextureRendererProperties;
  AddField(FFdLeft);

  FFdRight := TSFNode.Create(Self, true, 'right', [TAbstractX3DTexture2DNode]);
   FdRight.ChangeAlways := chTextureRendererProperties;
  AddField(FFdRight);

  FFdTop := TSFNode.Create(Self, true, 'top', [TAbstractX3DTexture2DNode]);
   FdTop.ChangeAlways := chTextureRendererProperties;
  AddField(FFdTop);

  { X3D spec 3.2 doesn't specify this, but it's natural,
    instantreality also uses this. }
  FFdTextureProperties := TSFNode.Create(Self, false, 'textureProperties', [TTexturePropertiesNode]);
   FdTextureProperties.ChangeAlways := chTextureRendererProperties;
  AddField(FFdTextureProperties);
end;

class function TComposedCubeMapTextureNode.ClassX3DType: string;
begin
  Result := 'ComposedCubeMapTexture';
end;

function TComposedCubeMapTextureNode.LoadSides: boolean;

  { Checks is given side has non-nil valid node class,
    and then if image there can be loaded. }
  function SideLoaded(SideField: TSFNode): boolean;
  var
    SideTex: TAbstractTexture2DNode;
  begin
    Result :=
      (SideField.Value <> nil) and
      (SideField.Value is TAbstractTexture2DNode);
    if Result then
    begin
      SideTex := TAbstractTexture2DNode(SideField.Value);
      Result := SideTex.IsTextureImage;

      if Result and not (SideTex.TextureImage is TCastleImage) then
      begin
        WritelnWarning('VRML/X3D', 'ComposedCubeMapTexture cannot contain images compressed GPU-compression algorithms, as we have to rotate images within, and we cannot do this (fast) with compressed textures');
        Result := false;
      end;
    end;

    { If any slice has full-range alpha, then assume whole texture does. }
    AlphaMaxVar(FAlphaChannelData, SideTex.AlphaChannelFinal);
  end;

begin
  FAlphaChannelData := acNone;
  Result :=
    SideLoaded(FdBack) and
    SideLoaded(FdBottom) and
    SideLoaded(FdFront) and
    SideLoaded(FdLeft) and
    SideLoaded(FdRight) and
    SideLoaded(FdTop);
end;

function TComposedCubeMapTextureNode.AlphaChannelData: TAlphaChannel;
begin
  Result := FAlphaChannelData;
end;

procedure TGeneratedCubeMapTextureNode.CreateNode;
begin
  inherited;

  FFdUpdate := TSFTextureUpdate.Create(Self, true, 'update', upNone);
  AddField(FFdUpdate);
  { X3D specification comment: ["NONE"|"NEXT_FRAME_ONLY"|"ALWAYS"] }

  FFdSize := TSFInt32.Create(Self, false, 'size', 128, { MustBeNonnegative } true);
   FdSize.ChangeAlways := chTextureRendererProperties;
  AddField(FFdSize);
  { X3D specification comment: (0,Inf) }

  FFdTextureProperties := TSFNode.Create(Self, false, 'textureProperties', [TTexturePropertiesNode]);
   FdTextureProperties.ChangeAlways := chTextureRendererProperties;
  AddField(FFdTextureProperties);

  FFdBias := TSFVec3f.Create(Self, true, 'bias', TVector3.Zero);
  AddField(FFdBias);

  FGeneratedTextureHandler := TGeneratedTextureHandler.Create;
  FGeneratedTextureHandler.FUpdate := FdUpdate;
end;

destructor TGeneratedCubeMapTextureNode.Destroy;
begin
  FreeAndNil(FGeneratedTextureHandler);
  inherited;
end;

class function TGeneratedCubeMapTextureNode.ClassX3DType: string;
begin
  Result := 'GeneratedCubeMapTexture';
end;

function TGeneratedCubeMapTextureNode.GetUpdate: TTextureUpdate;
begin
  Result := FdUpdate.Value;
end;

procedure TGeneratedCubeMapTextureNode.SetUpdate(const Value: TTextureUpdate);
begin
  FdUpdate.Send(Value);
end;

procedure TImageCubeMapTextureNode.CreateNode;
begin
  inherited;

  FFdUrl := TMFString.Create(Self, true, 'url', []);
  { The image loaded by LoadImage method isn't saved here.
    So we don't need chTextureImage in case of change,
    instead just let renderer load the texture again. }
   FdUrl.ChangeAlways := chTextureRendererProperties;
  AddField(FFdUrl);
  { X3D specification comment: [URI] }

  FFdTextureProperties := TSFNode.Create(Self, false, 'textureProperties', [TTexturePropertiesNode]);
   FdTextureProperties.ChangeAlways := chTextureRendererProperties;
  AddField(FFdTextureProperties);
end;

class function TImageCubeMapTextureNode.ClassX3DType: string;
begin
  Result := 'ImageCubeMapTexture';
end;

function TImageCubeMapTextureNode.LoadImage: TCompositeImage;
var
  I: Integer;
  FullUrl: string;
begin
  Result := TCompositeImage.Create;
  try
    for I := 0 to FdUrl.Items.Count - 1 do
    begin
      FullUrl := PathFromBaseUrl(FdUrl.Items[I]);

      if not TCompositeImage.MatchesURL(FullUrl) then
      begin
        WritelnWarning('VRML/X3D', Format('Only composite (DDS, KTX...) format is supported for ImageCubeMapTexture node, but URL is "%s"', [FullUrl]));
        Continue;
      end;

      try
        Result.LoadFromFile(FullUrl);
      except
        on E: Exception do
        begin
          Result.Close;
          WritelnWarning('VRML/X3D', Format('Error when loading composite file "%s": %s', [FullUrl, E.Message]));
          Continue;
        end;
      end;

      FAlphaChannelData := Result.Images[0].AlphaChannel;

      if Result.CompositeType <> ctCubeMap then
      begin
        Result.Close;
        WritelnWarning('VRML/X3D', Format('Composite image "%s" given for ImageCubeMapTexture doesn''t describe a cube map texture', [FullUrl]));
        Continue;
      end;

      if Result.CubeMapSides <> AllCubeMapSides then
      begin
        Result.Close;
        WritelnWarning('VRML/X3D', Format('Composite image "%s" given for ImageCubeMapTexture doesn''t contain all cube map sides', [FullUrl]));
        Continue;
      end;

      Exit;
    end;

    { If we got here, then no URL was good. So set Result to @nil. }
    FreeAndNil(Result);
  except FreeAndNil(Result); raise end;
end;

function TImageCubeMapTextureNode.AlphaChannelData: TAlphaChannel;
begin
  Result := FAlphaChannelData;
end;

procedure RegisterCubeMapTexturingNodes;
begin
  NodesManager.RegisterNodeClasses([
    TComposedCubeMapTextureNode,
    TGeneratedCubeMapTextureNode,
    TImageCubeMapTextureNode
  ]);
end;

{$endif read_implementation}
