/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#include <cdio/mmc.h>
#include <cdio/mmc_cmds.h>
#include <cdio/logging.h>
#include <cdio/iso9660.h>

#include "config_cdw_undefine.h"
#include "config_cdw.h"

#include "cdw_cdio.h"
#include "cdw_cdio_drives.h"
#include "cdw_ofs.h"
#include "cdw_drive.h"
#include "cdw_utils.h"
#include "cdw_processwin.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_logging.h"
#include "cdw_sys.h"


/**
  \file cdw_cdio.c

  File defines some wrappers around cdio calls. cdw uses cdio library
  for two main purposes:
  \li first step of recognizing CD/DVD disc
  \li operations on CD discs (further investigation of CD, reading cd content)

  There are also first attempts at reading some dvd meta info, but it's hard
  to tell how far this will go.


  This module implements a stack of functions doing reading/writing of data.
  The stack looks like this:
  cdw_cdio_copy_track()
      cdw_cdio_copy_sectors()
          loop {
              cdw_cdio_read_sectors()
                  cdio_read_audio_sectors() or read()
              write()
	      cdw_cdio_copy_update_processwin()
          }

  cdw_cdio_copy_track() is the top-level function in this stack,
  requiring the minimal amount of information to do its job.
  There is no cdw_cdio_copy_disc(), it is up to upper-level code
  to decide if and how a whole disc should be read.


  Parts of code in this file are heavily influenced by online documentation
  of cdio library: http://www.gnu.org/software/libcdio/libcdio.html

  Some technical details about CD sectors, CD types and other related
  information can be found in various online resources:

  \li http://www-is.informatik.uni-oldenburg.de/~dibo/teaching/mm98/buch/node113.html
  \li http://stian.lunafish.org/sample-cdrom.php
  \li http://www.digital-daydreams.com/tech/show_tech.php?id=42
  \li http://gcwebtw.com/Knowhow/AudioCD/AudioCD.htm
  \li http://club.cdfreaks.com/f52/sub-headers-207630/
  \li http://www.samsungodd.com/eng/Information/ODDTech/ODDTech.asp?FunctionValue=view&no=8&type_no=3
  \li http://fy.chalmers.se/~appro/CD-R.survey.html
  \li http://www.e-articles.info/e/a/title/Compact-Disc-information-and-CD-Drive-Formats/
  \li http://nets.rwth-aachen.de/content/teaching/lectures/sub/mms/mmsSS07/09_CD_4P.pdf
  \li http://www.icdia.co.uk/articles/filesystem.html
*/



/* maximal size of meta information area of CD disc sector;
   sync bytes = 12; header bytes = 4; subheader bytes = 8 */
#define CDW_CDIO_SECTOR_META_SIZE_MAX (12 + 4 + 8)



/* For selecting function that will read content of optical disc.
   Reading an audio CD is a simple case, there is a cdio function
   for this purpose.
   CDs and DVDs with ISO9660 optical file system will be read with
   well known read().
   Reading UDF file systems from DVDs will be probably done by function
   from libudf library - to be investigated. Support for UDF is only
   in planning phase.
 */
enum {
	CDW_CDIO_READ_AUDIO_CD,
	CDW_CDIO_READ_ISO9660,
	CDW_CDIO_READ_UDF
};



/* Index this table with values of cdw_track_format_t type */
struct {
	cdw_track_format_t cdw_track_format;
	const char        *cdw_track_format_label;
	uint16_t           payload_size;
} track_format_data[] = {{ CDW_TRACK_BLACK_BOOK_UNKNOWN,         "unknown",              0 },
			 { CDW_TRACK_RED_BOOK_AUDIO,             "audio",             2352 },
			 { CDW_TRACK_YELLOW_BOOK_MODE1,          "mode 1",            2048 },
			 { CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS, "mode 2 formless",   2336 },
			 { CDW_TRACK_YELLOW_BOOK_MODE2_FORM1,    "mode 2 form 1",     2048 },
			 { CDW_TRACK_YELLOW_BOOK_MODE2_FORM2,    "mode 2 form 2",     2324 },
			 { CDW_TRACK_MIXED,                      "mixed",                0 }, /* FIXME: it rather shouldn't be zero */
			 { CDW_TRACK_DVD,                        "dvd",               2048 }};




static cdw_rv_t cdw_cdio_get_info(cdw_cdio_t *disc, const char *device_fullpath);
static cdw_rv_t cdw_cdio_get_info_disc_type(cdw_cdio_t *disc);
static cdw_rv_t cdw_cdio_get_info_tracks(cdw_cdio_t *disc);
static cdw_rv_t cdw_cdio_get_info_fs(cdw_cdio_t *disc, const char *device_fullpath);
static void     cdw_cdio_get_info_cd(cdw_cdio_t *disc);
static void     cdw_cdio_get_info_dvd(cdw_cdio_t *disc);

static void cdw_cdio_get_iso9660_pvd(cdw_cdio_t *disc);

static void cdw_cdio_init_values(cdw_cdio_t *disc);

static cdw_track_format_t cdw_cdio_get_track_format_from_sector_header(cdw_cdio_t *disc, lsn_t sector);
static const char *cdw_cdio_get_driver_error_label(driver_return_code_t error_num);

static cdw_rv_t cdw_cdio_recognize_track(cdw_cdio_t *disc, track_t t);
static cdw_rv_t cdw_cdio_resolve_cd_track_cdw_format(cdw_cdio_t *disc, track_t t);
static cdw_rv_t cdw_cdio_resolve_sector_size(cdw_cdio_t *disc, track_t t);
static cdw_rv_t cdw_cdio_resolve_sectors_to_read(cdw_cdio_t *disc, track_t t);

static int  cdw_cdio_get_cd_sector_meta_data(cdw_cdio_t *disc, lsn_t sector, unsigned char *buffer);
static void cdw_cdio_copy_update_processwin(lsn_t current_relative_sector, lsn_t n_sectors_in_track, int sector_size);
static int  cdw_cdio_copy_sectors(cdw_cdio_t *disc, track_t t, int output_file, long long int sectors, int read_mode);
static inline void cdw_cdio_copy_debug_print(cdw_cdio_t *disc, track_t t, lsn_t sector);

static inline long int cdw_cdio_read_sectors(cdw_cdio_t *disc, track_t t, unsigned char *buffer, lsn_t sector, uint32_t n_sectors, int read_mode);
static void cdw_cdio_log_handler(cdio_log_level_t level, const char *message);

static void cdw_cdio_disc_debug_print(cdw_cdio_t *disc);




/**
   \brief Initialize cdio module

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18
*/
void cdw_cdio_init(void)
{
	/* set to CDIO_LOG_ERROR in release builds! */
	cdio_loglevel_default = CDIO_LOG_ERROR;
	cdio_log_set_handler(cdw_cdio_log_handler);

	cdw_assert (CDIO_FS_AUDIO != 0, "ERROR: you should change \"init\" value of fs type\n");
#ifndef NDEBUG
	/* this is just a test call */
	cdw_sys_check_file_system_support(CDIO_FS_ISO_9660);
#endif
	return;
}





/**
   \brief Clean up cdio module

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18
*/
void cdw_cdio_clean(void)
{
	return;
}





/**
   \brief Create new cdio data structure

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Function allocates new cdio data structure, and initializes all its
   fields to proper initial value.
   The function also call cdw_ofs_new() to create and initialize
   cdw_cdio_t->ofs field.

   State of correctly created cdio data structure can be considered
   as "empty disc". The cdio data structure needs to be filled by calling
   cdw_cdio_get(). The data structure should be destroyed/deallocated with
   cdw_cdio_delete().

   \return allocated and initialized data structure on success
   \return NULL on malloc() or cdw_ofs_new() failure
*/
cdw_cdio_t *cdw_cdio_new(void)
{
	cdw_cdio_t *disc = (cdw_cdio_t *) malloc(sizeof (cdw_cdio_t));
	if (!disc) {
		cdw_vdm ("ERROR: malloc()\n");
		return (cdw_cdio_t *) NULL;
	}
	cdw_cdio_init_values(disc);

	disc->ofs = cdw_ofs_new();
	if (!disc->ofs) {
		cdw_vdm ("ERROR: cdw_ofs_new()\n");
		cdw_cdio_delete(&disc);
	}

	return disc;
}





/**
   \brief Get disc meta-information

   \date Function's top-level comment reviewed on 2012-01-19
   \date Function's body reviewed on 2012-01-19

   Function collects information about an optical disc using cdio function
   calls. Information is stored in \p disc.
   This includes calls to cdio functions that collect optical file system
   information, which is stored in disc->ofs.

   Don't call the function when there is no drive, i.e. when \p device_fullpath
   is NULL or empty.

   \p disc needs to be allocated wit cdw_disc_new().

   \param disc - disc data structure to be filled with information
   \param device_fullpath - full path to disc device file

   \return CDW_ERROR on errors
   \return CDW_NO if disc turns out to be blank, or is of not supported type
   \return CDW_OK if disc is not blank, is supported, and function managed to get meta information
*/
cdw_rv_t cdw_cdio_get(cdw_cdio_t *disc, const char *device_fullpath)
{
	cdw_assert (disc, "ERROR: discs is NULL\n");
	cdw_assert (device_fullpath, "ERROR: fullpath is NULL\n");

	if (!device_fullpath) {
		cdw_vdm ("ERROR: no drive\n");
		return CDW_ERROR;
	}

	disc->p_cdio = cdio_open(device_fullpath, DRIVER_LINUX);
	if (!disc->p_cdio) {
		cdw_vdm ("WARNING: failed to open cdio disc using DRIVER_LINUX, using DRIVER_UNKNOWN\n");

		disc->p_cdio = cdio_open(device_fullpath, DRIVER_UNKNOWN);
		if (!disc->p_cdio) {
			cdw_vdm ("ERROR: failed to open cdio disc using DRIVER_UNKNOWN\n");
			return CDW_ERROR;
		}
	}

	/* these were the basics, now let's get disc meta information */

	cdw_rv_t crv = cdw_cdio_get_info(disc, device_fullpath);
	if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: failed to get disc meta info\n");
	}

	/* mark disc as open to avoid opening it for a second time */
	disc->open = true;

	cdw_cdio_disc_debug_print(disc);

	return CDW_NO;
}





/**
   \brief Deallocate all resources used by cdio data structure

   \date Function's top-level comment reviewed on 2012-01-19
   \date Function's body reviewed on 2012-01-19

   Deallocate and free all resources used by cdw_cdio_t data structure,
   close what needs to be closed, deallocate the data structure itself.
   Assign NULL to the data structure.

   \param disc - pointer to cdio data structure to be deleted
*/
void cdw_cdio_delete(cdw_cdio_t **disc)
{
	cdw_assert (disc, "ERROR: NULL pointer to disc\n");

	if (!(*disc)) {
		cdw_vdm ("WARNING: NULL disc\n");
		return;
	}

	if ((*disc)->ofs) {
		cdw_ofs_delete(&(*disc)->ofs);
	}
	if ((*disc)->p_cdio) {
		cdio_destroy((*disc)->p_cdio);
		(*disc)->p_cdio = (CdIo_t *) NULL;
	}

	free(*disc);
	*disc = (cdw_cdio_t *) NULL;

	return;
}





/**
   \brief Top level function for collecting cdio disc information

   \date Function's top-level comment reviewed on 2012-01-19
   \date Function's body reviewed on 2012-01-19

   This function calls few other functions to collect all possible data
   about cdio disc. Collected information is stored in provided \p disc.

   \param disc - data structure to be filled with information
   \param device_fullpath - full path to drive device

   \return CDW_ERROR on errors
   \return CDW_NO if disc turns out to be blank, or function can't get meta information
   \return CDW_OK if disc is not blank, and function managed to get meta information
*/
cdw_rv_t cdw_cdio_get_info(cdw_cdio_t *disc, const char *device_fullpath)
{
	cdw_rv_t crv = cdw_cdio_get_info_disc_type(disc);
	if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: failed to get basic disc info\n");
		return CDW_ERROR;
	}

	crv = cdw_cdio_get_info_tracks(disc);
	if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: failed to get disc tracks info\n");
		return CDW_ERROR;
	}

	if (disc->blank) {
		return CDW_NO;
	}

	/* get some more, more disc-specific data; even if a disc
	   is blank, get its type (CD-RW, or DVD+R etc.) */
	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		cdw_cdio_get_info_cd(disc);
	} else {
		cdw_cdio_get_info_dvd(disc);
	}

	cdw_cdio_get_iso9660_pvd(disc);

	crv = cdw_cdio_get_info_fs(disc, device_fullpath);

	return crv;
}





void cdw_cdio_get_iso9660_pvd(cdw_cdio_t *disc)
{
	iso9660_pvd_t pvd;
	bool r = iso9660_fs_read_pvd(disc->p_cdio, &pvd);

	if (!r) {
		cdw_vdm ("ERROR: failed to read ISO9660 PVD\n");
		return;
	}

	int block_size = iso9660_get_pvd_block_size(&pvd);
	int space_size = iso9660_get_pvd_space_size(&pvd);

	cdw_vdm ("INFO: PVD        space size = %d (%.2f MB)\n", space_size, space_size * block_size / (1024 * 1024.0));
	cdw_vdm ("INFO: PVD        block size = %d\n", block_size);
	cdw_vdm ("INFO: PVD              type = %d\n", pvd.type);

	return;
}





/**
   \brief Get disc type (CD/DVD)

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Using cdio calls get type of a disc: CD or DVD (this is called
   "simple type"). Store the information in disc->simple_type.
   The function also sets disc->mode.

   \param disc - disc to investigate, and place to store data

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_cdio_get_info_disc_type(cdw_cdio_t *disc)
{
	cdw_assert (disc->p_cdio, "ERROR: trying to get meta info from NULL disc\n");

	disc->mode = cdio_get_discmode(disc->p_cdio);

	if (cdio_is_discmode_cdrom(disc->mode)) {
		disc->simple_type = CDW_DISC_SIMPLE_TYPE_CD;
		strcpy(disc->simple_type_label, "CD");
	} else if (cdio_is_discmode_dvd(disc->mode)
		/* somehow cdio does not classify CDIO_DISC_MODE_DVD_OTHER as dvd */
		|| disc->mode == CDIO_DISC_MODE_DVD_OTHER) {

		disc->simple_type = CDW_DISC_SIMPLE_TYPE_DVD;
		strcpy(disc->simple_type_label, "DVD");
	} else {
		disc->simple_type = CDW_DISC_SIMPLE_TYPE_UNKNOWN;
		/* we won't work with unknown disc types at all */
		cdw_vdm ("ERROR: failed to get disc simple type, setting disc simple type to UNKNOWN\n");
		strcpy(disc->simple_type_label, "UNKNOWN");
		return CDW_ERROR;
	}

	cdw_vdm ("INFO: disc simple type is \"%s\"\n", disc->simple_type_label);

	return CDW_OK;
}





/**
   \brief Get information about disc tracks

   \date Function's top-level comment reviewed on 2012-01-19
   \date Function's body reviewed on 2012-01-19

   Check if a disc is blank. Get number of first and last track
   on a \p disc, number of tracks, and type of every track.
   Count tracks that you can't recognize. Save all this information
   in \p disc.

   \param disc - representation of disc to investigate, place to store data

   \return CDW_ERROR on errors, or if all tracks are unknown
   \return CDW_CANCEL on success (but there were some unknown tracks)
   \return CDW_OK on success (and there were no unknown tracks)
*/
cdw_rv_t cdw_cdio_get_info_tracks(cdw_cdio_t *disc)
{
	disc->first_track = cdio_get_first_track_num(disc->p_cdio);
	disc->last_track = cdio_get_last_track_num(disc->p_cdio);
	disc->n_tracks = (track_t) (disc->last_track - disc->first_track + 1);

	if (disc->first_track == CDIO_INVALID_TRACK) {
		/* this is empirical result: disc is blank, don't try to
		   recognize tracks, because this will cause ioctl() errors; */
		disc->blank = true;

		disc->first_track = CDIO_INVALID_TRACK;
		disc->last_track = CDIO_INVALID_TRACK;
		disc->n_tracks = 0;
	} else {
		disc->unknown_tracks = 0;
		/* inspect every track on disc */
		for (track_t t = disc->first_track; t <= disc->last_track; t++) {
			cdw_rv_t crv = cdw_cdio_recognize_track(disc, t);
			if (crv == CDW_ERROR) {
				disc->unknown_tracks++;
			}
		}

		cdio_fs_t fs = disc->tracks[disc->first_track].fs.cdio_fs;
		if (fs == (cdio_fs_t) CDIO_FS_UNKNOWN || fs == 0) { /* this may trigger compiler warning */
			disc->blank = true;
		} else {
			disc->blank = false;
		}
	}

	if (disc->unknown_tracks == disc->n_tracks) {
		cdw_vdm ("WARNING: all tracks are unknown\n");
		return CDW_ERROR;

	} else if (disc->unknown_tracks > 0
		   && disc->unknown_tracks < disc->n_tracks) {
		cdw_vdm ("WARNING: some tracks are unknown\n");
		return CDW_CANCEL;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Get additional information about a CD disc

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   Get some specific information about CD disc.
   Currently the function is very limited, it only makes sure that correct
   value is assigned to disc->mode under some very specific circumstances.

   \param disc - cdio disc data structure
*/
void cdw_cdio_get_info_cd(cdw_cdio_t *disc)
{
	if (disc->mode == CDIO_DISC_MODE_CD_DATA) {
		/* libcdio 0.78 calls mmc command to get disc type, but the
		   call returns CD_DATA for mixed mode CD; let's do
		   some additional checks */
		cdw_rv_t crv = get_cds_mixed_with_ioctl();
		if (crv == CDW_OK) {
			/* Linux ioctl() says that this is MIXED mode cd,
			   so let's stick with what ioctl() says */
			disc->mode = CDIO_DISC_MODE_CD_MIXED;
		}
	}
	return;
}





/**
   \brief Get additional information about a DVD disc

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   Get some specific information about DVD disc using mmc function calls.

   Currently the function is very limited, it only checks number of layers
   of a DVD disc. The check is performed if HAS_MMC_GET_DVD_STRUCT_PHYSICAL
   is defined.

   \param disc - cdio disc data structure
*/
void cdw_cdio_get_info_dvd(__attribute__((unused)) cdw_cdio_t *disc)
{
	/* FIXME: where it is defined? is this #ifdef valid anymore */
#ifdef HAS_MMC_GET_DVD_STRUCT_PHYSICAL
	/* this info is not too useful at this moment, but it's a good idea
	   to play with new functions from libcdio */
	/*
	  typedef struct cdio_dvd_physical {
	          uint8_t type;
		  uint8_t layer_num;
		  cdio_dvd_layer_t layer[CDIO_DVD_MAX_LAYERS];
	  } cdio_dvd_physical_t;

	  typedef struct cdio_dvd_layer {
	          uint8_t book_version  : 4;
		  uint8_t book_type     : 4;
		  uint8_t min_rate      : 4;
		  uint8_t disc_size     : 4;
		  uint8_t layer_type    : 4;
		  uint8_t track_path    : 1;
		  uint8_t nlayers       : 2;
		  uint8_t track_density : 4;
		  uint8_t linear_density: 4;
		  uint8_t bca           : 1;
		  uint32_t start_sector;
		  uint32_t end_sector;
		  uint32_t end_sector_l0;
	  } cdio_dvd_layer_t;
	*/

	discmode_t m;
	cdio_dvd_struct_t s;
	s.physical.type = CDIO_DVD_STRUCT_PHYSICAL;
	s.physical.layer_num = 0;

	m = mmc_get_dvd_struct_physical(disc->p_cdio, &s);
	if (-m == EINVAL) {
		cdw_vdm ("dvd info from mmc call: following info is INVALID\n");
	}
	cdw_vdm ("dvd info from mmc call: nlayers = %d\n",
		 s.physical.layer[0].nlayers);
	cdw_vdm ("dvd info from mmc call: layer%d: book type = %d, book version = %d, \n",
		 s.physical.layer_num, s.physical.layer[0].book_type, s.physical.layer[0].book_version);
	cdw_vdm ("dvd info from mmc call: layer%d: disc size = %d, layer type = %d, \n",
		 s.physical.layer_num, s.physical.layer[0].disc_size, s.physical.layer[0].layer_type);
	cdw_vdm ("dvd info from mmc call: layer%d: start sector = %d, end sector = %d, \n",
		 s.physical.layer_num, s.physical.layer[0].start_sector, s.physical.layer[0].end_sector);

	if (s.physical.layer[0].nlayers == 1) {/* nlayers == 0 -> 1 layer, nlayers == 1 -> 2 layers? */

		s.physical.layer_num = 1;
		cdw_vdm ("dvd info from mmc call: layer%d: book type = %d, book version = %d, \n",
			 s.physical.layer_num, s.physical.layer[0].book_type, s.physical.layer[0].book_version);
		cdw_vdm ("dvd info from mmc call: layer%d: disc size = %d, layer type = %d, \n",
			 s.physical.layer_num, s.physical.layer[0].disc_size, s.physical.layer[0].layer_type);
		cdw_vdm ("dvd info from mmc call: layer%d: start sector = %d, end sector = %d, \n",
			 s.physical.layer_num, s.physical.layer[0].start_sector, s.physical.layer[0].end_sector);
	}

	if (s.physical.layer[0].nlayers == 0) {/* nlayers == 0 -> 1 layer, nlayers == 1 -> 2 layers? */
		disc->dvd_layers = 1;
	} else if (s.physical.layer[0].nlayers == 1) {/* nlayers == 0 -> 1 layer, nlayers == 1 -> 2 layers? */
		disc->dvd_layers = 2;
	} else {
		;
	}

#endif

	return;
}





/**
   \brief Get information about optical file system

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Get information about optical file system stored in location
   pointed to by \p device_fullpath. Store the information in disc->ofs,
   which has to be allocated earlier.

   Function uses type of optical file system on first track as a file
   system on whole track.

   \param disc - cdio disc data structure; function will store information in disc->ofs
   \param device_fullpath - full path to optical file system on a disc

   \return CDW_NO if optical file system on disc is not supported by cdw_ofs/cdw_cdio
   \return CDW_OK if optical file system on disc is supported and full info has been collected successfully
   \return CDW_ERROR if optical file system on disc is supported, but function failed to collect info successfully
*/
cdw_rv_t cdw_cdio_get_info_fs(cdw_cdio_t *disc, const char *device_fullpath)
{
	int type = disc->tracks[disc->first_track].fs.cdio_fs;
	cdw_assert (type > 0 && type <= CDIO_FS_ISO_UDF,
		    "ERROR: disc file system type out of range: %d\n", type);
	cdw_rv_t crv = cdw_ofs_get(disc->ofs, device_fullpath, type);
	if (crv == CDW_OK) {
		for (track_t t = disc->first_track; t <= disc->last_track; t++) {
			disc->ofs->n_sectors += disc->tracks[t].n_sectors_to_read;
		}
		return CDW_OK;
	} else if (crv == CDW_NO) { /* unsupported file system */
		cdw_assert (disc->ofs, "ERROR: ofs is NULL for unsupported file system\n");
		cdw_vdm ("WARNING: unsupported file system \"%s\"\n", disc->ofs->type_label);
		return CDW_NO;
	} else {
		cdw_vdm ("ERROR: cdw_ofs_get()\n");
		return CDW_ERROR;
	}
}





/**
   \brief Initialize all fields in cdio data structure

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Function assigns initial values to all fields of cdw_cdio_t data structure.
   \p disc needs to be allocated with cdw_cdio_new().

   \param disc - data structure to initialize
*/
void cdw_cdio_init_values(cdw_cdio_t *disc)
{
	disc->p_cdio = (CdIo_t *) NULL;

	disc->mode = CDIO_DISC_MODE_NO_INFO;

	disc->simple_type = CDW_DISC_SIMPLE_TYPE_UNKNOWN;
	disc->simple_type_label[0] = '\0';

	disc->open = false;
	disc->blank = true;

	disc->first_track = CDIO_INVALID_TRACK;
	disc->last_track = CDIO_INVALID_TRACK;
	disc->n_tracks = 0;
	disc->unknown_tracks = 0;

	for (int i = 0; i < 99; i++) {
		disc->tracks[i].cdio_track_format = TRACK_FORMAT_ERROR;
		disc->tracks[i].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;

		disc->tracks[i].first_sector = CDIO_INVALID_LSN;
		disc->tracks[i].last_sector = CDIO_INVALID_LSN;
		disc->tracks[i].n_sectors = 0;
		disc->tracks[i].n_sectors_to_read = 0;

		disc->tracks[i].sector_size = 0;

		disc->tracks[i].fs.cdio_fs = 0;
		disc->tracks[i].fs.size = 0;
	}

	disc->ofs = (cdw_ofs_t *) NULL;

	return;
}





/**
   \brief Collect all available information about given track

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Function tries to collect all information about a track:
   \li start and end sector, number of sectors on track
   \li track formats: cdio track format and cdw track format
   \li size of sectors on the track
   \li number of sectors to read on given sector (I think that this value
       might be different that total number of sectors on track

   The function will fail if optical drive does not support MMC commands
   (if mmc_read_cd() fails).

   \param disc - disc to investigate
   \param t - number of track to check

   \return CDW_OK if track is fully recognized
   \return CDW_ERROR if track is unknown or unsupported
*/
cdw_rv_t cdw_cdio_recognize_track(cdw_cdio_t *disc, track_t t)
{
	cdw_assert (disc->p_cdio, "ERROR: trying to get info from NULL disc\n");
	/* if disc type is unknown then we won't be able to get number
	   of tracks, and this function shouldn't be called at all */
	cdw_assert (disc->simple_type != CDW_DISC_SIMPLE_TYPE_UNKNOWN,
		    "ERROR: called the function for unknown disc type\n")

	/* 1: get range of sectors in given track, these values will be used
	   to check track format, and may be used later when reading sectors */
	disc->tracks[t].first_sector = cdio_get_track_lsn(disc->p_cdio, t);
	disc->tracks[t].last_sector = cdio_get_track_last_lsn(disc->p_cdio, t);

	disc->tracks[t].n_sectors = disc->tracks[t].last_sector - disc->tracks[t].first_sector + 1;

	if (disc->tracks[t].first_sector == CDIO_INVALID_LSN
	    || disc->tracks[t].last_sector == CDIO_INVALID_LSN) {

		cdw_vdm ("ERROR: failed to fetch correct first_sector (%d) or last_sector (%d) of track #%d\n",
			 disc->tracks[t].first_sector, disc->tracks[t].last_sector, (int) t);
		return CDW_ERROR;
	}

	cdw_rv_t crv = CDW_OK;
	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		/* 2: get cdio format of track - for CD discs only! */
		disc->tracks[t].cdio_track_format = cdio_get_track_format(disc->p_cdio, t);
		if (disc->tracks[t].cdio_track_format == TRACK_FORMAT_ERROR) {
			cdw_vdm ("ERROR: failed to resolve cdio track format for track #%d\n", t);
			return CDW_ERROR;
		}

		/* 3: get cdw format of track (disc->tracks[t].cdw_format) */
		crv = cdw_cdio_resolve_cd_track_cdw_format(disc, t);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to resolve cdw track format for track #%d\n", t);
			return CDW_ERROR;
		}
	} else if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		disc->tracks[t].cdw_track_format = CDW_TRACK_DVD;
	} else { /* unknown disc type */
		disc->tracks[t].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
		cdw_vdm ("WARNING: track #%d has unknown format\n", t);
		return CDW_ERROR;
	}

	/* TODO: it looks like cdio_iso_analysis_t may contain some
	   interesting information; make use of it some day */
	cdio_iso_analysis_t iso_analysis;
	memset(&iso_analysis, 0, sizeof(cdio_iso_analysis_t));
	cdio_fs_anal_t a = cdio_guess_cd_type(disc->p_cdio, disc->tracks[t].first_sector, t, &iso_analysis);
	disc->tracks[t].fs.cdio_fs = CDIO_FSTYPE(a);

	if (cdw_ofs_is_iso(disc->tracks[t].fs.cdio_fs)) {
		disc->tracks[t].fs.size = iso_analysis.isofs_size;
	}

	/* 4: using cdio format and cdw format of track, set proper
	   value of disc->tracks[t].sector_size */
	crv = cdw_cdio_resolve_sector_size(disc, t);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to resolve sector size for track #%d\n", t);
		return CDW_ERROR;
	}

	/* 5: try to calculate value of disc->tracks[t].n_sectors_to_read,
	   using different methods, depending on track format */
	crv = cdw_cdio_resolve_sectors_to_read(disc, t);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to calculate number of sectors to read for track #%d\n", t);
		if (t == 1) {
			return CDW_ERROR;
		} else {
			/* cdw can't read data tracks #2 and following, so
			   there is no point in raising an error for these
			   tracks */
		}
	}

	return CDW_OK;
}





/**
   \brief Check cdw format of given track, save it in given disc data struct

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Function inspects given track to assign correct value to
   disc->tracks[t].cdw_format.

   If disc->tracks[t].cdio_track_format indicates Audio CD track, then this function
   simply assigns CDW_RED_BOOK_AUDIO to disc->tracks[t].cdw_format. In other
   case the function calls cdw_cdio_get_track_format_from_sector_header() to
   check cdw format of given track.

   If mmc call for given track fails or yields unknown results, the function
   returns CDW_ERROR.
   CDW_ERROR is also returned when function can't recognize track format,
   and assigns CDW_TRACK_BLACK_BOOK_UNKNOWN as track format.

   \param disc - disc data structure describing a disc
   \param t - number of track that you want to check

   \return CDW_OK if function sets cdw_format correctly
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_cdio_resolve_cd_track_cdw_format(cdw_cdio_t *disc, track_t t)
{
	if (disc->tracks[t].cdio_track_format == TRACK_FORMAT_AUDIO) {
		/* cdw_cdio_get_track_format_from_sector_header() should not
		   be called for audio tracks, since audio track has no
		   header; we resolve cdw format manually */
		disc->tracks[t].cdw_track_format = CDW_TRACK_RED_BOOK_AUDIO;
	} else {
		/* 160 goes beyond possible 2 second silence gap (150) at
		   the beginning of track */
		lsn_t sector = disc->tracks[t].first_sector + 160;
		if (sector > disc->tracks[t].last_sector) { /* I'm not sure if this is possible */
			sector = disc->tracks[t].last_sector - 1;
		}

		disc->tracks[t].cdw_track_format = cdw_cdio_get_track_format_from_sector_header(disc, sector);
	}

	if (disc->tracks[t].cdw_track_format > CDW_TRACK_DVD) {
		disc->tracks[t].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
		cdw_assert (0, "ERROR: track format '%d' for track #%d is out of bounds\n",
			    disc->tracks[t].cdw_track_format, t);
	}

	if (disc->tracks[t].cdw_track_format == CDW_TRACK_BLACK_BOOK_UNKNOWN) {
		cdw_vdm ("ERROR: failed to get cdw format for track #%d (is now CDW_TRACK_BLACK_BOOK_UNKNOWN)\n", t);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Use MMC command to obtain cdw format of sector (and its parent track)

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Read full sector (all 2352 bytes) using mmc command and extract track
   mode data from its header (and subheader if one exists).

   Don't call this function for sectors from CD Audio track.

   \param disc - disc data structure describing a disc
   \param sector - sector number of sector from track that we want to check

   \return CDW_BLACK_BOOK_UNKNOWN if function cannot get track format (perhaps mmc call failed)
   \return correct value of type cdw_track_format_t otherwise
*/
cdw_track_format_t cdw_cdio_get_track_format_from_sector_header(cdw_cdio_t *disc, lsn_t sector)
{
	unsigned char buffer[CDW_CDIO_SECTOR_META_SIZE_MAX]; /* 12 + 4 + 8 */
	memset(buffer, 0x00, CDW_CDIO_SECTOR_META_SIZE_MAX);

	int rv = cdw_cdio_get_cd_sector_meta_data(disc, sector, buffer);
	if (rv <= 0) {
		return CDW_TRACK_BLACK_BOOK_UNKNOWN;
	} /* else buffer has some valid meta data from recognizable track */
#if 0
	/* first 12 bytes are bytes of synchronization:
	   00 FF FF FF FF FF FF FF FF FF FF 00;
	   we collect them just for debug purposes - it doesn't cost
	   much in terms of computer resources */
	unsigned char sector_sync[12];
	for (int i = 0; i < 12; i++) {
		sector_sync[i] = buffer[i];
	}
#endif
	/* then 4-byte header - present in Data CD sectors of all formats
	   (CD-ROM Mode1/2-Formless, CD-ROM Mode2 Form1/2) */
	unsigned char sector_header[4];
	for (int i = 0; i < 4; i++) {
		sector_header[i] = buffer[12 + i];
	}

	/* then (possibly) 8-byte subheader - it exists only in XA (Yellow
	   Book eXtended Attributes, CD-ROM/XA Mode2 Form1/2) disc sectors */
	unsigned char sector_subheader[8];
	/* I don't check if (disc->mode == CDIO_DISC_MODE_CD_XA) because
	   this is what I want to learn myself, without relying on cdio
	   information */
	for (int i = 0; i < 8; i++) {
		sector_subheader[i] = buffer[12 + 4 + i];
	}

	/* debug code */
	/*
	cdw_vdm ("sector: %d, driver return value: %d\n", (int) sector, (int) drv);

	cdw_vdm ("sync: >>");
	for (i = 0; i < 12; i++) {
		cdw_vdm (" %x ", sector_sync[i]);
	}
	cdw_vdm ("<<\n");

	cdw_vdm ("header: >>");
	for (i = 0; i < 4; i++) {
		cdw_vdm (" %x ", sector_header[i]);
	}
	cdw_vdm ("<<\n");

	cdw_vdm ("subheader: >>");
	for (i = 0; i < 8; i++) {
		cdw_vdm (" %x ", sector_subheader[i]);
	}
	cdw_vdm ("<<\n");

	cdw_vdm ("\nAnalyzing header:\n");
	cdw_vdm ("header byte 0 = %u (minutes)\n", sector_header[0]);
	cdw_vdm ("header byte 1 = %u (seconds)\n", sector_header[1]);
	cdw_vdm ("header byte 2 = %u (sector number within one second (1-75))\n", sector_header[2]);
	cdw_vdm ("header byte 3 = %u (sector mode (0/1/2))\n", sector_header[3]);
	*/

	/* we have header and (possibly) subheader bytes, let's check them */

	/* CDW_RED_BOOK_AUDIO is not checked, because
	   cdw_cdio_get_track_format_from_sector_header() should not be
	   called for audio track; there are no headers for audio sector,
	   so this code would produce invalid results  */

	cdw_track_format_t track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;

	if (sector_header[3] == 2) {

		track_format = CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS;
		/* may be CD-ROM/XA Mode2 Form1/2 track as well,
		   this is to be checked (and perhaps modified) below */

	} else if (sector_header[3] == 1) {
		track_format = CDW_TRACK_YELLOW_BOOK_MODE1; /* no doubt */
	} else if (sector_header[3] == 0) {
		/* this is CDW_TRACK_YELLOW_BOOK_MODE0, but we don't support this */
		track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
	} else {
		track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
	}

	/* track_format may still be modified if the disc is XA */

	if (sector_header[3] == 2) {
		/* this is Mode 2 sector, so perhaps it has subheader */

		cdw_sdm ("\nChecking if subheader bytes are valid\n");

		if ( (sector_subheader[0] == sector_subheader[4])
		     && (sector_subheader[1] == sector_subheader[5])
		     && (sector_subheader[2] == sector_subheader[6])
		     && (sector_subheader[3] == sector_subheader[7])) {

			cdw_sdm ("there seems to be valid subheader\n");

			cdw_sdm ("subheader byte 0 = %u (?? (00 for not interleaved))\n", sector_subheader[0]);
			cdw_sdm ("subheader byte 1 = %u (channel number)\n",              sector_subheader[1]);
			cdw_sdm ("subheader byte 2 = %u (submode byte)\n",                sector_subheader[2]);
			cdw_sdm ("   subheader bit 2.0 = %u (EOR - End Of Record (?))\n",    (sector_subheader[2] & 0x01));
			cdw_sdm ("   subheader bit 2.1 = %u (Video)\n",                      (sector_subheader[2] & 0x02) >> 1);
			cdw_sdm ("   subheader bit 2.2 = %u (Audio)\n",                      (sector_subheader[2] & 0x04) >> 2);
			cdw_sdm ("   subheader bit 2.3 = %u (Data)\n",                       (sector_subheader[2] & 0x08) >> 3);
			cdw_sdm ("   subheader bit 2.4 = %u (?)\n",                          (sector_subheader[2] & 0x10) >> 4);
			cdw_sdm ("   subheader bit 2.5 = %u (Form 1/2)\n",                   (sector_subheader[2] & 0x20) >> 5);
			cdw_sdm ("   subheader bit 2.6 = %u (Real-Time-Sector (?))\n",       (sector_subheader[2] & 0x40) >> 6);
			cdw_sdm ("   subheader bit 2.7 = %u (EOF)\n",                        (sector_subheader[2] & 0x80) >> 7);
			cdw_sdm ("subheader byte 3 = %u (Audio/Video encoding (0 for Data))\n", sector_subheader[3]);
			cdw_sdm ("subheader byte 4 = %u (should be equal to byte 0)\n",    sector_subheader[4]);
			cdw_sdm ("subheader byte 5 = %u (should be equal to byte 1)\n",    sector_subheader[5]);
			cdw_sdm ("subheader byte 6 = %u (should be equal to byte 2)\n",    sector_subheader[6]);
			cdw_sdm ("subheader byte 7 = %u (should be equal to byte 3)\n",    sector_subheader[7]);


			/* check track format encoded in byte 2, bit 5 */
			if ( ((sector_subheader[2] & 0x20) >> 5) == 0x01) {
				track_format = CDW_TRACK_YELLOW_BOOK_MODE2_FORM2;
			} else {
				track_format = CDW_TRACK_YELLOW_BOOK_MODE2_FORM1;
			}
		} else {
			/* subheader is invalid so we have to assume that
			   this is _not_ a XA Form1 / Form2 track, we stick
			   with mode set previously, based only on value of
			   sector_header[3], that is
			   CDW_YELLOW_BOOK_MODE2_FORMLESS */

			/* "no sign of valid subheader" doesn't imply error */
			cdw_sdm ("INFO: no sign of valid subheader\n");
			if (disc->mode == CDIO_DISC_MODE_CD_XA) {
				cdw_vdm ("ERROR: (?) cdio reported CDIO_DISC_MODE_CD_XA, but this track does not have correct subheader\n\n");
			}
		}
	} /* if (sector_header[3] == 2); we don't have to do anything in case of Mode 1 track */

	return track_format; /* success, non-black cdw track format */
}





/**
   \brief Assign correct value to sector_size of given track

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   This simple function checks cdw_format field of a track data structure
   and assigns correct value to sector_size field of given track.

   Sector size for unknown (black-book) track is set to zero.

   \param disc - disc data structure describing a disc
   \param t - index of track, for which you want to check sector size

   \return CDW_OK if given track is supported by cdw and sector size was set as non-zero
   \return CDW_ERROR if given track is not supported by cdw and sector size was set as zero
*/
cdw_rv_t cdw_cdio_resolve_sector_size(cdw_cdio_t *disc, track_t t)
{
	if (disc->tracks[t].cdw_track_format > CDW_TRACK_DVD) {
		disc->tracks[t].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
		cdw_assert (0, "ERROR: track format '%d' for track #%d is out of bounds\n",
			    disc->tracks[t].cdw_track_format, t);
		disc->tracks[t].sector_size = 0;
		return CDW_ERROR;
	}

	disc->tracks[t].sector_size = track_format_data[disc->tracks[t].cdw_track_format].payload_size;

	if (disc->tracks[t].sector_size == 0) {
		cdw_vdm ("ERROR: sector size for track #%d is 0\n", t);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Try to calculate correct value of "readable" sector in given track

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Each track has two parameters: n_sectors and n_sectors_to_read. It may
   happen that the two values are different (the second one smaller than
   the first one). This function calculates (probably) correct value of
   n_sectors_to_read using cdio_track_format field value or some cdio calls.

   Number of sectors to read for unknown (black-book) track is set to zero.

   \param disc - disc data structure describing a disc
   \param t - index of track, for which you want to calculate n_sectors_to_read

   \return CDW_OK if given track is supported by cdw and n_sectors_to_read was set as non-zero
   \return CDW_ERROR if given track is not supported by cdw and n_sectors_to_read was set as zero
*/
cdw_rv_t cdw_cdio_resolve_sectors_to_read(cdw_cdio_t *disc, track_t t)
{
	if (disc->tracks[t].cdw_track_format == CDW_TRACK_RED_BOOK_AUDIO) {
		cdw_sdm ("INFO: calculating sectors to read for audio track #%d as \"n_sectors\"\n", t);
		disc->tracks[t].n_sectors_to_read = disc->tracks[t].n_sectors;

	} else if (disc->tracks[t].cdw_track_format == CDW_TRACK_BLACK_BOOK_UNKNOWN){
		cdw_vdm ("INFO: calculating sectors to read for black book track #%d as zero\n", t);
		disc->tracks[t].n_sectors_to_read = 0;

	} else {
		/* this will work for ISO file systems on both CDs and DVDs */
		cdw_vdm ("INFO: calculating sectors to read for track #%d with iso analysis...\n", t);

		if (cdw_ofs_is_iso(disc->tracks[t].fs.cdio_fs)) {

			/* WARNING: for DVDs size of ISO volume (counted
			   in sectors) may be very different than
			   "last_sector - first_sector + 1"; the difference
			   is *very* significant, and only one of the values
			   is correct. Correct value is isofs_size. Be careful
			   when reading data from such ISO volume on DVD, some
			   cdio read() functions will make sure that you don't
			   attempt to read past "last_sector". In such cases
			   use functions reading iso9660 volume instead of
			   functions reading from disc device. */

			disc->tracks[t].n_sectors_to_read = (lsn_t) disc->tracks[t].fs.size;
		} else {
			/* using n_sectors in this situation is the
			   last possibility of getting some value of
			   n_sectors_to_read; may not be the best option,
			   but may be the only option; TODO: to be checked */
			disc->tracks[t].n_sectors_to_read = disc->tracks[t].n_sectors;
			cdw_vdm ("WARNING: no ISO FS detected, setting \"sectors to read\" = \"n_sectors\"\n");
		}
	}

	cdw_vdm ("INFO \"sectors to read\" for track #%d resolved as %d\n", t, disc->tracks[t].n_sectors_to_read);

	if (disc->tracks[t].n_sectors_to_read == 0) {
		cdw_vdm ("ERROR: failed to calculate sectors to read for track #%d\n", t);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}

}





/**
   \brief Print debug information about given disc

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   This is a debug function. Print to stderr information from given \p disc.

   \param disc - disc data structure describing a disc
*/
void cdw_cdio_disc_debug_print(cdw_cdio_t *disc)
{
	cdw_assert (disc->open, "ERROR: trying to get information about disc that is not open\n");

	cdw_vdm ("INFO: disc simple type is \"%s\"\n", disc->simple_type_label);

	if (disc->blank) {
		cdw_vdm ("INFO: disc is blank\n");
		return;
	}

	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		if (disc->mode == CDIO_DISC_MODE_CD_DATA) {
			cdw_vdm ("INFO: disc is CD / CDIO_DISC_MODE_CD_DATA\n");
		} else if (disc->mode == CDIO_DISC_MODE_CD_XA) {
			cdw_vdm ("INFO: disc is CD / CDIO_DISC_MODE_CD_XA\n");
		} else if (disc->mode == CDIO_DISC_MODE_CD_DA) {
			cdw_vdm ("INFO: disc is CD / CDIO_DISC_MODE_CD_DA\n");
		} else {
			cdw_vdm ("INFO: disc is CD / UNKNOWN\n");
		}
	} else if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		cdw_vdm ("INFO: disc is DVD\n");
	} else {
		cdw_assert (0, "ERROR: missed case\n");
	}

	/* FIXME: maybe this is not the best place to put this
	   specific FIXME, but...

	   in main window ('Volume' window -> 'On disc') we display
	   total size of data on disc. Sometimes it is a sum of sizes
	   of file systems in tracks - and this is bad, because tracks
	   are larger than field systems in the tracks.

	   Try to use file system size for reading data from tracks,
	   but also try to use n_sectors to show usage of disc
	   area. */

	for (track_t t = disc->first_track; t <= disc->last_track; t++) {
		cdw_vdm ("## INFO: ---------------------------------------- \n");
		cdw_vdm ("## INFO:      track number = %8d\n", t);
		cdw_vdm ("## INFO:      first_sector = %8d\n", disc->tracks[t].first_sector);
		cdw_vdm ("## INFO:       last_sector = %8d\n", disc->tracks[t].last_sector);
		cdw_vdm ("## INFO:         n_sectors = %8d (%.2f MB)\n", disc->tracks[t].n_sectors,
			 (disc->tracks[t].n_sectors * disc->tracks[t].sector_size) / (1024.0 * 1024));
		cdw_vdm ("## INFO: fs size (sectors) = %8d (%.2f MB) \n", disc->tracks[t].fs.size,
			 (disc->tracks[t].fs.size * disc->tracks[t].sector_size) / (1024.0 * 1024));
		cdw_vdm ("## INFO: n_sectors_to_read = %8d\n", disc->tracks[t].n_sectors_to_read);
		cdw_vdm ("## INFO:              diff = %8d\n", disc->tracks[t].n_sectors - disc->tracks[t].n_sectors_to_read);
		cdw_vdm ("## INFO:       sector_size = %8d\n", disc->tracks[t].sector_size);
		cdw_vdm ("## INFO:           fs type = %d / \"%s\"\n", disc->tracks[t].fs.cdio_fs, cdw_ofs_short_label(disc->tracks[t].fs.cdio_fs));
		cdw_vdm ("## INFO:      track format = %d / \"%s\"\n", disc->tracks[t].cdw_track_format, track_format_data[disc->tracks[t].cdw_track_format].cdw_track_format_label);
		cdw_vdm ("## INFO: ---------------------------------------- \n");
	}

	return;
}





/**
   \brief Read a track from disc, write it to file

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Function calls cdw_cdio_copy_sectors() to perform
   reading of \p n_sectors sectors from track \p on disc \p disc, and
   to copy them to file specified by \p fd file descriptor.

   You can set \p n_sectors to -1, the function will then read whole
   track, using default value of sectors to read (n_sectors_to_read
   of given track).

   \param disc - data structure describing a disc
   \param t - number of track to read
   \param fd - file descriptor of file to which to write content of track \p t
   \param n_sectors - number of sectors to read from track \p t

   \return 0 on success
   \return -1, -2 or -3 on function's internal errors
   \return 'errno - 5' on write() error, caller can interpret value of 'errno + 5' to check the error
*/
int cdw_cdio_copy_track(cdw_cdio_t *disc, track_t t, int fd, long long int n_sectors)
{
	cdw_assert (disc->p_cdio, "ERROR: trying to get info from NULL disc\n");
	cdw_assert (disc->open, "ERROR: trying to read from closed disc\n");
	cdw_assert (fd != -1 && fd != 0, "ERROR: invalid output file: %d\n", fd);

	int read_mode = 0;
	if (disc->tracks[t].cdw_track_format == CDW_TRACK_BLACK_BOOK_UNKNOWN) {
		cdw_vdm ("ERROR: called the function for black book (unknown) track\n");
		return -1;
	} else if (disc->tracks[t].cdw_track_format == CDW_TRACK_RED_BOOK_AUDIO) {
		read_mode = CDW_CDIO_READ_AUDIO_CD;
	} else if (cdw_ofs_is_iso(disc->tracks[t].fs.cdio_fs)) {
		read_mode = CDW_CDIO_READ_ISO9660;
	// } else if (cdw_ofs_is_udf(disc->tracks[t].fs.cdio_fs)) {
	//	read_mode = CDW_CDIO_READ_UDF;
	} else {
		cdw_assert (0, "ERROR: missed case\n");
	}

	cdw_cdio_disc_debug_print(disc);

	/* most probably we are re-using the same processwin for
	   N tracks, so clean leftovers from reading previous track */
	cdw_processwin_reset_progress();

	long long int sectors = n_sectors != -1 ? n_sectors : disc->tracks[t].n_sectors_to_read;
	return cdw_cdio_copy_sectors(disc, t, fd, sectors, read_mode);
}





/**
   \brief Copy sectors from given tracks to given file

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   This is a debug function.

   Print progress of reading a disc track to stderr.

   \param disc - data structure describing a disc
   \param t - number of track being read
   \param sector - current sector number - relative to beginning of a disc
*/
static inline void cdw_cdio_copy_debug_print(__attribute__((unused)) cdw_cdio_t *disc, __attribute__((unused)) track_t t, __attribute__((unused)) lsn_t sector)
{
	cdw_sdm ("INFO: track #%2d: sector size=%d, sector=%7d/%7d (non-relative: %7d/%7d)\n",
		 t,
		 disc->tracks[t].sector_size,
		 sector - disc->tracks[t].first_sector + 1, /* +1 to display sector numbers from 1, not 0 */
		 disc->tracks[t].n_sectors_to_read,
		 sector + 1,                                /* +1 to display sector numbers from 1, not 0 */
		 disc->tracks[t].first_sector + disc->tracks[t].n_sectors_to_read);

	return;
}





/**
   \brief Read sectors from disc, write them to given buffer

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   This is the most low level function in the module, it calls
   system function or cdio library function to perform actual reading
   from track \p of optical \p disc. Decision which read function to
   call is based on value of \p read_mode.
   \p n_sectors, starting from \p sector, are read at once, and stored
   in given \p buffer.

   \p buffer should be allocated by caller.

   Function returns return value of read function. Negative value
   indicates problems with reading, and should be also interpreted
   depending on \p read_mode.

   \p read_mode indicates if disc should be treated as audio disc
   or as data disc with ISO9660 file system.

   \param disc - data structure representing a disc
   \param track - track number to read from
   \param buffer - place to store data being read
   \param sector - starting position for reading
   \param n_sectors - number of sectors to read
   \param read_mode - mode of reading a disc

   \return negative value in case of errors
   \return non-negative value otherwise
*/
static inline long int cdw_cdio_read_sectors(cdw_cdio_t *disc, track_t t, unsigned char *buffer, lsn_t sector, uint32_t n_sectors, int read_mode)
{
	const int n_tries = 2; /* two read attempts seem to be reasonable */
	long int rv = 0;
	for (int i = 0; i < n_tries; i++) {
		if (read_mode == CDW_CDIO_READ_AUDIO_CD) {
			rv = cdio_read_audio_sectors(disc->p_cdio, buffer, sector, n_sectors);

		} else if (read_mode == CDW_CDIO_READ_ISO9660) {
			/* FIXME: shouldn't we call lseek() here? */
			/*
			  off_t offset = disc->tracks[t].sector_size * sector;
			  offset = offset == 0 ? 0 : offset - 1;
			  off_t o = lseek(disc->ofs->fd, offset, SEEK_SET);
			 */
			rv = read(disc->ofs->fd, (void *) buffer, disc->tracks[t].sector_size * n_sectors);
			if (rv == -1) {
				/* no re-reads with read() */
				int e = errno;
				cdw_vdm ("ERROR: fd = %d, errno = \"%s\"\n", disc->ofs->fd, strerror(e));
				break; /* no re-reads when using read(); */

			};
		// } else if (read_mode == CDW_CDIO_READ_UDF) {
			//rv = read_udf();
		} else {
			cdw_assert (0, "ERROR: missed case\n");
		}

		if (rv >= 0) {
			cdw_sdm ("INFO: success in attempt #%d\n", i);
			break;
		} else {
			int e = errno;
			cdw_vdm ("ERROR: failed at read attempt #%d\n", i);
			cdw_vdm ("ERROR: sector #%d: %d / \"%s\" / %ld / \"%s\"\n",
				 sector, e, strerror(e), rv, cdw_cdio_get_driver_error_label(rv));
		}
	}

	return rv;
}





/**
   \brief Read sectors from given track, write them to given file descriptor

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   This is a wrapper for cdw_cdio_read_sectors() and write() pair of
   functions called in a loop (with some additional code). The function
   also calls process window update function in a loop to periodically
   update reading progress information.

   This function cannot be called for track with unsupported format.

   \p fd must be file descriptior for file that is already
   open for writing.
   \p n_sectors_total may not be -1

   \p read_mode indicates if disc should be treated as audio disc
   or as data disc with ISO9660 file system.

   \param disc - disc to read from
   \param t - track number of track to read sectors from
   \param fd - file descriptor to write the data to (already opened and with proper permissions)
   \param n_sectors_total - number of sectors to read from track \p t
   \param read_mode - defines which function should be called to do reading

   \return 0 on success
   \return -1, -2 on function's internal errors
   \return -3 if write() didn't write all data passed to it, which may mean not enough available space on hard disc
   \return 'errno - 5' on other write() errors, caller can interpret value of 'errno + 5' to check the error
*/
int cdw_cdio_copy_sectors(cdw_cdio_t *disc, track_t t, int fd, long long int n_sectors_total, int read_mode)
{
#define CDW_CDIO_CALC_N_SECTORS(n_left)					\
	(n_left >= CDW_CDIO_RW_N_SECTORS ? CDW_CDIO_RW_N_SECTORS : n_left % CDW_CDIO_RW_N_SECTORS);

	/* buffer for data read from disc, CDIO_CD_FRAMESIZE_RAWER
	   is maximal size of sector defined in cdio header files */
	unsigned char buffer[CDIO_CD_FRAMESIZE_RAWER * CDW_CDIO_RW_N_SECTORS];

	/* size of data left to read from a track */
	lsn_t n_sectors_left = (lsn_t) n_sectors_total;

	/* first sector of ISO volume to start reading from */
	lsn_t sector = disc->tracks[t].first_sector;
	/* number of sectors to read/write at one time;
	   one sector has disc->tracks[t].sector_size bytes */
	lsn_t n_sectors = CDW_CDIO_CALC_N_SECTORS(n_sectors_left);

	cdw_rv_t retval = CDW_OK;
	while (n_sectors > 0) {
		cdw_cdio_copy_debug_print(disc, t, sector);

		long int r = cdw_cdio_read_sectors(disc, t, buffer, sector, (uint32_t) n_sectors, read_mode);
		if (r < 0) {
			cdw_vdm ("ERROR: failed read from track #%d\n", t);
			retval = -2;
			break;
		} else {
			ssize_t w = write(fd, buffer, disc->tracks[t].sector_size * (uint32_t) n_sectors);
			int e = errno;
			if (w != ((int) disc->tracks[t].sector_size * n_sectors)) {
				cdw_vdm ("ERROR: write error for track #%d (read %d, write %zd)\n",
					 t, disc->tracks[t].sector_size * n_sectors, w);
				cdw_vdm ("ERROR: errno = \"%s\"\n", strerror(e));

				if (w == -1) {
					retval = e - 5;
				} else {
					retval = -3;
				}
				break;
			}

			lsn_t relative_sector = sector - disc->tracks[t].first_sector;
			/* don't update processwin too often */
			if ((relative_sector % (40 * n_sectors)) == 0
			     || n_sectors < CDW_CDIO_RW_N_SECTORS) {

				cdw_cdio_copy_update_processwin(relative_sector, (lsn_t) n_sectors_total, disc->tracks[t].sector_size);
			}
			sector += n_sectors;
			n_sectors_left -= n_sectors;
			n_sectors = CDW_CDIO_CALC_N_SECTORS(n_sectors_left);
		}
	}
	if (retval == 0) {
		/* because this function reads N sectors at once, it may happen
		   that it reads all data from disc, but last update of processwin
		   shows "99%" instead of "100%". Let's fake "100%" here */
		lsn_t relative_sector = (lsn_t) n_sectors_total - 1;
		cdw_cdio_copy_update_processwin(relative_sector, (lsn_t) n_sectors_total, disc->tracks[t].sector_size);

		return 0;
	} else {
		return retval;
	}
}





/**
   \brief Print to log file information for read error

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Function prints error message that corresponds to given \p error
   value. The message is printed to cdw's log file.
   Call this function when cdw_cdio_copy_sectors() returns
   value different than zero, and pass the value to the function.

   \param error - error code
*/
void cdw_cdio_copy_print_debug_on_error(int error)
{
	if (error == -3) {
		cdw_vdm ("ERROR: error = -3, probably not enough space on disc\n");

		/* 2TRANS: this is string written to log file when
		   reading one track from CD disc is finished;
		   it is written after "Track %d: " string */
		cdw_logging_write(_("There is probably not enough space on hard disc for copied data\n"));
	} else if (error < -3) {
		/* cdw_cdio_copy_track() may
		   return 'errno - 5' for some errors */
		cdw_vdm ("ERROR: errno = %s\n", strerror(error + 5));

		/* 2TRANS: this is string written to log file when
		   reading one track from CD disc is finished with
		   problems; "%s" is system error message;
		   it is written after "Track %d: " string */
		cdw_logging_write(_("Maybe this error message will be helpful: \"%s\"\n"), strerror(error + 5));
	} else {
		/* no more information to print */
	}

	return;
}





/**
   \brief Read sync bytes, header and subheader (if available) from given CD sector

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   No payload data is read.
   \p buffer has to have size of (12 + 4 + 8) bytes.

   On success the function returns number of bytes placed in \p sector bytes:
   either this is 16 bytes (12 bytes sync + 4 bytes header) or 24 (12 bytes
   sync + 4 bytes header + 8 bytes subheader). This value can be used by
   caller to learn if this function fetched a subheader or not.

   On errors the function returns error code (negative number) returned by
   mmc_read_cd(), caller can cast the value to driver_return_code_t type and
   process further.

   \param disc - disc to read from
   \param sector - number of sector to read
   \param buffer - place where to store sector bytes read from CD disc

   \return mmc error code (negative value) on mmc read error
   \return number of bytes read from cd and placed in \p sector_buffer
*/
int cdw_cdio_get_cd_sector_meta_data(cdw_cdio_t *disc, lsn_t sector, unsigned char *buffer)
{
	/* NOTE - this will fail if drive does not have MMC capabilities or
	   this functionality is not supported, but what other portable
	   mechanism could we use?

	   This version of function does not try to guess track type
	   before calling mmc_read_cd(), it just tries to get
	   header + subheader, and if it fails then tries again to
	   get header alone. It is up to caller to decide what to do
	   with data read into sector_buffer[] and how to handle error
	   returned by mmc_read_cd(); */

	/* note that CDIO_DISC_MODE_CD_CDDA should be ruled out before
	   calling this function */
	cdw_assert (disc->mode != CDIO_DISC_MODE_CD_DA, "ERROR: you shouldn't call this function for Audio CD\n");
	cdw_assert (disc->p_cdio, "ERROR: p_cdio is NULL\n");

	/* which sector header should be returned? (data header and subheader)
	    0: none, 1: 4-bit header, 2: 8-bit subheader, 3: both */
	uint8_t header_code = 0;
	uint16_t block_size = 0;
	driver_return_code_t drv;
	for (int i = 0; i < 2; i++) {
		if (i == 0) {
			/* first test most "aggressive" option */
			header_code = 3; /* get both header and subheader; (XA sector?) */
			/* 12: sync bytes; 4: header bytes; 8: subheader bytes */
			block_size = 12 + 4 + 8;
		} else {
			header_code = 1; /* get only header; (DATA sector?) */
			/* 12: sync bytes; 4: header bytes */
			block_size = 12 + 4;
		}
		cdw_sdm ("INFO: calling mmc_read_cd(), expecting to get %s from sector #%d\n",
			 (header_code == 3) ? "header + subheader" : "header", sector);

		drv = mmc_read_cd(disc->p_cdio,  /* object to read from */
				  buffer,        /* place to store data */
				  sector,        /* sector number to read */
				  0,             /* expected sector type; 0 - all sector types */
				  false,         /* b_digital_audio_play - audio related stuff, doesn't matter */
				  true,          /* b_sync_header - return synchronization header */
				  header_code,   /* uint8_t header_code - data header and subheader; 0: none, 1: 4-bit header, 2: 8-bit subheader. 3: both */
				  false,         /* bool b_user_data - return user data? */
				  false,         /* bool b_edc_ecc - error detection/correction data */
				  0,             /* uint8_t c2_error_information, yet another error-related field (?) */
				  0,             /* uint8_t subchannel_selection - subchannel selection bits (?)*/
				  block_size,    /* uint16_t i_blocksize - size of the block expected to be returned */
				  1);            /* uint32_t i_blocks - number of blocks expected to be returned */

		if (drv < 0) {
			cdw_vdm ("ERROR: mmc_read_cd() returned error value = %d\n", drv);
		} else {
			cdw_sdm ("INFO: mmc_read_cd() returned success value = %d\n", drv);
			return (int) block_size;
		}
	} /* for () */

	return (int) drv; /* this will be an error value */
}





/**
   \brief Display in processwin information about current state of copying a disc

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Display information about progress of copying a disc using provided
   information about sector number and size.

   Make sure that a process window is already active.

   \param relative_sector - sector number, relative to track start
   \param n_sectors - number of sectors in a track
   \param sector_size - size of payload in a sector
*/
void cdw_cdio_copy_update_processwin(lsn_t relative_sector, lsn_t n_sectors, int sector_size)
{
	cdw_assert (cdw_processwin_is_active(), "ERROR: calling the function for inactive processwin\n");

	/* dividing by 4s to avoid overflows */
	long current = ((relative_sector / 4) * (sector_size / 4)) / (256 * 256);
	long total = ((n_sectors / 4) * (sector_size / 4)) / (256 * 256);
	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];

	/* 2TRANS: this string will be displayed as message in progress window;
	   first %ld is amount of data already read from cd,
	   second %ld is total amount of data to be read */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("%ld/%ld MB"), current, total);
	cdw_sdm ("INFO: current value string = \"%s\"\n", current_value_string);
	cdw_processwin_display_progress_conditional(current, total, current_value_string);

	return;
}





/**
   \brief Check if given track is audio track

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   There can be two types of disc track: audio track, and data track.
   The function checks if given track \p t is audio track.

   Only tracks on Audio CD disc are treated as audio tracks.

   \param disc - data structure describing a disc
   \param t - track number

   \return true if given track is audio track
   \return false if given track is not audio track
*/
bool cdw_cdio_is_audio_track(cdw_cdio_t *disc, track_t t)
{
	cdw_assert (disc->open, "ERROR: trying to get information about disc that is not open\n");
	cdw_assert (t >= disc->first_track && t <= disc->last_track,
		    "ERROR: asking for track %d out of range %d-%d\n",
		    t, disc->first_track, disc->last_track);

	if (disc->tracks[t].cdw_track_format == CDW_TRACK_RED_BOOK_AUDIO) {
		return true;
	} else {
		return false;
	}
}





/**
   \brief Check if given track is a data track

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   There can be two types of disc track: audio track, and data track.
   The function checks if given track \p t is a data track.

   Tracks on DVD disc are treated as data tracks.

   \param disc - data structure describing a disc
   \param t - track number

   \return true if given track is data track
   \return false if given track is not a data track
*/
bool cdw_cdio_is_data_track(cdw_cdio_t *disc, track_t t)
{
	cdw_assert (disc->open, "ERROR: trying to get information about disc that is not open\n");
	cdw_assert (t >= disc->first_track && t <= disc->last_track,
		    "ERROR: asking for track %d out of range %d-%d\n",
		    t, disc->first_track, disc->last_track);

	if (disc->tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE1
	    || disc->tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS
	    || disc->tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE2_FORM1
	    || disc->tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE2_FORM2
	    /* let's pretend that all DVDs are data DVDs */
	    || disc->tracks[t].cdw_track_format == CDW_TRACK_DVD) {

		cdw_vdm ("INFO: track #%d is a data track\n", t);
		return true;
	} else {
		cdw_vdm ("INFO: track #%d is not a data track\n", t);
		return false;
	}
}




/* cdio driver errors */
cdw_id_clabel_t driver_errors[] = {
	{ DRIVER_OP_ERROR,          "DRIVER_OP_ERROR" },
	{ DRIVER_OP_UNSUPPORTED,    "DRIVER_OP_UNSUPPORTED" },
	{ DRIVER_OP_UNINIT,         "DRIVER_OP_UNINIT" },
	{ DRIVER_OP_NOT_PERMITTED,  "DRIVER_OP_NOT_PERMITTED" },
	{ DRIVER_OP_BAD_PARAMETER,  "DRIVER_OP_BAD_PARAMETER" },
	{ DRIVER_OP_BAD_POINTER,    "DRIVER_OP_BAD_POINTER" },
	{ DRIVER_OP_NO_DRIVER,      "DRIVER_OP_NO_DRIVER" },
	{ 0,                        (char *) NULL }};



/**
   \brief Debug function printing name of cdio driver error return value

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Some cdio functions return value of driver_return_code_t type. If this
   value is not DRIVER_OP_SUCCESS or greater than zero, then the value
   should be analyzed and (hopefully) acted upon. Here we print name of
   this error.

   Returned pointer is owned by the function.

   \param error_num - return value from cdio driver function

   \return string corresponding to \p error_num
*/
const char *cdw_cdio_get_driver_error_label(driver_return_code_t error_num)
{
	return cdw_utils_id_label_table_get_label(driver_errors, (int) error_num);
}





/**
   \brief Handler of cdio debug message events

   \date Function's top-level comment reviewed on 2012-01-28
   \date Function's body reviewed on 2012-01-28

   Function prints debug messages emitted by cdio. The function must be
   registered with cdio_log_set_handler().

   \param level - debug level of a message
   \param message - message text
*/
static void cdw_cdio_log_handler(cdio_log_level_t level, const char *message)
{
	cdw_id_clabel_t handler_levels[] = {
		{ CDIO_LOG_DEBUG,  "debug"       },
		{ CDIO_LOG_INFO,   "info"        },
		{ CDIO_LOG_WARN,   "warning"     },
		{ CDIO_LOG_ERROR,  "error"       },
		{ CDIO_LOG_ASSERT, "assert"      },
		{ 0,               (char *) NULL }};

	const char *prefix = (char *) NULL;
	if (level == CDIO_LOG_DEBUG || level == CDIO_LOG_INFO) {
		prefix = "INFO";
	} else if (level == CDIO_LOG_WARN) {
		prefix = "WARNING";
	} else {
		prefix = "ERROR";
	}

	cdw_vdm ("%s: libcdio %s message: \"%s\"\n",
		 prefix,
		 cdw_utils_id_label_table_get_label(handler_levels, level), message);

	return;
}
