/****************************************************************************
 *    lib/c/Program.cpp - This file is part of coala						*
 *																			*
 *    Copyright (C) 2009  Torsten Grote										*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/

#include "Program.h"
#include "Parser.h"

using namespace C;

Program::Program(Coala::CompilerOptions* options) {
	options_ = options;
    ltl_node_num_ = 1;
	ltlquery_ = NULL;

	// true/false fluent
	fluents_.insert( make_pair("0", new FluentAction(new string("0"))) );
}

Program::~Program() {
	for(map<string,FluentAction*>::iterator n = fluents_.begin(); n != fluents_.end(); ++n) {
		delete n->second;
	}
}

void Program::setLine(int* line) {
	line_ = line;
}

Variable* Program::addVariable(string* name) {
	map<string,Variable*>::iterator iter = variables_.find(*name);
	
	if(iter != variables_.end()) {
		return variables_[*name];
	}
	else {
		Variable* var = new Variable(name);
		variables_.insert(make_pair(*name, var));
		return var;
	}
}

Formula* Program::addFormula(char type, Formula* identifier) {
	// TODO check for duplicates
	return new Formula(type, identifier);
}

Formula* Program::addFormula(char type, vector<Formula*>* vec) {
	// TODO check for duplicates
	return new Formula(type, vec);
}

bool Program::addAction(FluentAction* action) {
	pair<map<string,FluentAction*>::iterator,bool> result = actions_.insert( make_pair(action->getName(), action) );
	return result.second;
}

void Program::addActionDef(Formula* action_list, Types* types) {
	FluentAction* a;

	for(vector<Formula*>::iterator action = action_list->begin(); action != action_list->end(); action++) {
		a = (FluentAction*) *action;
		
		a->setClass("Action");
		if(!addAction(a))
			cerr << "Warning: action " << a->getName() << " defined multiple times at line " << *line_ << ".\n";
	}
	action_definitions_.push_back(new ActionDefinition(*line_, action_list, types));
}

bool Program::addFluent(FluentAction* fluent) {
	pair<map<string,FluentAction*>::iterator,bool> result = fluents_.insert( make_pair(fluent->getName(), (FluentAction*)fluent) );
	return result.second;
}

void Program::addFluentDef(Formula* fluent_list, Types* types) {
	FluentAction* f;
	
	for(vector<Formula*>::iterator fluent = fluent_list->begin(); fluent != fluent_list->end(); fluent++) {
		if((*fluent)->getType() == 'n') {
			cerr << "Warning: fluent '" << f->getName() << "' can not be defined negatively at line " << *line_ << ".\n";
			f = (*fluent)->getFluentAction();
			*fluent = f;
		}
		else f = (FluentAction*) *fluent;
		
		f->setClass("Fluent");
		if(!addFluent(f))
			cerr << "Warning: fluent '" << f->getName() << "' defined multiple times at line " << *line_ << ".\n";
	}
	fluent_definitions_.push_back(new FluentDefinition(*line_, fluent_list, types));
}


void Program::addStaticRule(Formula* F, Formula* G, Types* types) {
	if(options_->getLanguage() == "c") {
		transformFormula(F, "Fluent");
		transformFormula(G, "Fluent");
	}
	else if(options_->getLanguage() == "m") {
		transformFormula(F, "Action");
		transformFormula(G, "Action");
	}
	
	static_rules_.push_back(new StaticRule(*line_, F, G, types));
}

void Program::addDynamicRule(Formula* F, Formula* G, Formula* H, Types* types) {
	transformFormula(F, "Fluent");
	transformFormula(G, "Fluent");
	transformFormula(H, "Action");
	
	dynamic_rules_.push_back(new DynamicRule(*line_, F, G, H, types));
}


// <default> f_1,...,f_n <if> g_1,...,g_m = <caused> f <if> f_1, g_1,...,g_m
void Program::addDefaultRule(Formula* F, Formula* G, Types* types) {
	for(vector<Formula*>::iterator n = F->begin(); n != F->end(); ++n) {
		Formula* F_new = addFormula(F->getType(), *n);
		
		vector<Formula*>* G_new = new vector<Formula*>(1, *n);
		if(G) G_new->insert(G_new->end(), G->begin(), G->end());
		
		addStaticRule(F_new, addFormula('c', G_new), types);
	}
	delete F;
	delete G;
}

// <inertial> f_1, ..., f_n = <caused> f <if> f <after> f
void Program::addInertialRule(Formula* F, Types* types) {
	for(vector<Formula*>::iterator n = F->begin(); n != F->end(); ++n) {
		Formula* F_new = addFormula(F->getType(), *n);
		
		addDynamicRule(F_new, F_new, F_new, types);
	}
	delete F;
}

// a1,...,a_o <causes> f_1, ..., f_n <if> g_1, ..., g_m = <caused> f <if> <true> <after> g, a
void Program::addCausesRule(Formula* A, Formula* F, Formula* G, Types* types) {
	Formula* G_new;

	if(G) {
		G->insert(G->begin(), A->begin(), A->end());
		G_new = G;
	}
	else {
		G_new = A;
	}
	
	addDynamicRule(F, NULL, G_new, types);
	//	Don't delete G here since it is G_new as well
}

// <always> f_1, ..., f_n = <caused> <false> <if> -f_1, ..., -f_n
void Program::addAlwaysRule(Formula* G, Types* types) {
	Formula* F = addFormula('c', addFormula('n', new FluentAction(new string("0"))));
	
	G->invert();
	
	addStaticRule(F, G, types);
}

// a_1,...,a_n <may cause> f_1,...,f_m <if> g_1,...,g_l = <caused> f_1,...,f_m <if> f_1,...,f_m <after> g_1,...,g_l,a_1,...,a_n
void Program::addMayCauseRule(Formula* A, Formula* F, Formula* G, Types* types) {
	if(G) G->insert(G->begin(), A->begin(), A->end());
	else  G = A;

	addDynamicRule(F, F, G, types);
}

// <nonexecutable> a_1,..., a_n <if> f_1,..., f_m = <caused> <false> <after> a_1,...,a_n,f_1,...,f_m
void Program::addNonexecutableRule(Formula* A, Formula* G, Types* types) {
	Formula* F = addFormula('c', new Formula('n', new FluentAction(new string("0"))));
	
	if(G) G->insert(G->begin(), A->begin(), A->end());
	else  G = A;
	
	addDynamicRule(F, NULL, G, types);
}

void Program::addQQuery(string type, Formula* A, string time, Types* types) {
	transformFormula(A, "Action");

	qqueries_.push_back(new QQuery(*line_, type, A, &time, types));
}

void Program::addRQuery(Formula* F, Formula* A, string time, Types* types) {
	transformFormula(F, "Fluent");
	transformFormula(A, "Action");
	
	rqueries_.push_back(new RQuery(*line_, F, A, &time, types));
}

LTLNode* Program::getLTLNode(int type, LTLNode* left, LTLNode* right, Formula* content) {
	if(content) {
		if(content->getType() != '_') throw std::runtime_error("Invalid operator in front of atomic LTL formula.");
		transformFormula(addFormula('c', content), "unknown");
		return new LTLNode(type, (FluentAction*)content, ltl_node_num_++);
	}
	else return new LTLNode(type, left, right, ltl_node_num_++);
}

void Program::addLTLQuery(LTLNode* node) {
	if(ltlquery_) ltlquery_->setNode(getLTLNode(Parser::LTLOR, ltlquery_->getNode(), node, NULL));
	else ltlquery_ = new LTLQuery(node, *line_);
}

void Program::print(Printer* p) {
	// prints all Variables of the Program
//	for(map<string, Variable*>::iterator n = variables_.begin(); n != variables_.end(); ++n) {
//		cout << n->first << "\n";
//	}

	p->add("\n%\n% Fluents\n%\n");
	for(vector<FluentDefinition*>::iterator n = fluent_definitions_.begin(); n != fluent_definitions_.end(); ++n) {
		(*n)->print(p, ltlquery_);
		delete *n;
	}
	
	p->setSection('c');
	p->add("\n%\n% Actions\n%\n");
	for(vector<ActionDefinition*>::iterator n = action_definitions_.begin(); n != action_definitions_.end(); ++n) {
		(*n)->print(p);
	}
	
	p->setSection('c');
	p->add("\n%\n% Static Rules\n%\n");
	for(vector<StaticRule*>::iterator n = static_rules_.begin(); n != static_rules_.end(); ++n) {
		(*n)->print(p);
	}
	
	p->add("\n%\n% Dynamic Rules\n%\n");
	for(vector<DynamicRule*>::iterator n = dynamic_rules_.begin(); n != dynamic_rules_.end(); ++n) {
		(*n)->print(p);
	}
	
	p->setSection('v');
	p->add("\n%\n% Queries\n%\n");
	for(vector<QQuery*>::iterator n = qqueries_.begin(); n != qqueries_.end(); ++n) {
		(*n)->print(p);
	}
	for(vector<RQuery*>::iterator n = rqueries_.begin(); n != rqueries_.end(); ++n) {
		(*n)->print(p);
	}
	
	if(ltlquery_) {
		ltlquery_->print(p, &fluents_);
		p->add("\n");
	}
	
	p->print();
}

void Program::transformFormula(Formula* formula, string type) {
	if(formula) {
		string error; stringstream ss; ss << *line_; ss >> error;
		
		for(vector<Formula*>::iterator n = formula->begin(); n != formula->end(); ++n) {
			FluentAction* i = (*n)->getFluentAction();

			map<string,FluentAction*>::iterator fluent_it = fluents_.find(i->getName());
			
			if(fluent_it != fluents_.end()) {
				i->setClass("Fluent");
// TODO merge fluents if value and arguments are equal
//				*n = fluent_it->second;
			}
			else if(type == "Fluent") {
				throw std::runtime_error(i->getName() + " was not defined as a fluent prior to line " + error + ".\n");
			}
			else {
				map<string,FluentAction*>::iterator action_it = actions_.find(i->getName());
				
				if(action_it != actions_.end()) {
					i->setClass("Action");
// TODO merge actions if value and arguments are equal
// 					*n = action_it->second;
				}
				else {
					throw std::runtime_error(i->getName() + " was not defined as action or fluent prior to line " + error + ".\n");
				}
			}
		} // end for
	} // end if != NULL
}
