#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "control_file.hpp"

#include "architecture.hpp"
#include "cross_compiling.hpp"

#include "unknown_argument.hpp"

#include "silent_progress.hpp"

#include "transformations.hpp"

#include <apt-pkg/init.h>
#include <apt-pkg/cachefile.h>

#include <iostream>
#include <exception>

int main(int const argc, char const *const *const argv) throw()
try
{
	typedef char const *const *arg_iterator;
	arg_iterator const args_begin = &argv[1];
	arg_iterator const args_end = &argv[argc];

	std::list<std::string> pkgs;
	bool archdep_only = false;

	enum
	{
		initial,
		pkg_name
	} state = initial;

	for(arg_iterator a = args_begin; a != args_end; ++a)
	{
		std::string arg(*a);
		switch(state)
		{
		case initial:
			if(arg == "-B")
				archdep_only = true;
			else if(arg == "-p")
				state = pkg_name;
			else
				throw unknown_argument(arg);
			break;
		case pkg_name:
			pkgs.push_back(arg);
			state = initial;
		}
	}

	control_file control;
	try
	{
		control.merge("debian/xcontrol");
	}
	catch(std::exception &)
	{
		control.merge("debian/control");
	}
	dependency_list dependencies = control.source().build_depends_tools;

	if(!archdep_only)
		dependencies += control.source().build_depends_indep;

	if(architecture::host != architecture::build)
		dependencies += cross_compiling::to_cross(architecture::host, control.source().build_depends);
	else
		dependencies += control.source().build_depends;

	bool successful = true;

	if(pkgs.empty())
		transform_per_pkg_build_depends(control);

	for(std::list<std::string>::const_iterator i = pkgs.begin();
			i != pkgs.end(); ++i)
	{
		control_file::binary_const_iterator sec = control.find_binary(*i);
		if(sec == control.binaries_end())
		{
			std::cerr << "E: package " << *i << " not found in control file" << std::endl;
			successful = false;
			continue;
		}
		dependencies += sec->build_depends_tools;

		if(!archdep_only)
			dependencies += sec->build_depends_indep;

		if(architecture::host != architecture::build)
			dependencies += cross_compiling::to_cross(architecture::host, sec->build_depends);
		else
			dependencies += sec->build_depends;
	}

	pkgInitConfig(*_config);
	pkgInitSystem(*_config, _system);

	silent_progress progress;

	pkgCacheFile cache;
	cache.Open(progress, false);
	
	for(dependency_list::const_iterator i = dependencies.begin(); i != dependencies.end(); ++i)
	{
		bool alternatives_successful = false;
		for(dependency_alternative_list::const_iterator j = i->begin(); j != i->end(); ++j)
		{
			pkgCache::PkgIterator pkg = cache->FindPkg(j->package);
			if(pkg.end())
			{
				std::cerr << "I: dependency on " << *j
					<< " not fulfilled (and package not available)"
					<< std::endl;
			}
			else
			{
				if(pkg->ProvidesList)
				{
					std::cerr << "I: " << j->package << " is a virtual package." << std::endl;
					for(pkgCache::PrvIterator p = pkg.ProvidesList(); !p.end(); ++p)
					{
						pkgCache::PkgIterator pp = p.OwnerPkg();
						if(cache[pp].Status == 2)
							continue;
						std::cerr << "I: " << pp.Name() << " "
							<< pp.CurrentVer().VerStr()
							<< " provides it." << std::endl;
						alternatives_successful = true;
					}
				}
				else
				{
					if(pkg->CurrentVer)
					{
						std::cerr << "I: dependency on " << *j
							<< " fulfilled by version " << pkg.CurrentVer().VerStr() << std::endl;
						alternatives_successful = true;
					}
					else
					{
						std::cerr << "I: dependency on " << *j
							<< " not fulfilled."
							<< std::endl;
					}
				}
			}
		}
		if(!alternatives_successful)
		{
			std::cerr << "E: dependency on " << *i
				<< " not fulfilled" << std::endl;
			successful = false;
		}
	}

	cache.Close();

	return successful?0:1;
}
catch(std::exception &e)
{
	std::cerr << "E: " << e.what() << std::endl;
	return 1;
}
