/**
 * Add a new thumb object to the parent.
 *
 * @param parent The parent object.
 * @return The new object or NULL if it cannot be created.
 *
 * @see elm_thumb_file_set()
 * @see elm_thumb_ethumb_client_get()
 *
 * @ingroup Thumb
 */
EAPI Evas_Object                *elm_thumb_add(Evas_Object *parent);

/**
 * Reload thumbnail if it was generated before.
 *
 * @param obj The thumb object to reload
 *
 * This is useful if the ethumb client configuration changed, like its
 * size, aspect or any other property one set in the handle returned
 * by elm_thumb_ethumb_client_get().
 *
 * If the options didn't change, the thumbnail won't be generated again, but
 * the old one will still be used.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_reload(Evas_Object *obj);

/**
 * Set the file that will be used as thumbnail @b source.
 *
 * @param obj The thumb object.
 * @param file The path to file that will be used as thumbnail source.
 * @param key The key used in case of an EET file.
 *
 * The file can be an image or a video (in that case, acceptable
 * extensions are: avi, mp4, ogv, mov, mpg and wmv). To start the
 * video animation, use the function elm_thumb_animate().
 *
 * @see elm_thumb_file_get()
 * @see elm_thumb_reload()
 * @see elm_thumb_animate()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_file_set(Evas_Object *obj, const char *file, const char *key);

/**
 * Get the image or video path and key used to generate the thumbnail.
 *
 * @param obj The thumb object.
 * @param file Pointer to filename.
 * @param key Pointer to key.
 *
 * @see elm_thumb_file_set()
 * @see elm_thumb_path_get()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_file_get(const Evas_Object *obj, const char **file, const char **key);

/**
 * Get the path and key to the image or video thumbnail generated by ethumb.
 *
 * One just needs to make sure that the thumbnail was generated before getting
 * its path; otherwise, the path will be NULL. One way to do that is by asking
 * for the path when/after the "generate,stop" smart callback is called.
 *
 * @param obj The thumb object.
 * @param file Pointer to thumb path.
 * @param key Pointer to thumb key.
 *
 * @see elm_thumb_file_get()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_path_get(const Evas_Object *obj, const char **file, const char **key);

/**
 * Set the aspect for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param aspect The aspect setting.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_aspect_set(Evas_Object *obj, Ethumb_Thumb_Aspect aspect);

/**
 * Get the aspect of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The aspect setting.
 *
 * @see elm_thumb_aspect_set()
 *
 * @ingroup Thumb
 */
EAPI Ethumb_Thumb_Aspect elm_thumb_aspect_get(const Evas_Object *obj);

/**
 * Set the FDO size for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param size The FDO size setting.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_fdo_size_set(Evas_Object *obj, Ethumb_Thumb_FDO_Size size);

/**
 * Get the fdo size of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The FDO size setting.
 *
 * @see elm_thumb_fdo_size_set()
 *
 * @ingroup Thumb
 */
EAPI Ethumb_Thumb_FDO_Size elm_thumb_fdo_size_get(const Evas_Object *obj);

/**
 * Set the format for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param size The format setting.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_format_set(Evas_Object *obj, Ethumb_Thumb_Format format);

/**
 * Get the format of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The Format setting.
 *
 * @see elm_thumb_format_set()
 *
 * @ingroup Thumb
 */
EAPI Ethumb_Thumb_Format elm_thumb_format_get(const Evas_Object *obj);

/**
 * Set the orientation for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param size The orientation setting.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_orientation_set(Evas_Object *obj, Ethumb_Thumb_Orientation orient);

/**
 * Get the orientation of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The orientation setting.
 *
 * @see elm_thumb_orientation_set()
 *
 * @ingroup Thumb
 */
EAPI Ethumb_Thumb_Orientation elm_thumb_orientation_get(const Evas_Object *obj);

/**
 * Set the size for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param tw The width of the thumb.
 * @param th The height of the thumb.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_size_set(Evas_Object *obj, int tw, int th);

/**
 * Get the size of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The size setting.
 *
 * @see elm_thumb_size_set()
 *
 * @ingroup Thumb
 */
EAPI void elm_thumb_size_get(const Evas_Object *obj, int *tw, int *th);

/**
 * Set the crop alignment for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param cropx The x coordinate of the crop.
 * @param cropy The y coordinate of the crop.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_crop_align_set(Evas_Object *obj, double cropx, double cropy);

/**
 * Get the crop alignment of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The crop align setting.
 *
 * @see elm_thumb_crop_align_set()
 *
 * @ingroup Thumb
 */
EAPI void elm_thumb_crop_align_get(const Evas_Object *obj, double *cropx, double *cropy);

/**
 * Set the compression for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param compress The compression of the thumb.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_compress_set(Evas_Object *obj, int compress);

/**
 * Get the compression of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The compress setting.
 *
 * @see elm_thumb_compress_set()
 *
 * @ingroup Thumb
 */
EAPI void elm_thumb_compress_get(const Evas_Object *obj, int *compress);


/**
 * Set the quality for the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @param quality The quality of the thumb.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_quality_set(Evas_Object *obj, int quality);

/**
 * Get the quality of the thumb object.
 * @since 1.8
 *
 * @param obj The thumb object.
 * @return getting The quality setting.
 *
 * @see elm_thumb_quality_set()
 *
 * @ingroup Thumb
 */
EAPI void elm_thumb_quality_get(const Evas_Object *obj, int *quality);

/**
 * Set the animation state for the thumb object. If its content is an animated
 * video, you may start/stop the animation or tell it to play continuously and
 * looping.
 *
 * @param obj The thumb object.
 * @param s The animation setting.
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_animate_set(Evas_Object *obj, Elm_Thumb_Animation_Setting s);

/**
 * Get the animation state for the thumb object.
 *
 * @param obj The thumb object.
 * @return getting The animation setting or @c ELM_THUMB_ANIMATION_LAST,
 * on errors.
 *
 * @see elm_thumb_animate_set()
 *
 * @ingroup Thumb
 */
EAPI Elm_Thumb_Animation_Setting elm_thumb_animate_get(const Evas_Object *obj);

/**
 * Make the thumbnail 'editable'.
 *
 * @param obj Thumb object.
 * @param edit Turn on or off editability. Default is @c EINA_FALSE.
 *
 * This means the thumbnail is a valid drag target for drag and drop, and can be
 * cut or pasted too.
 *
 * @see elm_thumb_editable_get()
 *
 * @ingroup Thumb
 */
EAPI Eina_Bool                   elm_thumb_editable_set(Evas_Object *obj, Eina_Bool edit);

/**
 * Make the thumbnail 'editable'.
 *
 * @param obj Thumb object.
 * @return Editability.
 *
 * This means the thumbnail is a valid drag target for drag and drop, and can be
 * cut or pasted too.
 *
 * @see elm_thumb_editable_set()
 *
 * @ingroup Thumb
 */
EAPI Eina_Bool                   elm_thumb_editable_get(const Evas_Object *obj);
