SUBROUTINE CONSINT (KLAT_IN, KLON_IN, PLAT_IN, KIN,&
                         & KLAT_OUT,KLON_OUT,PLAT_OUT,KOUT,&
                         & PFLD_IN, PFLD_OUT, &
                         & LDPAR)

!    Purpose.
!    --------
!      Conserving bicubic interpolation of grid field from model 
!      resolution to grid defined by input parameter. Assumes the 
!      whole field is in memory.

!    Explicit arguments:
!    -------------------

!    Input:
!      KLAT_IN  - number of latitude rows of the input grid
!      KLON_IN  - number of longitudes for each row of the input grid
!      PLAT_IN  - latitude (degrees) of each row of the input grid
!      KIN      - size of input array
!      KLAT_OUT - number of latitude rows of the output grid
!      KLON_OUT - number of longitudes for each row of the output grid
!      PLAT_OUT - latitude (degrees) of each row of the output grid
!      KOUT     - size of output array
!      PFLD_IN  - array of grid values at input resolution

!    Output:
!      PFLD_OUT - array of interpolated values

!    Optional:
!      LDPAR    - parity for extra-polar calculations
!                 .TRUE. for scalar fields
!                 .FASE. for vector fields

!    Author.
!    -------
!      E. Holm - modification of GRID_BICUBIC by Y.Tremolet

!    Modifications.
!    --------------
!      Original: 27/05/04
!      20050507 Elias Holm: Introduce order and limiter in call and
!                           generalize code for up to quintic interpolation
!        Y.Tremolet    26-Jan-2005 Added optional arguments
!      20080204 Elias Holm: Surface pressure on input/output and
!                           simplified
!      20081015 Elias Holm: Simplify and reorder calculations for speed
!      20101230 Elias Holm: Simplified version without pressure thickness 
!                           weighting
!      20110125 Elias Holm: Latitude inputs in degrees
!     ------------------------------------------------------------------

IMPLICIT NONE

INTEGER,INTENT(IN)    :: KLAT_IN 
INTEGER,INTENT(IN)    :: KLON_IN(KLAT_IN) 
REAL   ,INTENT(IN)    :: PLAT_IN(KLAT_IN) 
INTEGER,INTENT(IN)    :: KIN 
INTEGER,INTENT(IN)    :: KLAT_OUT 
INTEGER,INTENT(IN)    :: KLON_OUT(KLAT_OUT) 
REAL   ,INTENT(IN)    :: PLAT_OUT(KLAT_OUT) 
INTEGER,INTENT(IN)    :: KOUT 
REAL   ,INTENT(IN)    :: PFLD_IN(KIN)
REAL   ,INTENT(OUT)   :: PFLD_OUT(KOUT) 
LOGICAL, OPTIONAL, INTENT(IN)    :: LDPAR

INTEGER :: IND_IN  (KLAT_IN)
INTEGER :: ILON_IN (-2:KLAT_IN+3)
REAL    :: ZLAT_IN (-2:KLAT_IN+3)
REAL    :: ZLON_IN (-2:KLAT_IN+3)
REAL    :: ZDLAT_IN(-2:KLAT_IN+3), ZDLATI_IN(-2:KLAT_IN+3)
REAL    :: ZDLON_IN(-2:KLAT_IN+3), ZDLONI_IN(-2:KLAT_IN+3)
REAL    :: ZHLAT_IN(-2:KLAT_IN+3)
REAL    :: ZHLON_IN(-2:KLAT_IN+3)
REAL    :: ZDHLAT_IN(-2:KLAT_IN+3), ZDHLATI_IN(-2:KLAT_IN+3)
REAL    :: ZG_IN   (-2:KLAT_IN+3)
INTEGER :: ILAT_IO  (0:KLAT_OUT)
REAL    :: ZLAT_OUT (0:KLAT_OUT), ZLON_OUT (1:KLAT_OUT)
REAL    :: ZDLAT_OUT(1:KLAT_OUT), ZDLON_OUT(1:KLAT_OUT)
REAL    :: ZHLAT_OUT(1:KLAT_OUT), ZHLON_OUT(1:KLAT_OUT)
REAL    :: ZG_OUT   (1:KLAT_OUT)

INTEGER :: J, JLAT, JLON, JNUM, JINX, JINY, JINYD, JOUTY,IRET
INTEGER :: JXMAX_IN, JXMAX_OUT, JYMAX_OUT, IBL
INTEGER :: JBL, IBL_OUT, JINYSTA, JINYEND, IBLATSTA_OUT(1:KLAT_OUT)
INTEGER :: IBLATEND_OUT(1:KLAT_OUT), IBLON_OUT(1:KLAT_OUT)
REAL    :: ZSUM, ZOUT, ZX, ZY, ZY2, ZDHI
REAL    :: ZC(0:3), ZDC, ZDL, ZDLL, ZDR, ZDRR, ZCC2, ZCC3
REAL    :: ZMINMOD, ZFRAC, ZCURVLIM, ZLIML, ZLIMR, ZLIM
REAL    :: ZPARITY, ZPRIMADD
REAL, ALLOCATABLE :: ZPRIMX(:,:), ZPRIMY(:,:), ZPRIM(:,:), ZFLD_IN(:)
REAL, ALLOCATABLE :: ZDIFX(:,:), ZDIFY(:,:), ZOUTPRE(:)

REAL    :: RPI, RD2R

!     ------------------------------------------------------------------
ZMINMOD(ZDC)=(0.5+SIGN(0.5,ZDC))*MIN(1.,ABS(ZDC))
ZFRAC(ZDC,ZDR)=ZDC/SIGN(ABS(ZDR)+1.E-30,ZDR)
ZCURVLIM(ZDL,ZDR)=SIGN(MIN(ABS(ZDL),ABS(ZDR)/3.),ZDL)


!     *    0. CONSTANTS AND FUNCTIONS

RPI=ACOS(-1.)
RD2R=RPI/180.
ZCC2=1.0/1536.
ZCC3=1.0/15360.

ZPARITY=1.0
IF (PRESENT(LDPAR)) THEN
  IF (.NOT.LDPAR) ZPARITY=-1.0
ENDIF

ALLOCATE(ZFLD_IN(KIN))
DO J=1,KIN
  ZFLD_IN(J)=PFLD_IN(J)
ENDDO

!     *    1. INPUT GRID AND FIELD

!*         1.1 Geometry of input grid
!              Assuming lat_(j+.5) = 0.5*(lat_(j) + lat_(j+1))

ZLAT_IN(0)      = SIGN(RPI/2.,RD2R*PLAT_IN(1))
ZLAT_IN(KLAT_IN)= SIGN(RPI/2.,RD2R*PLAT_IN(KLAT_IN))
JNUM=1
JXMAX_IN=0
DO JLAT=1,KLAT_IN
  IF (JLAT<KLAT_IN) ZLAT_IN(JLAT)=RD2R*0.5*(PLAT_IN(JLAT)+PLAT_IN(JLAT+1))
  ZDLAT_IN(JLAT)=ABS(ZLAT_IN(JLAT-1)-ZLAT_IN(JLAT))
  ILON_IN(JLAT)=KLON_IN(JLAT)
  ZDLON_IN(JLAT)=2.*RPI/ILON_IN(JLAT)
  IND_IN(JLAT)=JNUM
  JNUM=JNUM+ILON_IN(JLAT)
  JXMAX_IN=MAX(JXMAX_IN,ILON_IN(JLAT))
!              Starting longitude (must be less or equal to zero)
  ZLON_IN(JLAT)=-ZDLON_IN(JLAT)/2.
!              Scale factors
  ZHLON_IN(JLAT) = (SIN(ZLAT_IN(JLAT-1))-SIN(ZLAT_IN(JLAT)))/ZDLAT_IN(JLAT)
  ZHLAT_IN(JLAT) = 1.
  ZG_IN(JLAT)=ZHLON_IN(JLAT)*ZHLAT_IN(JLAT)*ZDLON_IN(JLAT)*ZDLAT_IN(JLAT)
ENDDO

!          1.1.1 Extension over poles
! Starting longitude for extension over the poles for even ILON_IN is 0.,
! and for odd ILON_IN shift by half a gridcell to -RPI/ILON_IN.

DO JINY = 1, 3
  ILON_IN (1-JINY)       = ILON_IN (JINY)
  ZDLAT_IN(1-JINY)       = ZDLAT_IN(JINY)
  ZDLON_IN(1-JINY)       = ZDLON_IN(JINY)
  ZHLAT_IN(1-JINY)       = ZHLAT_IN(JINY)
  ZHLON_IN(1-JINY)       = ZHLON_IN(JINY)
  ZG_IN   (1-JINY)       = ZG_IN   (JINY)
  ZLAT_IN (1-JINY)       = 2.*ZLAT_IN(0)-ZLAT_IN(JINY-1)
  ZLON_IN(1-JINY)        = ZLON_IN(JINY)&
                          &+MOD(ILON_IN(1-JINY),2)*RPI/ILON_IN(1-JINY)
  ILON_IN (KLAT_IN+JINY) = ILON_IN (KLAT_IN+1-JINY)
  ZDLAT_IN(KLAT_IN+JINY) = ZDLAT_IN(KLAT_IN+1-JINY)
  ZDLON_IN(KLAT_IN+JINY) = ZDLON_IN(KLAT_IN+1-JINY)
  ZHLAT_IN(KLAT_IN+JINY) = ZHLAT_IN(KLAT_IN+1-JINY)
  ZHLON_IN(KLAT_IN+JINY) = ZHLON_IN(KLAT_IN+1-JINY)
  ZG_IN   (KLAT_IN+JINY) = ZG_IN   (KLAT_IN+1-JINY)
  ZLAT_IN (KLAT_IN+JINY) = 2.*ZLAT_IN(KLAT_IN)-ZLAT_IN(KLAT_IN-JINY)
  ZLON_IN (KLAT_IN+JINY) = ZLON_IN (KLAT_IN+1-JINY)&
                          &+MOD(ILON_IN(KLAT_IN+JINY),2)*RPI/ILON_IN(KLAT_IN+JINY)
ENDDO

!              1.1.2 Inverses
ZDLONI_IN(:)=1./ZDLON_IN(:)
ZDLATI_IN(:)=1./ZDLAT_IN(:)
ZDHLAT_IN(:)=ZDLAT_IN(:)*ZHLAT_IN(:)
ZDHLATI_IN(:)=1./ZDHLAT_IN(:)

!*         1.2 Primitive function of input field with respect to longitude
!              Valid at grid-cell longitude boundaries.
!              Make an extension zone of +/-3 grid-cells for interpolation.
!              The extension over the poles shifts fields by RPI.

!          1.2.1 Primitive function
ALLOCATE(ZPRIMX(-2:JXMAX_IN+3,-2:KLAT_IN+3))
ALLOCATE(ZDIFX (-2:JXMAX_IN+2,-2:KLAT_IN+3))

DO JLAT=1,KLAT_IN
  ZSUM=0.
  DO JLON=1,ILON_IN(JLAT)
    ZSUM=ZSUM+ZFLD_IN(IND_IN(JLAT)-1+JLON)*ZG_IN(JLAT)
    ZPRIMX(JLON,JLAT)=ZSUM
  ENDDO
ENDDO

!          1.2.2 The extension over the poles shifts fields by RPI.

DO JINY = 1, 3
  ZSUM=0.
  DO JLON=1,ILON_IN(1-JINY)
    JINX = 1 + MOD( ILON_IN(1-JINY)/2 + JLON - 1, ILON_IN(1-JINY) )
    ZSUM=ZSUM+ZPARITY*ZFLD_IN(IND_IN(JINY)-1+JINX)*ZG_IN(1-JINY)
    ZPRIMX(JLON,1-JINY) = ZSUM
  ENDDO
  ZSUM=0.
  DO JLON=1,ILON_IN(KLAT_IN+JINY)
    JINX = 1+MOD( ILON_IN(KLAT_IN+JINY)/2 + JLON - 1, ILON_IN(KLAT_IN+JINY) )
    ZSUM=ZSUM+ZPARITY*ZFLD_IN(IND_IN(KLAT_IN+1-JINY)-1+JINX)*ZG_IN(KLAT_IN+JINY)
    ZPRIMX(JLON,KLAT_IN+JINY) = ZSUM
  ENDDO
ENDDO

!          1.2.3 Extension in x-direction

DO JINX = 1, 3
  DO JLAT=-2,KLAT_IN+3
    ZPRIMX(1-JINX            ,JLAT) =-ZPRIMX(ILON_IN(JLAT),JLAT)&
                                    &+ZPRIMX(ILON_IN(JLAT)+1-JINX,JLAT)
    ZPRIMX(ILON_IN(JLAT)+JINX,JLAT) = ZPRIMX(ILON_IN(JLAT),JLAT)&
                                    &+ZPRIMX(JINX,JLAT)
  ENDDO
ENDDO

!              1.2.5 Difference of primitive function

DO JLAT=-2,KLAT_IN+3
  DO JLON=-2,ILON_IN(JLAT)+2
    ZDIFX(JLON,JLAT)=ZPRIMX(JLON+1,JLAT)-ZPRIMX(JLON,JLAT)
  ENDDO
ENDDO



!     *    2. OUTPUT GRID AND FIELD

!*         2.1 Geometry of output grid
!              Output grid can be finer or coarser than input grid.
!              Therefore dimensioning [KLAT_IN/KLAT_OUT]+4+1
!              to allow for 4th order interpolation of primitive
!              function wrt latitude at top and bottom of grid-cell
!              boundaries (output latitudes).

!          2.1.1 Grid blocks with constant longitude spacing

IBL_OUT = 1
IBLATSTA_OUT(1) = 1
IBLON_OUT(1) = KLON_OUT(1)
DO JLAT=2,KLAT_OUT
  IF((KLON_OUT(JLAT)/=KLON_OUT(JLAT-1)) .OR. (JLAT == KLAT_OUT/2))THEN
    IBL_OUT = IBL_OUT + 1
    IBLATEND_OUT(IBL_OUT-1) = JLAT-1
    IBLATSTA_OUT(IBL_OUT) = JLAT
    IBLON_OUT(IBL_OUT) = KLON_OUT(JLAT)
  ENDIF
ENDDO
IBLATEND_OUT(IBL_OUT)=KLAT_OUT

!          2.1.2 Latitudes and scale factors

ZLAT_OUT(0)       = SIGN(RPI/2.,RD2R*PLAT_OUT(1))
ZLAT_OUT(KLAT_OUT)= SIGN(RPI/2.,RD2R*PLAT_OUT(KLAT_OUT))
DO JLAT=1,KLAT_OUT
  IF (JLAT<KLAT_OUT) ZLAT_OUT(JLAT)=RD2R*0.5*(PLAT_OUT(JLAT)+PLAT_OUT(JLAT+1))
  ZDLAT_OUT(JLAT)=ABS(ZLAT_OUT(JLAT-1)-ZLAT_OUT(JLAT))
  ZDLON_OUT(JLAT)=2.*RPI/KLON_OUT(JLAT)
!              Starting longitude (must be less or equal to zero)
  ZLON_OUT(JLAT)=-ZDLON_OUT(JLAT)/2.
!              Scale factors
  ZHLON_OUT(JLAT) = ABS(SIN(ZLAT_OUT(JLAT-1))-SIN(ZLAT_OUT(JLAT)))/ZDLAT_OUT(JLAT)
  ZHLAT_OUT(JLAT) = 1.
  ZG_OUT(JLAT) = ZHLON_OUT(JLAT)*ZHLAT_OUT(JLAT)*ZDLON_OUT(JLAT)*ZDLAT_OUT(JLAT)
ENDDO

!          2.1.3 Closest input row above each output row

ILAT_IO(0)=0
DO JLAT=1,KLAT_OUT
  ILAT_IO(JLAT)=ILAT_IO(JLAT-1)
  DO WHILE (ZLAT_IN(ILAT_IO(JLAT)) > ZLAT_OUT(JLAT))
    ILAT_IO(JLAT)=ILAT_IO(JLAT)+1
  ENDDO
  ILAT_IO(JLAT)=ILAT_IO(JLAT)-1
ENDDO
JXMAX_OUT=0
JYMAX_OUT=0
DO JBL=1, IBL_OUT
  JXMAX_OUT = MAX(JXMAX_OUT,IBLON_OUT(JBL))
  JYMAX_OUT = MAX(JYMAX_OUT,IBLATEND_OUT(JBL)-IBLATSTA_OUT(JBL)+1)
ENDDO

  
!*         2.2 Interpolation of primitive function

JNUM=0
ALLOCATE(ZPRIM(0:JXMAX_OUT,-3:3+KLAT_IN))
ALLOCATE(ZDIFY(1:JXMAX_OUT,-3:3+KLAT_IN))
ALLOCATE(ZPRIMY(0:JXMAX_OUT,0:JYMAX_OUT))
ALLOCATE(ZOUTPRE(0:JXMAX_OUT))
DO JBL=1, IBL_OUT

!          2.2.1 Interpolate PF of input field with respect to longitude
!                to output grid-cell longitude boundaries.
!                (output longitude / input latitude)

  JINYSTA=ILAT_IO(IBLATSTA_OUT(JBL)-1)
  JINYEND=ILAT_IO(IBLATEND_OUT(JBL))
  JINYD=JINYEND-JINYSTA
  DO JLON=0,IBLON_OUT(JBL)
    ZPRIM(JLON,-3)=0.
    ZOUTPRE(JLON)=ZLON_OUT(IBLATSTA_OUT(JBL))+JLON*ZDLON_OUT(IBLATSTA_OUT(JBL))
  ENDDO
  DO JINY=JINYSTA-2,JINYEND+3
    DO JLON=0,IBLON_OUT(JBL)
      ZOUT =  ZOUTPRE(JLON)-ZLON_IN(JINY)
      ZPRIMADD=0.
      IF (ZOUT<0.) THEN
        ZOUT    = ZOUT+ILON_IN(JINY)*ZDLON_IN(JINY)
        ZPRIMADD=-ZPRIMX(ILON_IN(JINY),JINY)
      ENDIF
      IF (ZOUT>ILON_IN(JINY)*ZDLON_IN(JINY)) THEN
        ZOUT    = ZOUT-ILON_IN(JINY)*ZDLON_IN(JINY)
        ZPRIMADD= ZPRIMX(ILON_IN(JINY),JINY)
      ENDIF
      JINX=INT(ZOUT*ZDLONI_IN(JINY))
      ZX=-1.+2.*(ZOUT-JINX*ZDLON_IN(JINY))*ZDLONI_IN(JINY)
!          2.2.1.1 Basic coefficients
      ZDLL=ZDIFX(JINX-2,JINY)
      ZDL =ZDIFX(JINX-1,JINY)
      ZDC =ZDIFX(JINX  ,JINY)
      ZDR =ZDIFX(JINX+1,JINY)
      ZDRR=ZDIFX(JINX+2,JINY)
      ZC(0) = (ZPRIMX(JINX+1,JINY)+ZPRIMX(JINX  ,JINY))*.5
      ZC(1) = ZDC*.5
!          2.2.1.2 Economization of the power series (5th to 3rd order)
      ZC(2)=(130.*(ZDR-ZDL)-17.*(ZDRR-ZDLL))*ZCC2
      ZC(3)=(452.*(ZDR-2.*ZDC+ZDL)-33.*(ZDRR-2.*ZDC+ZDLL))*ZCC3
!          2.2.1.3 Limiting
      ZC(3) = ZCURVLIM(ZC(3),ZC(2))
      ZLIMR = ZMINMOD( ZFRAC( (ZDR-ZDC)*.5,       2.*( ZC(2)+ZC(3) ) ) )
      ZLIML = ZMINMOD( ZFRAC( (ZDL-ZDC)*.5, ZLIMR*2.*(-ZC(2)+ZC(3) ) ) )
      ZLIM  = ZLIMR*ZLIML
      ZC(2) = ZLIM*ZC(2)
      ZC(3) = ZLIM*ZC(3)
!          2.2.1.5 Evaluation
      ZPRIM(JLON,JINY-JINYSTA)= ZC(0)+ZC(1)*ZX   &
                 & +(ZX*ZX-1.)*(ZC(2)+ZC(3)*ZX)   &
                 & +ZPRIMADD
    ENDDO
  ENDDO

!          2.2.2  Differentiate wrt longitude

  DO JINY=-2,3+JINYD
    ZDHI = 1./(ZDLON_OUT(IBLATSTA_OUT(JBL))*ZHLON_IN(JINYSTA+JINY))
    DO JLON=IBLON_OUT(JBL),1,-1
      ZPRIM(JLON,JINY) = (ZPRIM(JLON,JINY)-ZPRIM(JLON-1,JINY))*ZDHI
    ENDDO
  ENDDO

!          2.2.3 PF wrt latitude and scalefactor*cellarea (G=h_y, dA=dy ==> G*dA)
!                (output longitude / input latitude)

  DO JINY=-2,3+JINYD
    DO JLON=1, IBLON_OUT(JBL)
      ZPRIM(JLON,JINY)= ZPRIM(JLON,JINY-1)+ZPRIM(JLON,JINY)
    ENDDO
  ENDDO
  DO JINY=-2,2+JINYD
    DO JLON=1, IBLON_OUT(JBL)
      ZDIFY(JLON,JINY) = (ZPRIM(JLON,JINY+1)-ZPRIM(JLON,JINY))*ZDHLATI_IN(JINYSTA+JINY+1)
    ENDDO
  ENDDO

!          2.2.4 Interpolation PF to output latitude cell boundaries

  DO JOUTY = 0, IBLATEND_OUT(JBL)-IBLATSTA_OUT(JBL)+1
    JLAT =         JOUTY+IBLATSTA_OUT(JBL)-1
    JINY = ILAT_IO(JOUTY+IBLATSTA_OUT(JBL)-1)-JINYSTA
    ZY=-1.+2.*(-ZLAT_OUT(JLAT)+ZLAT_IN(JINYSTA+JINY))*ZDLATI_IN(JINYSTA+JINY+1)
    ZY2 = (ZY*ZY-1.)
    DO JLON=1, IBLON_OUT(JBL)
!          2.2.4.1 Basic coefficients
      ZDLL=ZDIFY(JLON,JINY-2)
      ZDL =ZDIFY(JLON,JINY-1)
      ZDC =ZDIFY(JLON,JINY  )
      ZDR =ZDIFY(JLON,JINY+1)
      ZDRR=ZDIFY(JLON,JINY+2)
      ZC(0) = (ZPRIM(JLON,JINY+1)+ZPRIM(JLON,JINY))*.5
      ZC(1) = ZDC*.5
!          2.2.1.2 Economization of the power series (5th to 3rd order)
      ZC(2)=(130.*(ZDR-ZDL)-17.*(ZDRR-ZDLL))*ZCC2
      ZC(3)=(452.*(ZDR-2.*ZDC+ZDL)-33.*(ZDRR-2.*ZDC+ZDLL))*ZCC3
!          2.2.1.3 Limiting
      ZC(3) = ZCURVLIM(ZC(3),ZC(2))
      ZLIMR = ZMINMOD( ZFRAC( (ZDR-ZDC)*.5,       2.*( ZC(2)+ZC(3) ) ) )
      ZLIML = ZMINMOD( ZFRAC( (ZDL-ZDC)*.5, ZLIMR*2.*(-ZC(2)+ZC(3) ) ) )
      ZLIM  = ZLIMR*ZLIML
      ZC(2) = ZLIM*ZC(2)
      ZC(3) = ZLIM*ZC(3)
!          2.2.5.4 Back to real coefficients
      ZC(1:3) = ZDHLAT_IN(JINYSTA+JINY+1)*ZC(1:3)
!          2.2.4.5 Evaluation
      ZPRIMY(JLON,JOUTY)= ZC(0)+ZC(1)*ZY+ZY2*(ZC(2)+ZC(3)*ZY)
    ENDDO
  ENDDO

!          2.2.5 Take difference of primitive function to get output field
!                (output longitude / output latitude)

  DO JOUTY = 1, IBLATEND_OUT(JBL)-IBLATSTA_OUT(JBL)+1
    JLAT = JOUTY+IBLATSTA_OUT(JBL)-1
    ZDHI = 1./(ZDLAT_OUT(JLAT)*ZHLAT_OUT(JLAT))
    DO JLON=1, IBLON_OUT(JBL)
      JNUM=JNUM+1
      PFLD_OUT(JNUM) =  (ZPRIMY(JLON,JOUTY)-ZPRIMY(JLON,JOUTY-1))*ZDHI
    ENDDO
  ENDDO
ENDDO
IF (ALLOCATED(ZPRIM ))  DEALLOCATE(ZPRIM )
IF (ALLOCATED(ZPRIMY))  DEALLOCATE(ZPRIMY)
IF (ALLOCATED(ZDIFY))   DEALLOCATE(ZDIFY)
IF (ALLOCATED(ZOUTPRE)) DEALLOCATE(ZOUTPRE)

IF (ALLOCATED(ZPRIMX)) DEALLOCATE(ZPRIMX)
IF (ALLOCATED(ZFLD_IN)) DEALLOCATE(ZFLD_IN)
IF (ALLOCATED(ZDIFX)) DEALLOCATE(ZDIFX)


END SUBROUTINE CONSINT
