/*global Components: false */
/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

/* global gAccount: false, gIdentity: false, onOk: false, smimeOnAcceptEditor: false */

"use strict";

Components.utils.import("resource://enigmail/funcs.jsm"); /* global EnigmailFuncs: false */
Components.utils.import("resource://enigmail/locale.jsm"); /* global EnigmailLocale: false */
Components.utils.import("resource://enigmail/log.jsm"); /* global EnigmailLog: false */
Components.utils.import("resource://enigmail/windows.jsm"); /* global EnigmailWindows: false */
Components.utils.import("resource://enigmail/dialog.jsm"); /* global EnigmailDialog: false */
Components.utils.import("resource://enigmail/pEpAdapter.jsm"); /* global EnigmailPEPAdapter: false */

if (!Enigmail) var Enigmail = {};

Enigmail.edit = {
  account: null,
  identity: null,
  enablePgp: null,
  pgpKeyMode: null,
  pgpKeyId: null,
  cryptoChoicesEnabled: null,
  signingPolicy: null, // account specific: by default sign
  encryptionPolicy: null, // account specific: by default encrypt
  pgpMimeMode: null, // account specific: by default pgp/mime
  pgpSignPlainPolicy: null,
  pgpSignEncPolicy: null,
  autoEncryptDrafts: null,
  advancedSettings: null,
  usingPep: null,

  onInit: function() {
    EnigmailLog.DEBUG("enigmailEditIdentity.js: onInit()\n");

    // initialize all of our elements based on the current identity values....
    EnigmailFuncs.collapseAdvanced(document.getElementById("enigmail_PrefsBox"), "hidden");

    this.enablePgp = document.getElementById("enigmail_enablePgp");
    this.pgpKeyMode = document.getElementById("enigmail_pgpKeyMode");
    this.pgpKeyId = document.getElementById("enigmail_identity.pgpkeyId");
    this.signingPolicy = document.getElementById("enigmail_sign_ifPossible");
    this.encryptionPolicy = document.getElementById("enigmail_encrypt_ifPossible");
    this.pgpMimeMode = document.getElementById("enigmail_pgpMimeMode");
    this.pgpSignEncPolicy = document.getElementById("enigmail_sign_encrypted");
    this.pgpSignPlainPolicy = document.getElementById("enigmail_sign_notEncrypted");
    this.autoEncryptDrafts = document.getElementById("enigmail_autoEncryptDrafts");
    this.mimePreferOpenPGP = document.getElementById("enigmail_mimePreferOpenPGP");
    this.enableAc = document.getElementById("enigmail_enableAutocrypt");
    this.acPreferEncrypt = document.getElementById("enigmail_acPreferEncrypt");
    this.isSingleIdEditor = document.getElementById("enigmail_singleId") ? true : false;

    if (this.isSingleIdEditor) {
      let acTab = document.getElementById("enigmail_autocryptTab");
      acTab.setAttribute("collapsed", "true");
    }

    this.usingPep = EnigmailPEPAdapter.usingPep();

    if (this.identity) {
      this.enablePgp.checked = this.identity.getBoolAttribute("enablePgp");
      this.cryptoChoicesEnabled = this.enablePgp.checked;

      var selectedItemId = null;
      var keyPolicy = this.identity.getIntAttribute("pgpKeyMode");
      switch (keyPolicy) {
        case 1:
          selectedItemId = 'enigmail_keymode_usePgpkeyId';
          break;
        default:
          selectedItemId = 'enigmail_keymode_useFromAddress';
          break;
      }
      this.pgpKeyMode.selectedItem = document.getElementById(selectedItemId);

      var mimePolicy = this.identity.getIntAttribute("mimePreferOpenPGP");
      switch (mimePolicy) {
        case 1:
          selectedItemId = "enigmail_mime_preferEnigmail";
          break;
        default:
          selectedItemId = "enigmail_mime_preferSMime";
          break;
      }
      this.mimePreferOpenPGP.selectedItem = document.getElementById(selectedItemId);

      this.pgpKeyId.value = this.identity.getCharAttribute("pgpkeyId");
      EnigmailFuncs.getSignMsg(this.identity);
      this.signingPolicy.checked = (this.identity.getIntAttribute("defaultSigningPolicy") > 0);
      this.encryptionPolicy.checked = (this.identity.getIntAttribute("defaultEncryptionPolicy") > 0);
      this.pgpMimeMode.checked = this.identity.getBoolAttribute("pgpMimeMode");
      this.pgpSignEncPolicy.checked = this.identity.getBoolAttribute("pgpSignEncrypted");
      this.pgpSignPlainPolicy.checked = this.identity.getBoolAttribute("pgpSignPlain");
      this.autoEncryptDrafts.checked = this.identity.getBoolAttribute("autoEncryptDrafts");
      this.advancedSettings = {
        openPgpHeaderMode: this.identity.getIntAttribute("openPgpHeaderMode"),
        openPgpUrlName: this.identity.getCharAttribute("openPgpUrlName"),
        attachPgpKey: this.identity.getBoolAttribute("attachPgpKey")
      };

    }
    else {
      this.enablePgp.checked = false;
      this.cryptoChoicesEnabled = false;
      this.pgpMimeMode.checked = true;
      this.advancedSettings = {
        openPgpHeaderMode: 0,
        openPgpUrlName: "",
        attachPgpKey: false
      };
    }

    if (this.account) {
      this.enableAc.checked = this.account.incomingServer.getBoolValue("enableAutocrypt");
      this.acPreferEncrypt.checked = (this.account.incomingServer.getIntValue("acPreferEncrypt") > 0);
    }


    // Disable all locked elements on the panel
    //onLockPreference();
    this.enableAllPrefs();
  },

  onLoadEditor: function() {
    EnigmailLog.DEBUG("enigmailEditIdentity.js: onLoadEditor()\n");
    if (typeof(gAccount) == "object") {
      this.account = gAccount;
      this.identity = gIdentity;
    }
    else if ("arguments" in window) {
      this.identity = window.arguments[0].identity;
      this.account = window.arguments[0].account;
    }

    if (this.identity) {
      var idLabel = EnigmailLocale.getString("identityName", [this.identity.identityName]);
      document.getElementById("enigmail_identityName").value = idLabel;
    }

    var dlg = document.getElementsByTagName("dialog")[0];
    dlg.setAttribute("ondialogaccept", "return Enigmail.edit.onAcceptEditor();");

    this.onInit();
  },

  onAcceptEditor: function() {
    EnigmailLog.DEBUG("enigmailEditIdentity.js: onAcceptEditor()\n");
    try {
      if (onOk() === false) {
        return false;
      }
    }
    catch (ex) {}
    this.onSave();
    if (typeof(smimeOnAcceptEditor) == "function") {
      return smimeOnAcceptEditor();
    }
    else
      return true;
  },

  onSave: function() {
    EnigmailLog.DEBUG("enigmailEditIdentity.js: onSave()\n");
    try {
      if (!this.identity) {
        this.identity = gIdentity;
      }
      this.identity.setBoolAttribute("enablePgp", this.enablePgp.checked);

      if (this.enablePgp.checked) {
        // PGP is enabled
        this.identity.setIntAttribute("pgpKeyMode", this.pgpKeyMode.selectedItem.value);
        this.identity.setIntAttribute("mimePreferOpenPGP", this.mimePreferOpenPGP.selectedItem.value);
        this.identity.setCharAttribute("pgpkeyId", this.pgpKeyId.value);
        this.identity.setIntAttribute("defaultSigningPolicy", (this.signingPolicy.checked ? 1 : 0));
        this.identity.setIntAttribute("defaultEncryptionPolicy", (this.encryptionPolicy.checked ? 1 : 0));
        this.identity.setBoolAttribute("pgpMimeMode", this.pgpMimeMode.checked);
        this.identity.setBoolAttribute("pgpSignEncrypted", this.pgpSignEncPolicy.checked);
        this.identity.setBoolAttribute("pgpSignPlain", this.pgpSignPlainPolicy.checked);
        this.identity.setBoolAttribute("autoEncryptDrafts", this.autoEncryptDrafts.checked);
        this.identity.setIntAttribute("openPgpHeaderMode", this.advancedSettings.openPgpHeaderMode);
        this.identity.setCharAttribute("openPgpUrlName", this.advancedSettings.openPgpUrlName);
        this.identity.setBoolAttribute("attachPgpKey", this.advancedSettings.attachPgpKey);
      }

      let usingPep = EnigmailPEPAdapter.usingPep();

      if (usingPep !== this.usingPep) {
        EnigmailPEPAdapter.handleJuniorModeChange();
        this.usingPep = usingPep;
      }

      if (usingPep) {
        EnigmailPEPAdapter.setOwnIdentities(0);
      }

      if (!this.isSingleIdEditor) {
        this.account.incomingServer.setBoolValue("enableAutocrypt", this.enableAc.checked);
        this.account.incomingServer.setIntValue("acPreferEncrypt", this.acPreferEncrypt.checked ? 1 : 0);
      }
    }
    catch (ex) {
      EnigmailLog.ERROR(`enigmailEditIdentity.js: onSave: error: ${ex.message}\n`);
    }
  },

  toggleEnable: function() {
    let newCryptoEnabled = (!this.cryptoChoicesEnabled);

    if (this.usingPep && newCryptoEnabled) {
      if (!EnigmailDialog.confirmDlg(window, EnigmailLocale.getString("switchPepMode"), EnigmailLocale.getString("enableEnigmail")))
        return;
    }

    this.cryptoChoicesEnabled = newCryptoEnabled;
    this.enableAllPrefs();
  },

  enableAllPrefs: function() {
    var elem = document.getElementById("enigmail_bcEnablePgp");
    if (this.cryptoChoicesEnabled) {
      if (elem) elem.removeAttribute("disabled");
    }
    else {
      if (elem) elem.setAttribute("disabled", "true");
    }

    this.enableKeySel(this.cryptoChoicesEnabled && (this.pgpKeyMode.value == 1));
    this.enableAcSettings();
  },

  enableKeySel: function(enable) {
    if (enable) {
      document.getElementById("enigmail_bcUseKeyId").removeAttribute("disabled");
    }
    else {
      document.getElementById("enigmail_bcUseKeyId").setAttribute("disabled", "true");
    }
  },

  enableAcSettings: function() {
    if (this.cryptoChoicesEnabled && this.enableAc.checked) {
      this.acPreferEncrypt.removeAttribute("disabled");
    }
    else {
      this.acPreferEncrypt.setAttribute("disabled", "true");
    }
  },

  handleClick: function(event) {
    if (event.target.hasAttribute("href")) {
      EnigmailWindows.openMailTab(event.target.getAttribute("href"));
    }
  },

  selectKeyId: function() {
    var resultObj = {};
    var inputObj = {};
    inputObj.dialogHeader = EnigmailLocale.getString("encryptKeyHeader");
    inputObj.options = "single,hidexpired,private,nosending";
    var button = document.getElementById("enigmail_selectPgpKey");
    var label = button.getAttribute("label");
    inputObj.options += ",sendlabel=" + label;
    inputObj.options += ",";

    window.openDialog("chrome://enigmail/content/enigmailKeySelection.xul", "", "dialog,modal,centerscreen,resizable", inputObj, resultObj);
    try {
      if (resultObj.cancelled) return;
      var selKey = resultObj.userList[0];
      //selKey = "0x"+selKey.substring(10,18);
      this.pgpKeyId.value = selKey;
    }
    catch (ex) {
      // cancel pressed -> don't send mail
      return;
    }
  },

  advancedIdentitySettings: function() {
    var inputObj = {
      identitySettings: this.advancedSettings,
      pgpKeyMode: this.pgpKeyMode.selectedItem.value
    };
    window.openDialog("chrome://enigmail/content/enigmailAdvancedIdentityDlg.xul", "", "dialog,modal,centerscreen", inputObj);
  }
};

window.addEventListener("load-enigmail", Enigmail.edit.onLoadEditor.bind(Enigmail.edit), false);
