/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.rust.options.cargo;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.nio.file.Path;
import java.nio.file.Paths;
import javax.swing.JFileChooser;
import javax.swing.SwingWorker;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.filechooser.FileFilter;
import org.netbeans.modules.rust.options.impl.CargoOptionsImpl;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;

final class CargoPanel extends javax.swing.JPanel implements DocumentListener {

    private final CargoOptionsPanelController controller;
    private JFileChooser fileChooser;
    private SwingWorker<String, String> versionWorker;

    CargoPanel(CargoOptionsPanelController controller) {
        this.controller = controller;
        initComponents();
        txtCargoPath.getDocument().addDocumentListener(this);
    }

    private SwingWorker<String, String> newWorker() {
        return new SwingWorker<String, String>() {
            @Override
            protected String doInBackground() throws Exception {
                String cargo = txtCargoPath.getText();
                return getVersionOf(cargo);
            }

            @Override
            protected void done() {
                versionWorker = null;
                String version;
                try {
                    version = get();
                    lblCargoVersion.setText(version);
                } catch (Exception ex) {
                    lblCargoVersion.setText("");
                }
            }
        };
    }

    private String getVersionOf(String aPossibleCargoPath) throws Exception {
        File cargo = Paths.get(aPossibleCargoPath).toFile();
        if (cargo.isFile() && cargo.canExecute()) {
            return getVersionOf(cargo);
        }
        return null;
    }

    private String getVersionOf(File cargo) throws Exception {
        Process p = Runtime.getRuntime().exec(new String[]{cargo.getAbsolutePath(), "-V"});
        int ok = p.waitFor();
        if (ok == 0) {
            String version = null;
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(p.getInputStream()))) {
                do {
                    String line = reader.readLine();
                    if (line == null) {
                        break;
                    }
                    version = line;
                } while (true);
            }
            p.destroy();
            return version;
        }
        return null;
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        lblCargoLocation = new javax.swing.JLabel();
        txtCargoPath = new javax.swing.JTextField();
        cmdBrowse = new javax.swing.JButton();
        cmdGetVersion = new javax.swing.JButton();
        lblCargoVersion = new javax.swing.JLabel();
        jPanel1 = new javax.swing.JPanel();

        setLayout(new java.awt.GridBagLayout());

        lblCargoLocation.setLabelFor(txtCargoPath);
        org.openide.awt.Mnemonics.setLocalizedText(lblCargoLocation, org.openide.util.NbBundle.getMessage(CargoPanel.class, "CargoPanel.lblCargoLocation.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 8, 8);
        add(lblCargoLocation, gridBagConstraints);

        txtCargoPath.setText(org.openide.util.NbBundle.getMessage(CargoPanel.class, "CargoPanel.txtCargoPath.text")); // NOI18N
        txtCargoPath.setToolTipText(org.openide.util.NbBundle.getMessage(CargoPanel.class, "CargoPanel.txtCargoPath.toolTipText")); // NOI18N
        txtCargoPath.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                txtCargoPathActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.BASELINE;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 8, 0, 0);
        add(txtCargoPath, gridBagConstraints);

        org.openide.awt.Mnemonics.setLocalizedText(cmdBrowse, org.openide.util.NbBundle.getMessage(CargoPanel.class, "CargoPanel.cmdBrowse.text")); // NOI18N
        cmdBrowse.setToolTipText(org.openide.util.NbBundle.getMessage(CargoPanel.class, "CargoPanel.cmdBrowse.toolTipText")); // NOI18N
        cmdBrowse.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cmdBrowseActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.BASELINE;
        gridBagConstraints.insets = new java.awt.Insets(0, 8, 0, 0);
        add(cmdBrowse, gridBagConstraints);

        org.openide.awt.Mnemonics.setLocalizedText(cmdGetVersion, org.openide.util.NbBundle.getMessage(CargoPanel.class, "CargoPanel.cmdGetVersion.text")); // NOI18N
        cmdGetVersion.setEnabled(false);
        cmdGetVersion.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cmdGetVersionActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(0, 8, 0, 8);
        add(cmdGetVersion, gridBagConstraints);

        lblCargoVersion.setFont(lblCargoVersion.getFont().deriveFont(lblCargoVersion.getFont().getStyle() | java.awt.Font.BOLD));
        org.openide.awt.Mnemonics.setLocalizedText(lblCargoVersion, org.openide.util.NbBundle.getMessage(CargoPanel.class, "CargoPanel.lblCargoVersion.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 8, 8);
        add(lblCargoVersion, gridBagConstraints);

        jPanel1.setPreferredSize(new java.awt.Dimension(8, 8));

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 746, Short.MAX_VALUE)
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 799, Short.MAX_VALUE)
        );

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(jPanel1, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void txtCargoPathActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_txtCargoPathActionPerformed
        controller.changed();
    }//GEN-LAST:event_txtCargoPathActionPerformed

    private void cmdBrowseActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cmdBrowseActionPerformed

        showFileChooser();

    }//GEN-LAST:event_cmdBrowseActionPerformed

    private void cmdGetVersionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cmdGetVersionActionPerformed
        if (versionWorker == null) {
            versionWorker = newWorker();
            versionWorker.execute();
        }
    }//GEN-LAST:event_cmdGetVersionActionPerformed

    @NbBundle.Messages({
        "TXT_EXECUTABLE=Executable files"
    })
    void showFileChooser() {
        if (fileChooser == null) {
            String cargoName = "cargo" + (Utilities.isWindows() ? ".exe" : ""); // NOI18N
            fileChooser = new JFileChooser(System.getProperty("user.home")); // NOI18N
            fileChooser.setFileFilter(new FileFilter() {
                @Override
                public boolean accept(File f) {
                    boolean valid = f.isDirectory();
                    valid = valid
                            || (f.isFile() && f.exists() && f.canExecute() && cargoName.equals(f.getName()));
                    return valid;
                }

                @Override
                public String getDescription() {
                    return Bundle.TXT_EXECUTABLE();
                }
            });
            fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
            fileChooser.setFileHidingEnabled(false);
        }
        fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        int result = fileChooser.showDialog(this, null);
        if (result == JFileChooser.APPROVE_OPTION) {
            txtCargoPath.setText(fileChooser.getSelectedFile().getAbsolutePath());
        }
    }

    void load() {
        Path cargoLocation = CargoOptionsImpl.getCargoLocation(true);
        txtCargoPath.setText(cargoLocation == null ? "" : cargoLocation.toString());
    }

    void store() {
        CargoOptionsImpl.setCargoLocation(txtCargoPath.getText());
        Path cargoLocation = CargoOptionsImpl.getCargoLocation(true);
        txtCargoPath.setText(cargoLocation == null ? "" : cargoLocation.toString());
    }

    boolean valid() {
        File cargo = new File(txtCargoPath.getText());
        return cargo.exists() && cargo.isFile() && cargo.canExecute();
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton cmdBrowse;
    private javax.swing.JButton cmdGetVersion;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JLabel lblCargoLocation;
    private javax.swing.JLabel lblCargoVersion;
    private javax.swing.JTextField txtCargoPath;
    // End of variables declaration//GEN-END:variables

    @Override
    public void insertUpdate(DocumentEvent e) {
        documentChanged(e);
    }

    @Override
    public void removeUpdate(DocumentEvent e) {
        documentChanged(e);
    }

    @Override
    public void changedUpdate(DocumentEvent e) {
        documentChanged(e);
    }

    private void documentChanged(DocumentEvent e) {
        controller.changed();
        File file = Paths.get(txtCargoPath.getText()).toFile();
        boolean executable = file.exists() && file.canExecute();
        cmdGetVersion.setEnabled(executable);
    }

}
