/*
    libodsstream is a library to read and write ODS documents as streams
    Copyright (C) 2013  Olivier Langella <Olivier.Langella@moulon.inra.fr>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <QFileInfo>
#include <QDebug>
#include "tsvdirectorywriter.h"
#include "odsexception.h"


TsvDirectoryWriter::TsvDirectoryWriter()
{
}

TsvDirectoryWriter::TsvDirectoryWriter(const QDir & directory): _directory(directory)
{
    if (_directory.exists()) {
    } else {
        if (!_directory.mkdir(_directory.absolutePath())) {
            throw OdsException(QString("unable to create output directory %1").arg(_directory.absolutePath()));
        }
    }
}

TsvDirectoryWriter::~TsvDirectoryWriter()
{
    close();
}

void TsvDirectoryWriter::close() {
    if (_p_otxtstream != nullptr) {
        *_p_otxtstream << _end_of_line;
        _p_otxtstream->flush();
        //_p_otxtstream->close();
        delete (_p_otxtstream);
        _p_otxtstream = nullptr;
    }
    if (_p_ofile != nullptr) {
        _p_ofile->flush();
        _p_ofile->close();
        delete (_p_ofile);
        _p_ofile = nullptr;
    }
}

void TsvDirectoryWriter::writeSheet(const QString & sheetName) {
    close();
    // qDebug() << " TsvDirectoryWriter::writeSheet " <<  _directory.absolutePath();
    _p_ofile = new QFile( QString(_directory.absolutePath()).append("/").append(sheetName).append(this->_file_extension) );
    // qDebug() << " TsvDirectoryWriter::writeSheet " << QFileInfo(*_p_ofile).absoluteFilePath();
    if (_p_ofile->open(QIODevice::WriteOnly) )
    {
        _p_otxtstream = new  QTextStream ( _p_ofile);
    }
    else {
        throw OdsException(QString("unable to write into file %1").arg(QFileInfo(*_p_ofile).absoluteFilePath()));
    }
    _tableRowStart = true;
}


void TsvDirectoryWriter::ensureSheet() {
    if (_p_otxtstream == nullptr) {
        writeSheet("default");
    }
}

void TsvDirectoryWriter::writeLine() {
    ensureSheet();
    *_p_otxtstream << _end_of_line;
    _tableRowStart = true;
}
void TsvDirectoryWriter::writeCell(const char * cstr) {
    writeCell( QString(cstr));
}
void TsvDirectoryWriter::writeCell(const QString & text) {
    ensureSheet();
    if (!_tableRowStart) {
        *_p_otxtstream << _separator;
    }
    _tableRowStart = false;
    *_p_otxtstream << text;
}
void TsvDirectoryWriter::writeEmptyCell()  {
    ensureSheet();
    if (!_tableRowStart) {
        *_p_otxtstream << _separator;
    }
    _tableRowStart = false;
}
void TsvDirectoryWriter::writeCell(int num)  {
    writeCell( QString::number(num, 'g', 10));
}
void TsvDirectoryWriter::writeCell(float num) {
    writeCell( QString::number(num, 'g', numFloatPrecision));
}
void TsvDirectoryWriter::writeCell(double num) {
    writeCell( QString::number(num, 'g', numFloatPrecision));
}

void TsvDirectoryWriter::writeCellPercentage(double value) {
    writeCell(value);
}
void TsvDirectoryWriter::writeCell(bool isOk) {
    if (isOk) {
        writeCell( QString("T"));
    }
    else {
        writeCell( QString("F"));
    }
}
void TsvDirectoryWriter::writeCell(const QDate & date)  {
    writeCell( date.toString("yyyy-MM-dd"));
}
void TsvDirectoryWriter::writeCell(const QDateTime & date)  {
    writeCell( date.toString("yyyy-MM-ddThh:mm:ss"));
}
void TsvDirectoryWriter::writeCell(const QUrl & url, const QString & text) {
    writeCell( text);
}
