#! /bin/bash
#
# This script contains all the fixups run when systemd package is installed or
# updated.
#

#
# /etc/machine-id might have been created writeable incorrectly (boo#1092269).
#
# Note: run at each package update.
#
fix_machine_id_perms() {
	if [ "$(stat -c%a /etc/machine-id)" != 444 ]; then
		echo "Incorrect file mode bits for /etc/machine-id which should be 0444, fixing..."
		chmod 444 /etc/machine-id
	fi
}

#
# v228 wrongly set world writable suid root permissions on timestamp files used
# by permanent timers. Fix the timestamps that might have been created by the
# affected versions of systemd (bsc#1020601).
#
# Note: run at each package update.
#
fix_bsc_1020601() {
	for stamp in $(ls /var/lib/systemd/timers/stamp-*.timer 2>/dev/null); do
		chmod 0644 $stamp
	done

	# Same for user lingering created by logind.
	for username in $(ls /var/lib/systemd/linger/* 2>/dev/null); do
		chmod 0644 $username
	done
}

#
# Due to the fact that DynamicUser= was turned ON during v235 and then switched
# back to off in v240, /var/lib/systemd/timesync might be a symlink pointing to
# /var/lib/private/systemd/timesync, which is inaccessible for systemd-timesync
# user as /var/lib/private is 0700 root:root, see
# https://github.com/systemd/systemd/issues/11329 for details.
#
# Note: only TW might be affected by this bug.
# Note: run at each package update.
#
fix_issue_11329() {
	if [ -L /var/lib/systemd/timesync ]; then
		rm /var/lib/systemd/timesync
		mv /var/lib/private/systemd/timesync /var/lib/systemd/timesync
	fi
}

#
# We don't ship after-local.service anymore however as a courtesy we install a
# copy in /etc for users who are relying on it.
#
# Note: should run only once since it is conditionalized on the presence of
# %{_unitdir}/after-local.service
#
drop_after_local_support() {
	if [ -x /etc/init.d/after.local ] &&
	   [ -f /usr/lib/systemd/system/after-local.service ]; then
		echo "after-local.service is no more provided by systemd but a copy has been installed in /etc"
		cp /usr/lib/systemd/system/after-local.service /etc/systemd/system/
		ln -s ../after-local.service /etc/systemd/system/multi-user.target.wants/after-local.service
	fi
}

#
# We have stopped shipping the main config files in /etc but we don't try to
# clean them up automatically as it can have unexepected side effects
# (bsc#1226415). Instead we simply suggest users to convert them (if they exist)
# into drop-ins.
#
# Note: run at each package update
#
check_config_files () {
	config_files=(systemd/journald.conf systemd/logind.conf systemd/system.conf systemd/user.conf
		      systemd/pstore.conf systemd/sleep.conf systemd/timesyncd.conf systemd/coredump.conf
		      systemd/journal-remote.conf systemd/journal-upload.conf systemd/networkd.conf
		      systemd/resolved.conf systemd/oomd.conf udev/iocost.conf udev/udev.conf)

	for f in ${config_files[*]}; do
		[ -e /etc/$f ] || continue

		cat >&2 <<EOF
Main configuration files are deprecated in favor of drop-ins.
Hence, we suggest that you remove /etc/$f if it doesn't contain any customization, or convert it into drop-in otherwise.
For more details, please visit https://en.opensuse.org/Systemd#Configuration.
EOF
	done
}

r=0
fix_machine_id_perms || r=1
fix_bsc_1020601 || r=1
fix_issue_11329 || r=1
drop_after_local_support || r=1

exit $r
