/*
 * Copyright (C) 2022 Andreas Steffen, strongSec GmbH
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include "test_suite.h"

#include <credentials/certificates/x509.h>
#include <credentials/certificates/crl.h>

static certificate_t* parse_cert(chunk_t encoding)
{
	certificate_t *cert;

	cert = lib->creds->create(lib->creds, CRED_CERTIFICATE, CERT_X509,
						BUILD_BLOB_ASN1_DER, encoding,
						BUILD_END);
	ck_assert(cert);

	return cert;
}

static certificate_t* parse_crl(chunk_t encoding)
{
	certificate_t *crl;

	crl = lib->creds->create(lib->creds, CRED_CERTIFICATE, CERT_X509_CRL,
						BUILD_BLOB_ASN1_DER, encoding,
						BUILD_END);
	ck_assert(crl);

	return crl;
}

typedef struct {
	chunk_t serial;
	chunk_t cert_encoding;
	chunk_t crl_encoding;
} serial_number_t;

static serial_number_t serial_numbers[] = {
	{ chunk_from_chars(0x00),
	  chunk_from_chars(
		0x30,0x82,0x01,0xD6,0x30,0x82,0x01,0x3F,0xA0,0x03,0x02,0x01,0x02,0x02,0x01,0x00,
		0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,
		0x31,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x48,0x31,0x13,
		0x30,0x11,0x06,0x03,0x55,0x04,0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,0x6E,0x67,0x53,
		0x77,0x61,0x6E,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x03,0x13,0x04,0x74,0x65,
		0x73,0x74,0x30,0x1E,0x17,0x0D,0x32,0x32,0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,
		0x34,0x34,0x5A,0x17,0x0D,0x32,0x33,0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,0x34,
		0x34,0x5A,0x30,0x31,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x43,
		0x48,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,
		0x6E,0x67,0x53,0x77,0x61,0x6E,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x03,0x13,
		0x04,0x74,0x65,0x73,0x74,0x30,0x81,0x9F,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,
		0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8D,0x00,0x30,0x81,0x89,0x02,0x81,
		0x81,0x00,0xB1,0x9B,0xD4,0x51,0x24,0xFC,0x56,0x1D,0x3D,0xFB,0xA2,0xEA,0x37,0x02,
		0x70,0x72,0x87,0x84,0x2F,0x3B,0x2D,0x6E,0x22,0xEF,0x3F,0x37,0x04,0xB2,0x6F,0xB7,
		0xE7,0xD8,0x58,0x05,0xDE,0x34,0xBF,0x99,0xE6,0x40,0x7A,0x56,0xA7,0x73,0xF5,0x98,
		0xCB,0xB0,0x37,0x90,0x5E,0xD1,0x3F,0xF4,0x73,0x50,0x7F,0x53,0x8E,0xF1,0x04,0x25,
		0xB4,0x77,0x22,0x4E,0x8A,0x9D,0x27,0x8F,0x6F,0xAF,0x59,0xBD,0xB0,0x0F,0xF0,0xAA,
		0x11,0x94,0x66,0x16,0x10,0x58,0xAD,0x77,0xA1,0xAC,0x58,0xB4,0xD0,0x0D,0xBC,0x11,
		0xE0,0xC0,0xE9,0x29,0xDC,0x42,0x63,0x01,0x23,0x4F,0x28,0x41,0x6D,0x34,0x9E,0x0C,
		0x4A,0xC8,0x62,0x83,0xB5,0x71,0x71,0x0B,0x51,0xC0,0x4C,0x37,0xD4,0x68,0x19,0x52,
		0x9A,0x8B,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,
		0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x81,0x81,0x00,0x0D,0x3E,0x13,0x3D,0x58,0x72,
		0x90,0x21,0x06,0x29,0xBD,0xA5,0x06,0x87,0x67,0x80,0x2C,0xE0,0x61,0xF5,0x66,0x76,
		0x63,0xAB,0x97,0xD5,0x45,0x9B,0x2B,0x3C,0x6B,0xA7,0xB0,0xB4,0x31,0x52,0xC6,0xD9,
		0x72,0xFC,0xC5,0x37,0xE5,0xFF,0xD0,0x80,0x63,0x09,0xD2,0x1E,0xC1,0x77,0x92,0xCC,
		0x07,0x08,0x5D,0xD0,0x30,0x67,0x9A,0x6B,0x82,0x19,0x89,0x0E,0x10,0xC7,0xA4,0xA7,
		0x7C,0x96,0x76,0x8C,0x72,0xDB,0x73,0x13,0x49,0xE5,0x8B,0xAC,0x0B,0x1E,0xEB,0x31,
		0x74,0xEB,0xE4,0xA0,0x5D,0x49,0x9A,0x76,0x3C,0xA5,0xEF,0x55,0xE2,0x32,0x25,0x1A,
		0xE3,0x05,0x37,0xAC,0xFF,0x9F,0x94,0x92,0xE6,0x0E,0x53,0xC0,0xFC,0x52,0xB8,0xD0,
		0xFA,0x66,0x0B,0xCE,0xCA,0x88,0x66,0x3B,0x83,0x48),
	  chunk_from_chars(
		0x30,0x82,0x01,0x5D,0x30,0x81,0xC7,0x02,0x01,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,
		0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,0x31,0x31,0x0B,0x30,0x09,0x06,
		0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x48,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,
		0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,0x6E,0x67,0x53,0x77,0x61,0x6E,0x31,0x0D,0x30,
		0x0B,0x06,0x03,0x55,0x04,0x03,0x13,0x04,0x74,0x65,0x73,0x74,0x17,0x0D,0x32,0x32,
		0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,0x17,0x0D,0x32,0x32,0x31,
		0x32,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,0x30,0x22,0x30,0x20,0x02,0x01,
		0x00,0x17,0x0D,0x32,0x32,0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,
		0x30,0x0C,0x30,0x0A,0x06,0x03,0x55,0x1D,0x15,0x04,0x03,0x0A,0x01,0x01,0xA0,0x3E,
		0x30,0x3C,0x30,0x1F,0x06,0x03,0x55,0x1D,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0xE2,
		0x6D,0x1E,0xDF,0x83,0x8E,0xA2,0x1F,0xC3,0x00,0xDD,0x44,0x6F,0x8A,0x4D,0x70,0x0C,
		0x02,0xE3,0x1F,0x30,0x0A,0x06,0x03,0x55,0x1D,0x14,0x04,0x03,0x02,0x01,0x00,0x30,
		0x0D,0x06,0x03,0x55,0x1D,0x1B,0x01,0x01,0xFF,0x04,0x03,0x02,0x01,0x00,0x30,0x0D,
		0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x81,0x81,
		0x00,0x35,0x72,0x26,0x38,0x87,0x21,0xA2,0x5C,0xED,0x5C,0x04,0xD9,0x49,0xC0,0xB6,
		0x75,0x7C,0x5A,0xEA,0x46,0x6E,0x1E,0xED,0x3C,0x9B,0x41,0x31,0x37,0x3F,0xAA,0xE7,
		0x16,0x39,0x17,0x48,0x5F,0x84,0x48,0x6F,0xA4,0xF6,0x9D,0x79,0xDE,0xC3,0xE9,0x82,
		0x87,0xEE,0xD4,0xD0,0x2F,0xBF,0x8B,0x74,0x1E,0xA7,0x21,0x63,0xB6,0x5A,0x39,0xFF,
		0xDE,0xD0,0x6E,0xE3,0xB5,0x3B,0x0C,0x42,0x46,0x97,0x43,0x2E,0x6B,0x4D,0xF8,0x54,
		0x59,0x8F,0xD8,0x72,0xB3,0xB0,0x29,0xCB,0x56,0xA7,0x8A,0x01,0xD6,0xEA,0xE0,0x69,
		0xF7,0x36,0xC4,0x06,0xE6,0x05,0xC0,0x10,0xD2,0xB7,0x43,0x46,0xCC,0x8A,0x53,0xA1,
		0xA6,0xD3,0x88,0x73,0x53,0x29,0x10,0xC2,0xC6,0xCE,0x24,0xC3,0xCE,0x14,0xED,0xB0,
		0x64)
	},
	{ chunk_from_chars(0x7f),
	  chunk_from_chars(
		0x30,0x82,0x01,0xD6,0x30,0x82,0x01,0x3F,0xA0,0x03,0x02,0x01,0x02,0x02,0x01,0x7F,
		0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,
		0x31,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x48,0x31,0x13,
		0x30,0x11,0x06,0x03,0x55,0x04,0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,0x6E,0x67,0x53,
		0x77,0x61,0x6E,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x03,0x13,0x04,0x74,0x65,
		0x73,0x74,0x30,0x1E,0x17,0x0D,0x32,0x32,0x31,0x31,0x31,0x38,0x31,0x33,0x30,0x37,
		0x32,0x31,0x5A,0x17,0x0D,0x32,0x33,0x31,0x31,0x31,0x38,0x31,0x33,0x30,0x37,0x32,
		0x31,0x5A,0x30,0x31,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x43,
		0x48,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,
		0x6E,0x67,0x53,0x77,0x61,0x6E,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x03,0x13,
		0x04,0x74,0x65,0x73,0x74,0x30,0x81,0x9F,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,
		0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8D,0x00,0x30,0x81,0x89,0x02,0x81,
		0x81,0x00,0xB1,0x9B,0xD4,0x51,0x24,0xFC,0x56,0x1D,0x3D,0xFB,0xA2,0xEA,0x37,0x02,
		0x70,0x72,0x87,0x84,0x2F,0x3B,0x2D,0x6E,0x22,0xEF,0x3F,0x37,0x04,0xB2,0x6F,0xB7,
		0xE7,0xD8,0x58,0x05,0xDE,0x34,0xBF,0x99,0xE6,0x40,0x7A,0x56,0xA7,0x73,0xF5,0x98,
		0xCB,0xB0,0x37,0x90,0x5E,0xD1,0x3F,0xF4,0x73,0x50,0x7F,0x53,0x8E,0xF1,0x04,0x25,
		0xB4,0x77,0x22,0x4E,0x8A,0x9D,0x27,0x8F,0x6F,0xAF,0x59,0xBD,0xB0,0x0F,0xF0,0xAA,
		0x11,0x94,0x66,0x16,0x10,0x58,0xAD,0x77,0xA1,0xAC,0x58,0xB4,0xD0,0x0D,0xBC,0x11,
		0xE0,0xC0,0xE9,0x29,0xDC,0x42,0x63,0x01,0x23,0x4F,0x28,0x41,0x6D,0x34,0x9E,0x0C,
		0x4A,0xC8,0x62,0x83,0xB5,0x71,0x71,0x0B,0x51,0xC0,0x4C,0x37,0xD4,0x68,0x19,0x52,
		0x9A,0x8B,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,
		0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x81,0x81,0x00,0xA4,0x92,0x32,0x35,0xD2,0xC7,
		0x67,0x12,0x1E,0x5E,0x7C,0x0C,0x08,0x92,0x95,0xEE,0xD4,0x43,0x59,0xEF,0x0C,0x0A,
		0x73,0x2C,0xB4,0x6C,0xB6,0x4A,0x98,0x85,0xD5,0xA2,0x96,0x6B,0x90,0xBA,0xB7,0xC2,
		0x67,0x5F,0x0D,0xCD,0x9E,0xB3,0x5B,0xDE,0xA0,0xF4,0x5C,0x49,0x80,0x66,0x7B,0x33,
		0x08,0x7F,0xCB,0x65,0xE6,0xCB,0x4F,0x50,0x59,0x16,0x23,0xEA,0xB6,0x70,0xFD,0x98,
		0xD3,0x65,0x2A,0x94,0x65,0x2E,0x77,0x93,0x48,0x99,0x12,0xC3,0x95,0xDE,0x56,0xA8,
		0xFE,0x33,0x1E,0x36,0x62,0x29,0x0A,0x5A,0x07,0xF9,0x19,0xF6,0x04,0x86,0xB5,0x43,
		0x2C,0x49,0x4B,0xC2,0x98,0x3F,0x96,0x95,0x61,0x96,0xF8,0x22,0x4C,0xFB,0x45,0x41,
		0x27,0x06,0x0E,0x44,0x36,0x19,0x7F,0x51,0x96,0x20),
	  chunk_from_chars(
		0x30,0x82,0x01,0x5D,0x30,0x81,0xC7,0x02,0x01,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,
		0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,0x31,0x31,0x0B,0x30,0x09,0x06,
		0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x48,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,
		0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,0x6E,0x67,0x53,0x77,0x61,0x6E,0x31,0x0D,0x30,
		0x0B,0x06,0x03,0x55,0x04,0x03,0x13,0x04,0x74,0x65,0x73,0x74,0x17,0x0D,0x32,0x32,
		0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,0x17,0x0D,0x32,0x32,0x31,
		0x32,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,0x30,0x22,0x30,0x20,0x02,0x01,
		0x7F,0x17,0x0D,0x32,0x32,0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,
		0x30,0x0C,0x30,0x0A,0x06,0x03,0x55,0x1D,0x15,0x04,0x03,0x0A,0x01,0x01,0xA0,0x3E,
		0x30,0x3C,0x30,0x1F,0x06,0x03,0x55,0x1D,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0xE2,
		0x6D,0x1E,0xDF,0x83,0x8E,0xA2,0x1F,0xC3,0x00,0xDD,0x44,0x6F,0x8A,0x4D,0x70,0x0C,
		0x02,0xE3,0x1F,0x30,0x0A,0x06,0x03,0x55,0x1D,0x14,0x04,0x03,0x02,0x01,0x7F,0x30,
		0x0D,0x06,0x03,0x55,0x1D,0x1B,0x01,0x01,0xFF,0x04,0x03,0x02,0x01,0x7F,0x30,0x0D,
		0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x81,0x81,
		0x00,0x00,0x93,0xAD,0x02,0x09,0x74,0x97,0x3C,0xBE,0x82,0x3D,0x39,0xCC,0xD0,0x55,
		0x8F,0xBA,0xE4,0xB5,0x53,0x83,0xF0,0x11,0x29,0xE8,0x2B,0x77,0x8D,0xC5,0xA2,0x0F,
		0x86,0x31,0x87,0x4D,0xAA,0x4B,0x78,0x44,0xFB,0x42,0xDB,0x81,0xC7,0xF0,0xA6,0x65,
		0x68,0x36,0xC9,0x2D,0x37,0xA7,0x1C,0x23,0xD3,0xA4,0x75,0x85,0x0B,0x09,0xF1,0x1E,
		0x24,0x19,0xB7,0xEE,0x5A,0x89,0x1C,0xF9,0x98,0xE7,0x6F,0xB8,0xF2,0x9A,0xB2,0x5E,
		0xC5,0x47,0xE3,0x6D,0x50,0x9D,0x13,0x61,0x85,0x71,0x0A,0xF2,0xF3,0xBC,0x03,0xE6,
		0xB8,0x1F,0x32,0x92,0x4C,0x95,0x31,0xF0,0xF4,0x85,0x41,0x97,0x1F,0x43,0xC7,0x51,
		0xD9,0x90,0xBE,0xA6,0xE5,0x06,0x92,0xEF,0xF6,0x81,0xC7,0xD4,0xB3,0xF7,0x1B,0xA3,
		0x1C),
	},
	{ chunk_from_chars(0x80),
	  chunk_from_chars(
		0x30,0x82,0x01,0xD7,0x30,0x82,0x01,0x40,0xA0,0x03,0x02,0x01,0x02,0x02,0x02,0x00,
		0x80,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,
		0x30,0x31,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x48,0x31,
		0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,0x6E,0x67,
		0x53,0x77,0x61,0x6E,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x03,0x13,0x04,0x74,
		0x65,0x73,0x74,0x30,0x1E,0x17,0x0D,0x32,0x32,0x31,0x31,0x31,0x38,0x31,0x33,0x30,
		0x37,0x32,0x31,0x5A,0x17,0x0D,0x32,0x33,0x31,0x31,0x31,0x38,0x31,0x33,0x30,0x37,
		0x32,0x31,0x5A,0x30,0x31,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
		0x43,0x48,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x0A,0x13,0x0A,0x73,0x74,0x72,
		0x6F,0x6E,0x67,0x53,0x77,0x61,0x6E,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x03,
		0x13,0x04,0x74,0x65,0x73,0x74,0x30,0x81,0x9F,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,
		0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8D,0x00,0x30,0x81,0x89,0x02,
		0x81,0x81,0x00,0xB1,0x9B,0xD4,0x51,0x24,0xFC,0x56,0x1D,0x3D,0xFB,0xA2,0xEA,0x37,
		0x02,0x70,0x72,0x87,0x84,0x2F,0x3B,0x2D,0x6E,0x22,0xEF,0x3F,0x37,0x04,0xB2,0x6F,
		0xB7,0xE7,0xD8,0x58,0x05,0xDE,0x34,0xBF,0x99,0xE6,0x40,0x7A,0x56,0xA7,0x73,0xF5,
		0x98,0xCB,0xB0,0x37,0x90,0x5E,0xD1,0x3F,0xF4,0x73,0x50,0x7F,0x53,0x8E,0xF1,0x04,
		0x25,0xB4,0x77,0x22,0x4E,0x8A,0x9D,0x27,0x8F,0x6F,0xAF,0x59,0xBD,0xB0,0x0F,0xF0,
		0xAA,0x11,0x94,0x66,0x16,0x10,0x58,0xAD,0x77,0xA1,0xAC,0x58,0xB4,0xD0,0x0D,0xBC,
		0x11,0xE0,0xC0,0xE9,0x29,0xDC,0x42,0x63,0x01,0x23,0x4F,0x28,0x41,0x6D,0x34,0x9E,
		0x0C,0x4A,0xC8,0x62,0x83,0xB5,0x71,0x71,0x0B,0x51,0xC0,0x4C,0x37,0xD4,0x68,0x19,
		0x52,0x9A,0x8B,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,
		0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x81,0x81,0x00,0x54,0xC6,0x85,0x31,0x85,
		0x17,0x86,0xCB,0xFF,0x63,0x09,0x2A,0x90,0xDD,0x2C,0xC8,0x7E,0xD7,0x8B,0xE0,0x37,
		0xF7,0x2D,0xE2,0xCF,0x7E,0x57,0x26,0x4C,0x1D,0x3D,0x57,0xF0,0x6D,0x0D,0x18,0xA3,
		0xAB,0x49,0xD3,0x4E,0x08,0x70,0x4D,0xAE,0x05,0xF7,0xC5,0x47,0x99,0x89,0xFA,0xCA,
		0x15,0x36,0x3F,0xDA,0xA7,0x64,0xEE,0xE6,0x1C,0x6E,0x6E,0x9D,0x39,0x61,0xCB,0x5E,
		0x8F,0xAD,0x5C,0x90,0xD6,0xAE,0xCD,0xE9,0xBB,0x62,0xB9,0xCB,0x0E,0x51,0xDD,0x27,
		0xAF,0xF3,0xE2,0xD0,0xAC,0x9E,0x99,0x55,0xB5,0x2F,0x46,0x99,0xDB,0x2F,0xEC,0x23,
		0x76,0x0E,0x82,0xE3,0xA7,0xC3,0xF6,0xA2,0x61,0x32,0xC7,0x1F,0xD6,0x22,0x9B,0xFA,
		0xD8,0xFD,0x8F,0xB2,0xB7,0x6B,0x71,0xF0,0x92,0x1F,0x44),
	  chunk_from_chars(
		0x30,0x82,0x01,0x60,0x30,0x81,0xCA,0x02,0x01,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,
		0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,0x31,0x31,0x0B,0x30,0x09,0x06,
		0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x48,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,
		0x0A,0x13,0x0A,0x73,0x74,0x72,0x6F,0x6E,0x67,0x53,0x77,0x61,0x6E,0x31,0x0D,0x30,
		0x0B,0x06,0x03,0x55,0x04,0x03,0x13,0x04,0x74,0x65,0x73,0x74,0x17,0x0D,0x32,0x32,
		0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,0x17,0x0D,0x32,0x32,0x31,
		0x32,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,0x5A,0x30,0x23,0x30,0x21,0x02,0x02,
		0x00,0x80,0x17,0x0D,0x32,0x32,0x31,0x31,0x32,0x30,0x31,0x30,0x31,0x37,0x34,0x34,
		0x5A,0x30,0x0C,0x30,0x0A,0x06,0x03,0x55,0x1D,0x15,0x04,0x03,0x0A,0x01,0x01,0xA0,
		0x40,0x30,0x3E,0x30,0x1F,0x06,0x03,0x55,0x1D,0x23,0x04,0x18,0x30,0x16,0x80,0x14,
		0xE2,0x6D,0x1E,0xDF,0x83,0x8E,0xA2,0x1F,0xC3,0x00,0xDD,0x44,0x6F,0x8A,0x4D,0x70,
		0x0C,0x02,0xE3,0x1F,0x30,0x0B,0x06,0x03,0x55,0x1D,0x14,0x04,0x04,0x02,0x02,0x00,
		0x80,0x30,0x0E,0x06,0x03,0x55,0x1D,0x1B,0x01,0x01,0xFF,0x04,0x04,0x02,0x02,0x00,
		0x80,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,
		0x03,0x81,0x81,0x00,0x7D,0x98,0x63,0x80,0x7B,0x43,0xE6,0x56,0xCD,0x59,0x0E,0xEC,
		0xD7,0x45,0x93,0xE5,0xD1,0x97,0x3E,0x47,0x87,0xFD,0x6F,0x8D,0x10,0xB0,0xDA,0x7C,
		0x5D,0xDC,0x0B,0x1B,0xC8,0x63,0x5D,0x8D,0x02,0xDE,0xD0,0xC7,0xBB,0xE4,0x50,0xEA,
		0xA7,0x2E,0x06,0xBC,0xF2,0x36,0x8D,0x7A,0xA2,0xE5,0x67,0x0F,0x03,0x9A,0x75,0x1D,
		0x18,0x0E,0x57,0x39,0x86,0xF3,0xEF,0x29,0xED,0xDE,0x24,0x50,0x9E,0x1F,0xC7,0x90,
		0x51,0x95,0xA0,0xD5,0x11,0x5E,0x96,0x0A,0xA5,0x4A,0x4D,0xFE,0x97,0xE4,0x1A,0xEA,
		0xC3,0x56,0x2A,0x21,0xA6,0x6E,0xE5,0xBA,0xAC,0x70,0xA5,0xE5,0x90,0x9A,0x3C,0x36,
		0xFF,0xD6,0x8B,0xC2,0x09,0x19,0xF4,0x21,0x68,0x50,0xEC,0x83,0xF8,0xF6,0xB5,0x24,
		0x8F,0x2B,0x77,0xB2)
	}
};

START_TEST(test_parse_serial_numbers)
{
	enumerator_t *enumerator;
	certificate_t *cert, *crl;
	crl_t *x509_crl;
	x509_t *x509;
	chunk_t serial;

	/* parse ASN.1 DER encoded certificate */
	cert = parse_cert(serial_numbers[_i].cert_encoding);

	/* check parsed serial number */
	x509 = (x509_t*)cert;
	ck_assert_chunk_eq(x509->get_serial(x509), serial_numbers[_i].serial);
	cert->destroy(cert);

	/* parse ASN.1 DER encoded crl */
	crl = parse_crl(serial_numbers[_i].crl_encoding);

	/* check parsed serial number */
	x509_crl = (crl_t*)crl;
	ck_assert_chunk_eq(x509_crl->get_serial(x509_crl), serial_numbers[_i].serial);

	enumerator = x509_crl->create_enumerator(x509_crl);
	if (enumerator->enumerate(enumerator, &serial, NULL, NULL))
	{
		ck_assert_chunk_eq(serial, serial_numbers[_i].serial);
	}
	enumerator->destroy(enumerator);

	crl->destroy(crl);
}
END_TEST

Suite *serial_parse_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("serial_parse");

	tc = tcase_create("parse serial numbers");
	tcase_add_loop_test(tc, test_parse_serial_numbers, 0, countof(serial_numbers));
	suite_add_tcase(s, tc);

	return s;
}
