// ---------------------------------------------------------------------------
// - Time.cpp                                                                -
// - standard object library - time class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Time.hpp"
#include "Vector.hpp"
#include "Regex.hpp"
#include "Utility.hpp"
#include "Integer.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "cclk.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the default utc zone
  static const String UTC_ZONE_DEF = "+0000";

  // return the absolute value
  static inline long abs (const long abs) {
    return (abs < 0) ? -abs : abs;
  }
  static inline t_long abs (const t_long abs) {
    return (abs < 0) ? -abs : abs;
  }

  // -------------------------------------------------------------------------
  // - public section                                                        -
  // -------------------------------------------------------------------------

  // return the time in second since the origin

  t_long Time::gettclk (void) {
    return c_time ();
  }

  // return the timezone in seconds
  
  t_long Time::getzone (void) {
    return c_tzone ();
  }
  
  // return a formated time by value

  String Time::totfmt (const t_long wclk, const String& tsep) {
    // compute the seconds
    long secs = (long) (wclk % 60LL);
    // compute minute
    long mins = (long) ((wclk / 60LL) % 60LL);
    // compute hours
    long hour = (long) ((wclk / 3600LL) % 24LL);
    // fix negative time
    if (secs < 0) secs += 60LL;
    if (mins < 0) mins += 60LL;
    if (hour < 0) hour += 24LL;
    // initialize the result
    String result;
    // format the hour
    if (hour < 10) {
      result = result + '0' + hour;
    } else {
      result = result + hour;
    }
    if (result.isnil () == false) result += tsep;
    // format the minute
    if (mins < 10) {
      result = result + '0' + mins;
    } else {
      result = result + mins;
    }
    if (result.isnil () == false) result += tsep;
    // format the seconds
    if (secs < 10) {
      result = result + '0' + secs;
    } else {
      result = result + secs;
    }
    return result;
  }

  // return a formated zone by value
  
  String Time::tozfmt (const long zclk, const String& zsep) {
    // save the sign
    bool negt = (zclk < 0);
    long wclk = negt ? -zclk : zclk;
    // get the minutes
    long mins = (long) ((wclk / 60LL) % 60LL);
    // compute hours
    long hour = (long) ((wclk / 3600LL) % 24LL);
     // format the result
    String result = negt ? '-' : '+';
    // add the hour
    if (hour < 10) result += '0';
    result += hour;
    // add the separator
    if (zsep.isnil () == false) result += zsep;
    // add the minutes
    if (mins < 10) result += '0';
    result += mins;
    // here we are
    return result;
  }

  // return a formated string of the time zone

  String Time::tozone (void) {
    // get the time zone difference
    long zclk = c_tzone ();
    // format the zone
    return Time::tozfmt (zclk, "");
  }

  // -------------------------------------------------------------------------
  // - class section                                                        -
  // -------------------------------------------------------------------------

  // create a current atc clock

  Time::Time (void) {
    d_tclk  = c_time ();
  }

  // create a specific time clock

  Time::Time (const t_long tclk) {
    d_tclk  = tclk;
  }

  // create a specific time clock

  Time::Time (const long hour, const long mins, const long secs) {
    settime (hour, mins, secs);
  }

  // copy construct this time

  Time::Time (const Time& that) {
    that.rdlock ();
    try {
      d_tclk = that.d_tclk;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // return the class name

  String Time::repr (void) const {
    return "Time";
  }

  // return a clone of this object

  Object* Time::clone (void) const {
    return new Time (*this);
  }

  // assign a time object to this one

  Time& Time::operator = (const Time& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      d_tclk = that.d_tclk;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // add a specific time in second to the atc clock

  void Time::add (const t_long tclk) {
    wrlock ();
    d_tclk += tclk;
    unlock ();
  }

  // set the atc clock in seconds

  void Time::settime (const t_long tclk) {
    wrlock ();
    d_tclk = tclk;
    unlock ();
  }

  // set the atc clock by iso representation

  void Time::settime (const String& time) {
    wrlock ();
    try {
      // create a time regex
      Regex re ("($d$d):($d$d):($d$d)Z");
      // check the time
      if (re == time) {
	// extract tme information
	long hour = Utility::tolong (re.getstr (0));
	long mins = Utility::tolong (re.getstr (1));
	long secs = Utility::tolong (re.getstr (2));
	// set the atc time
	settime (hour, mins,secs);
	unlock ();
      } else {
	throw Exception ("time-error", "invalid time format", time);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  // set the time by specific elements

  void Time::settime (const long hour, const long mins, const long secs) {
    wrlock ();
    t_long tclk = hour * HSEC + mins * MSEC + secs;
    settime (tclk);
    unlock ();
  }
      
  // return the atc clock

  t_long Time::gettime (const bool utc) const {
    rdlock ();
    try {
      t_long result = d_tclk;
      if (utc == false) result += (t_long) c_tzone ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // get the number of seconds
  
  long Time::getsecs (const bool utc) const {
    rdlock ();
    // compute the working clock based on the timezone
    t_long wclk = abs (d_tclk);
    if (utc == false) wclk += (t_long) c_tzone ();
    // compute the seconds
    long result = (long) (wclk % 60LL);
    unlock ();
    return result;
  }

  // get the number of minutes

  long Time::getmins (const bool utc) const {
    rdlock ();
    // compute the working clock based on the timezone
    t_long wclk = abs (d_tclk);
    if (utc == false) wclk += (t_long) c_tzone ();
    // compute the minutes
    long result = (long) ((wclk / 60LL) % 60LL);
    unlock ();
    return result;
  }

  // get the number of hour

  long Time::gethour (const bool utc) const {
    rdlock ();
    // compute the working clock based on the timezone
    t_long wclk = abs (d_tclk);
    if (utc == false) wclk += (t_long) c_tzone ();
    // compute the hour
    long result = (long) ((wclk / 3600LL) % 24LL);
    unlock ();
    return result;
  }

  // return the base day reference time
  
  t_long Time::getbday (void) const {
    rdlock ();
    try {
      t_long result = (d_tclk / DSEC) * DSEC;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // format the time
  
  String Time::format (const bool utc) const {
    rdlock ();
    try {
      // compute the working clock based on the timezone
      t_long wclk = abs (d_tclk);
      if (utc == false) wclk += (t_long) c_tzone ();
      // format the time
      String result = Time::totfmt (wclk, ':');
      // unlock and return
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // format the time in a form similar to ISO-8601
  
  String Time::toiso (const bool utc) const {
    rdlock ();
    try {
      // get the time representation (local)
      String result = Time::format (utc);
      // add the suffix if any
      if (utc == true) result += 'Z';
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // format the time in a form similar to RFC-2822

  String Time::torfc (const bool utc) const {
    rdlock ();
    try {
      // get the time representation (local)
      String result = Time::format (utc);
      // add the timezone difference (including daylight if any)
      result += ' ';
      if (utc == true) {
	result += UTC_ZONE_DEF;
      } else {
	result += Time::tozone ();
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 13;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ADD     = zone.intern ("add");
  static const long QUARK_TOISO   = zone.intern ("to-iso");
  static const long QUARK_TORFC   = zone.intern ("to-rfc");
  static const long QUARK_FORMAT  = zone.intern ("format");
  static const long QUARK_ADDMNS  = zone.intern ("add-minutes");
  static const long QUARK_ADDHRS  = zone.intern ("add-hours");
  static const long QUARK_ADDDAYS = zone.intern ("add-days");
  static const long QUARK_SETTIME = zone.intern ("set-time");
  static const long QUARK_GETTIME = zone.intern ("get-time");
  static const long QUARK_GETSECS = zone.intern ("seconds");
  static const long QUARK_GETMINS = zone.intern ("minutes");
  static const long QUARK_GETHOUR = zone.intern ("hours");
  static const long QUARK_GETBDAY = zone.intern ("get-base-day");

  // create a new object in a generic way
 
  Object* Time::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default time object
    if (argc == 0) return new Time;
    if (argc == 1) {
     t_long tval = argv->getlong (0);
      return new Time (tval);
    }
    if (argc == 3) {
      long hour = argv->getlong (0);
      long mins = argv->getlong (1);
      long secs = argv->getlong (2);
      return new Time (hour, mins, secs);
    }
    throw Exception ("argument-error",
                     "too many argument with time constructor");
  }
 
  // return true if the given quark is defined

  bool Time::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* Time::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 arguments
    if (argc == 0) {
      if (quark == QUARK_TOISO)   return new String  (toiso   (false));
      if (quark == QUARK_TORFC)   return new String  (torfc   (false));
      if (quark == QUARK_FORMAT)  return new String  (format  (false));
      if (quark == QUARK_GETTIME) return new Integer (gettime (true));
      if (quark == QUARK_GETSECS) return new Integer (getsecs (true));
      if (quark == QUARK_GETMINS) return new Integer (getmins (true));
      if (quark == QUARK_GETHOUR) return new Integer (gethour (true));
      if (quark == QUARK_GETBDAY) return new Integer (getbday ());
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADD) {
	t_long tval = argv->getlong (0);
	add (tval);
	return nilp;
      }
      if (quark == QUARK_GETTIME) {
	bool utc = argv->getbool (0);
	return new Integer (gettime (utc));
      }
      if (quark == QUARK_SETTIME) {
	t_long tclk = argv->getlong (0);
	settime (tclk);
	return nilp;
      }
      if (quark == QUARK_ADDMNS) {
	t_long num = argv->getlong (0);
	t_long sec = num * (t_long) MSEC;
	add (sec);
	return nilp;
      }
      if (quark == QUARK_ADDHRS) {
	t_long num = argv->getlong (0);
	t_long sec = num * (t_long) HSEC;
	add (sec);
	return nilp;
      }
      if (quark == QUARK_ADDDAYS) {
	t_long num = argv->getlong (0);
	t_long sec = num * (t_long) DSEC;
	add (sec);
	return nilp;
      }
      if (quark == QUARK_FORMAT) {
	bool utc = argv->getbool (0);
	return new String (format (utc));
      }
      if (quark == QUARK_TOISO) {
	bool utc = argv->getbool (0);
	return new String (toiso (utc));
      }
      if (quark == QUARK_GETSECS) {
	bool utc = argv->getbool (0);
	return new Integer (getsecs (utc));
      }
      if (quark == QUARK_GETMINS) {
	bool utc = argv->getbool (0);
	return new Integer (getmins (utc));
      }
      if (quark == QUARK_GETHOUR) {
	bool utc = argv->getbool (0);
	return new Integer (gethour (utc));
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
