//
// anyRemote
// a wi-fi or bluetooth remote for your PC.
//
// Copyright (C) 2012-2014 Mikhail Fedotov <anyremote@mail.ru>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <errno.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "atsend.h"
#include "btio.h"
#include "common.h"
#include "conf.h"
#include "executor.h"
#include "gen_ar.h"
#include "peer.h"
#include "pr_btspp.h"
#include "pr_frontend.h"
#include "pr_l2cap.h"
#include "pr_rfcomm.h"
#include "pr_serial.h"
#include "pr_socket.h"
#include "pr_stdin.h"
#include "pr_web.h"
#include "state.h"
#include "str.h"
#include "utils.h"

extern char tmp[MAXMAXLEN];

static int peerWrite     (ConnectInfo* peer, const dMessage* msg);
static int peerWriteBytes(ConnectInfo* peer, const char* command);
static void writeIViewerHeartbeat(int fd);
static void writeBemusedHeartbeat(int fd);

SingleList * _connections = NULL;

static ConnectInfo* allocPeer()
{
    ConnectInfo* peer = (ConnectInfo*) malloc(sizeof(ConnectInfo));

    peer->mode                 = SERVER_MAX;
    peer->state                = PEER_DISCONNECTED;
    peer->port                 = -1;
    peer->connectionData       = NULL;  // specific to connection
    peer->portStr              = NULL;

    return peer;
}

static void freePeer(void* data)
{
    ConnectInfo* peer = (ConnectInfo* ) data;
    INFO2("[DS]: freePeer() %d", peer->mode);
    if (peer->connectionData) {
        free(peer->connectionData);
    }
    free(peer);
}


// do not take into account connection-less peers (HTML + XML)
static int countConnections(int all)
{
    int cnum = 0;
    SingleList* list = _connections;
    //INFO2("[DS]: countConnections() enter %d", listSingleLength(list));
    
    while (list) {
       ConnectInfo* peer = (ConnectInfo*) list->data;
       if (peer->state == PEER_CONNECTED) {
           if (all || (peer->mode != SERVER_WEB && peer->mode != SERVER_CMXML)) {
                cnum++;
           }
       }
       list = listSingleNext(list);
    }
    return cnum;
}

static int dummyOpen(ConnectInfo* connInfo)
{
    return EXIT_OK;
}

static int dummyFD(ConnectInfo* connInfo)
{
    return -1;
}

static int setupNope(ConnectInfo* p)
{
    return 1;
}

static int acceptNope(ConnectInfo* p)
{
    switch (p->mode) {
        case SERVER_WEB:
            logger(L_INF, "[DS]: Built-in web server: init OK");
            break;
        case SERVER_CMXML:
            logger(L_INF, "[DS]: Built-in XML server: init OK");
            break;
        case CLIENT_ILIRC:
        case CLIENT_NOAT:
        case SERVER_STDIN:
            logger(L_INF, "[DS]: Unix client: init OK");
            break;
    }
    return 1;
}

static struct {
    int  id;
    int  (*descriptor)           (ConnectInfo* p);
    int  (*openConnection)       (ConnectInfo* p);
    int  (*setupConnection)      (ConnectInfo* p);
    int  (*acceptConnection)     (ConnectInfo* p);
    int  (*writeConnection)      (ConnectInfo* p, const dMessage* msg);
    int  (*writeBytesConnection) (ConnectInfo* p, const char* value);
    void (*closeConnection)      (ConnectInfo* p, int final);

} _peerHandlers[] = {

    // type         get FD    open          setup       accept       write       write bytes    close
    //              (EXIT_OK  (1/-1)        (1/-1)      (EXIT_OK/   (EXIT_OK/NOK)  (void)
    //               /NOK)                                   NOK)
    {CLIENT_RFCOMM, rfcommFD, rfcommConnect,rfcommSetup,acceptNope,  peerWrite,  NULL,          rfcommClose  },
    {CLIENT_AT,     serialFD, serialOpen,   serialSetup,acceptNope,  peerWrite,  NULL,          serialClose  },
    {SERVER_TCP,    socketFD, socketOpen,   socketSetup,socketAccept,socketWrite,peerWriteBytes,socketClose  },
    {SERVER_BT,     btsppFD,  btsppOpen,    btsppSetup, btsppAccept, peerWrite,  peerWriteBytes,btsppClose   },
    {CLIENT_ILIRC,  uxFD,     openUxSocket, setupNope,  acceptNope,  NULL,       NULL,          closeUxSocket},
    {CLIENT_NOAT,   serialFD, serialOpen,   setupNope,  acceptNope,  peerWrite,  NULL,          serialClose  },
    {SERVER_STDIN,  dummyFD,  openStdin,    setupNope,  acceptNope,  NULL,       NULL,          closeStdin   },
    {SERVER_WEB,    dummyFD,  openWeb,      setupNope,  acceptNope,  writeWeb,   NULL,          closeWebPort },
    {SERVER_CMXML,  dummyFD,  openWeb,      setupNope,  acceptNope,  writeWeb,   NULL,          closeWebPort },
    #ifdef USE_L2CAP
    {SERVER_L2CAP,  l2capFD,  l2capOpen,    l2capSetup, l2capAccept, peerWrite,  peerWriteBytes,l2capClose   },
    #endif
    {SERVER_UX ,    socketFD, dummyOpen,    socketSetup,socketAccept,peerWrite,  peerWriteBytes,socketClose  } // not used
};

void freePeers()
{
    INFO2("[DS]: freePeers() start");
    listSingleFullFree(_connections, freePeer);
    _connections = NULL;
        
    freeState();

    INFO2("[DS]: freePeers() end");
}

static int addTcpPeer(ConnectInfo* peer, char *strPort)
{
    // check is it port or unix socket
    char * ch = strPort;
    int isPort = 1;
    while (*ch != '\0') {
        if (isdigit(*ch)) {
            ch++;
        } else {
            isPort = 0;
            break;
        }
    }

    if (isPort) {

        peer->mode  = SERVER_TCP;
        peer->port  = atoi(strPort);

        if (peer->port <= 0) {
            printf("ERROR: Improper port to use %d!\n", peer->port);
            ERROR2("[DS]: Improper port %d to use", peer->port);
            return EXIT_ABORT;
        }

        DEBUG2("[DS]: TCP Server mode, port %d\n", peer->port);

    } else {

        /*sprintf(tmp, "Unix socket Server mode. Use socket %s\n", strPort);
        logger(L_DBG, tmp);

        if ((ret = openSocketPort(SERVER_UX, -1, strPort)) < 0) {
            return -1;
        }
        peer->mode = SERVER_UX;
        peer->portStr = stringNew(strPort);
        */
        printf("ERROR: incorrect port\n");
        return EXIT_ABORT;

    }
    return EXIT_OK;
}

//
// fill ConnectInfo structure, return EXIT_ABORT in unsuccessful
//
static int addPeer(char * portIn)
{

    if (portIn == NULL) {
        return EXIT_ABORT;
    }
    INFO2("[DS]: addPeer() %s", portIn);

    char* strPort;

    // device should be in format:

    // deprecated
    // 1. (AT mode)     /dev/something                            used with AT mode commands

    // 1. (AT mode)     rfcomm:00:12:EF:32:21:1A:xx (xx is integer from 1 to 32)
    // 2. (Server mode) socket:NNNN or                             used with java client
    //                  socket:/path/to/socket                    used with java client (SERVER_UX, not used)
    // 3. (Server mode) bluetooth:NN or bluetooth (port will be = DEFAULT_BT_CHANNEL)     used with java client
    // 4. (Server mode) web:NNNN                            built-in web server
    // 5. (Server mode) cmxml:NNNN                            XML services interface
    // 6. local:/some/path                                    used with java client, like 1, but no AT commands
    // 7. ilirc:/dev/something                                 like 1, but no AT commands
    // 8. stdio                                                used with java client, like 1, but no AT commands
    // 9. avahi                                                skipped here


    ConnectInfo* peer = allocPeer();

    if ((strPort = strstr(portIn, INET_SOCKET)) != NULL) {

        strPort += strlen(INET_SOCKET);
    
        if (addTcpPeer(peer, strPort) == EXIT_ABORT) {
            free(peer);
            return EXIT_ABORT;
        }
    
    } else if ((strPort = strstr(portIn, PEER_TCP)) != NULL) {

        strPort += strlen(PEER_TCP);
    
        if (addTcpPeer(peer, strPort) == EXIT_ABORT) {
            free(peer);
            return EXIT_ABORT;
        }
    
    } else if ((strPort = strstr(portIn, ILIRC_SOCKET)) != NULL) {

        strPort += strlen(ILIRC_SOCKET);

        peer->mode  = CLIENT_ILIRC;
        peer->portStr = stringNew(strPort);

        DEBUG2("[DS]: Unix socket client mode, socket %s\n", peer->portStr->str);

    } else if ((strPort = strstr(portIn, BT_SOCKET)) != NULL) {

        strPort += strlen(BT_SOCKET);

        peer->mode = SERVER_BT;
        if (strstr(strPort, ":") == NULL) { // just "bluetooth
            peer->port = DEFAULT_BT_CHANNEL;
        } else {
            strPort++;
            peer->port = atoi(strPort);
        }
        DEBUG2("[DS]: Bluetooth Sockets Server mode, channel %d", peer->port);

    } else if ((strPort = strstr(portIn, L2CAP_SOCKET)) != NULL) {

        #ifdef USE_L2CAP
        strPort += strlen(L2CAP_SOCKET);

        peer->mode = SERVER_L2CAP;
        if (strstr(strPort, ":") == NULL) { // just "l2cap"
            peer->port = DEFAULT_L2CAP_PORT;
        } else {
            peer->port = atoi(strPort);
        }
        DEBUG2("[DS]: L2CAP Server mode, port %d", peer->port);

        #endif

    } else if ((strPort = strstr(portIn, UNIX_SOCKET)) != NULL) {

        strPort += strlen(UNIX_SOCKET);

        peer->mode    = CLIENT_NOAT;
        peer->portStr = stringNew(strPort);

        DEBUG2("[DS]: Serial Client mode (no AT). Use device %s", peer->portStr->str);

    } else if ((strPort = strstr(portIn, STDIN_STREAM)) != NULL) {

        peer->mode = SERVER_STDIN;

    } else if ((strPort = strstr(portIn, RFCOMM_DEVICE)) != NULL) {

        if (!(strlen(portIn) == strlen(RFCOMM_DEVICE) + BT_ADDR_LEN + 3    || // 00:12:EF:32:21:1A:p
                strlen(portIn) == strlen(RFCOMM_DEVICE) + BT_ADDR_LEN + 4) || // 00:12:EF:32:21:1A:pp
                portIn[strlen(RFCOMM_DEVICE)] != ':' ||
                portIn[strlen(RFCOMM_DEVICE) + BT_ADDR_LEN + 1] != ':') {
            printf("ERROR: Improper connect string !\n");
            free(peer);
            return EXIT_ABORT;
        }

        char sBtAddr[18];
        strncpy(sBtAddr,portIn + strlen(RFCOMM_DEVICE) + 1,17);
        sBtAddr[17] = '\0';

        peer->mode  = CLIENT_RFCOMM;
        peer->port  = atoi(portIn + strlen(RFCOMM_DEVICE) + BT_ADDR_LEN + 2);
        peer->portStr = stringNew(sBtAddr);

        DEBUG2("[DS]: Serial Client mode, device %s : %d", peer->portStr->str, peer->port);

    } else if ((strPort = strstr(portIn, WEB_SOCKET)) != NULL) {

        strPort += strlen(WEB_SOCKET);

        // check is it port
        char * ch = strPort;
        int isPort = 1;
        while (*ch != '\0') {
            if (isdigit(*ch)) {
                ch++;
            } else {
                isPort = 0;
                break;
            }
        }

        if (isPort) {

            peer->mode  = SERVER_WEB;
            peer->port  = atoi(strPort);
            if (peer->port <= 0) {
                printf("ERROR: Improper port to use !\n");
                ERROR2("[DS]: Improper port %d to use", peer->port);
                free(peer);
                return EXIT_ABORT;
            }

            DEBUG2("[DS]: Web Server mode. Use port %d", peer->port);

        } else {
            logger(L_ERR,"[DS]: can not determine web server port");
            free(peer);
            return EXIT_ABORT;
        }

    } else if ((strPort = strstr(portIn, CMXML_SOCKET)) != NULL) {

        strPort += strlen(CMXML_SOCKET);

        // check is it port
        char * ch = strPort;
        int isPort = 1;
        while (*ch != '\0') {
            if (isdigit(*ch)) {
                ch++;
            } else {
                isPort = 0;
                break;
            }
        }

        if (isPort) {

            peer->mode  = SERVER_CMXML;
            peer->port  = atoi(strPort);

            DEBUG2("[DS]: XML Server mode. Use port %d", peer->port);

        } else {
            logger(L_ERR,"[DS]: can not determine XML server port");
            free(peer);
            return EXIT_ABORT;
        }

    } else if ((strPort = strstr(portIn, AT_DEVICE)) != NULL) {

        strPort += strlen(AT_DEVICE);

        peer->mode  = CLIENT_AT;
        peer->portStr = stringNew(strPort);
        DEBUG2("[DS]: Serial Client mode. Use device %s", peer->portStr->str);

    } else if (strstr(portIn, AVAHI_USE)) {
       
        // do nothing
        logger(L_DBG, "[DS]: Avahi flag is ON");
        freePeer(peer);
        return EXIT_OK;
       
    } else {

        peer->mode  = CLIENT_AT;
        peer->portStr = stringNew(portIn);
        DEBUG2("[DS]: Default: use serial Client mode. Use device %s", peer->portStr->str);
    }

    _connections = listSingleAppend(_connections, peer);

    return EXIT_OK;
}

void connectNotify()
{
    sendToFrontEnd("Connected");
    sendEventToExecutor(ID_EVT_CONNECT);
}

static boolean_t needStoreState()
{
    int connCnt = 0;
    
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* peer = (ConnectInfo*) list->data;

        if (peer->mode == SERVER_WEB || 
            peer->mode == SERVER_CMXML) {
            connCnt += 2;   // need to store state in any case if WEB/CMXML is used
        } else if (peer->mode == SERVER_TCP || 
                   peer->mode == SERVER_BT  ||
                   peer->mode == CLIENT_NOAT||
                   peer->mode == SERVER_UX  ||
                  peer->mode ==  SERVER_L2CAP) {
            connCnt += 1;
        }
        if (connCnt > 1) {
            break;
        }
        list = listSingleNext(list);
    }
    return (connCnt > 1 ? BOOL_YES : BOOL_NO);
}

int definePeers()
{
    DEBUG2("[DS]: definePeers");
    char* peers = getDevice();

    // split peers by ','
    char *sep;
    char* peer = peers;
    int ret = EXIT_ABORT;

    while ((sep = index(peer,','))) {

        *sep = '\0';

        INFO2("[DS]: definePeers() peer %s", peer);
        int ret1 = addPeer(peer);
        if (ret1 != EXIT_ABORT) {
            ret = EXIT_OK;
        }

        peer = sep+1;

    }

    int ret1 = addPeer(peer);
    if (ret1 != EXIT_ABORT) {
        ret = EXIT_OK;
    }

    free(peers);
    
    if (needStoreState()) {
        initState();
    }

    DEBUG2("[DS]: definePeers done %d (peers #%d)", ret, listSingleLength(_connections));
    return ret;
}

int openPeers()
{
    DEBUG2("[DS]: openPeers %d", listSingleLength(_connections));

    int ret = EXIT_NOK;
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* peer = (ConnectInfo*) list->data;
        
        if (peer->state == PEER_DISCONNECTED) {
            INFO2("[DS]: Open peer connect mode %d", peer->mode);

            int ret1 = _peerHandlers[peer->mode].openConnection(peer);

            if (peer->state != PEER_DISCONNECTED) {  // for TCP (etc.) connection still can be in DISCONNECT
                INFO2("[DS]: Open peer connect mode %d fdescriptor %d", peer->mode, _peerHandlers[peer->mode].descriptor(peer));
            }

            if (ret1 != EXIT_NOK) {
                ret = ret1;
            }
        } else {
            DEBUG2("[DS]: openPeers peer with mode %d have improper state %d", peer->mode, peer->state);
        }
        list = listSingleNext(list);
    }
    return ret;
}

void closePeers(int final)
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* peer = (ConnectInfo*) list->data;

        _peerHandlers[peer->mode].closeConnection(peer, final);

        list = listSingleNext(list);
    }
}

int disconnectPeersInternal(ConnectInfo* peer)
{
    int canForceExit = 0;

    dMessage* dm = (dMessage*) malloc(sizeof(dMessage));
    dm->type  = DM_SET;
    dm->size  = 16;
    dm->value = strdup(CMD_STR_DISCONNECT);
    dm->file  = NULL;

    if (peer->mode == SERVER_BT) {

        if (getIViewer() || getBemused()) { 
            canForceExit = 1;
        } else {
            logger(L_INF, "[DS]: Got exit event: send disconnect message");
            if (btsppWrite(peer, dm) != EXIT_OK) {
                canForceExit = 1;
            }
        }

    } else if (peer->mode == SERVER_TCP) {

        if (getIViewer() || getBemused()) { 
            canForceExit = 1;
        } else {
            logger(L_INF, "[DS]: Got exit event: send disconnect message");
            if (socketWrite(peer, dm) != EXIT_OK) {
                canForceExit = 1;
            }
        }
        
    } else if (peer->mode == SERVER_UX) {

        logger(L_INF, "[DS]: Got exit event: send disconnect message");
        if (socketWrite(peer, dm) != EXIT_OK) {
            canForceExit = 1;
        }

    } else if (peer->mode == CLIENT_RFCOMM ||
               peer->mode == CLIENT_AT) {

        int fd = _peerHandlers[peer->mode].descriptor(peer);
        if (fd >= 0) {
            sendCMER(fd, CMER_OFF);
        }
        canForceExit = 1;

    } else if (peer->mode == SERVER_L2CAP) {

        #ifdef USE_L2CAP
        if (l2capWrite(peer, dm) != EXIT_OK) {
            canForceExit = 1;
        }
        #endif

    } else {
        canForceExit = 1;
    }

    free(dm->value);
    free(dm);

    DEBUG2("[DS]: disconnectPeersInternal mode=%d can force exit=%d", peer->mode, canForceExit);
    return canForceExit;
}

// retuns 1/-1
// return code is used in case of exit only
int disconnectPeers(void)
{
    DEBUG2("[DS]: disconnectPeers");
    int ret = 1;
    SingleList* list = _connections;
    while (list) {

        ConnectInfo* peer = (ConnectInfo*) list->data;

        int ret1 = disconnectPeersInternal(peer);
        if (ret1 != 1) {
            ret = ret1;
        }
        list = listSingleNext(list);
    }
    DEBUG2("[DS]: disconnectPeers can force exit=%d", ret);
    return ret;
}

// Support single socket connection + web server at once for now
int setupPeers(void)
{
    DEBUG2("[DS]: setupPeers");
    int ret = -1;
    SingleList* list = _connections;
    while (list) {

        ConnectInfo* peer = (ConnectInfo*) list->data;
        
        int retS = _peerHandlers[peer->mode].setupConnection(peer);

        if (retS != -1) {
            ret = retS;
        }
        list = listSingleNext(list);
    }
    return ret;
}

//
// Default peer writer
//
static int writePeer(int fd, const char* buffer, int n)
{
    int nbytes = write(fd,buffer,n);
    if (nbytes < 0) {
        ERROR2("[DS]: writePeer() error %d",errno);
        errnoDebug("[DS]: writePeer() write ",errno);  // testing debug
    }
    return nbytes;
}

static int peerWrite(ConnectInfo* peer, const dMessage* msg)
{
    const char* command = msg->value;
    int count           = msg->size;

    if (!command || count <= 0) {
        return EXIT_OK;
    }

    if (strcmp("End();",command) == 0) {  // used only for WEB/CMXML
        return EXIT_OK;
    }

    int fd = _peerHandlers[peer->mode].descriptor(peer);
    if (fd < 0) {
        return EXIT_NOK;
    }

    return (writePeer(fd, command, count) > 0 ? EXIT_OK : EXIT_NOK);

}

//
// Sync peer after connect
//
static void syncPeer(ConnectInfo* peer)
{
    if (peer->mode == SERVER_TCP  ||
        peer->mode == SERVER_BT   ||
        peer->mode == SERVER_UX   ||
        peer->mode == CLIENT_NOAT ||
        peer->mode == SERVER_L2CAP) {
        
        ERROR2("[DS]: syncPeer %d", peer->mode);

        static struct {
           string_t* (*hooks[6]) (int) ; // CF,TX,LI,FM,WM,EF
        } _renderHooks[] = {
           //                  CF                  TX                  LI                  FM    WM                  EF 
           /* SERVER_TCP+  */{{renderCtrlForm,     renderTextForm,     renderListForm,     NULL, renderWmanForm,     renderEditForm}}
        };

         
        int f   = curForm() - 1;
        INFO2("[DS]: syncPeer form %d", f);
    
        if (_renderHooks[0].hooks[f]) {
            string_t* content = _renderHooks[0].hooks[f](peer->port);
	        if (content) {
                
                //char buf[256];
                //strncpy(buf,content->str,255);
                //buf[255] = '\0';
                //INFO2("[DS]: syncPeer sync content %s", buf);
                
                int fd = _peerHandlers[peer->mode].descriptor(peer);
                if (fd >= 0) {
                    INFO2("[DS]: syncPeer write to peer %d %d", f, fd);
                    writePeer(fd, content->str, content->len);
                }
	            stringFree(content, BOOL_YES);
	        }
        }
        
        if (curForm() == CF) {  // take care about cover 

            int fd = _peerHandlers[peer->mode].descriptor(peer);
            if (fd >= 0) {
                
                string_t* page = renderCtrlFormCover();
                writePeer(fd, page->str, page->len);
	            stringFree(page, BOOL_YES);

                const char* nc = cfNamedCover();
                const char* cv = cfCover();
                
                if (!nc && cv) {
                
                    string_t* page = stringNew("Set(cover,noname,");
                    stringAppend(page, cv);
                    stringAppend(page, ");");
               
                    eMessage* em = (eMessage*) malloc(sizeof(eMessage));
                    em->type  = EM_STRING;
                    em->value = strdup(page->str);
                    
                    stringFree(page, BOOL_YES);
                    
                    sendToExecutor(em);
                }
            }
        }
    }
}

//
// Default peer reader
//
int readPeer(int fd, char* buffer, int max)
{
    int nbytes = read(fd, buffer, max);
    if (nbytes < 0)  { // Read error
        //ERROR2("[DS]: readPeer() error %d",errno);
        errnoDebug("[DS]: readPeer() read ",errno);  // testing debug
    } else if (nbytes == 0) {
        DEBUG2("[DS]: readPeer() EOF");
    }

    //DEBUG2("[DS]: readPeer() got >%s< %d", buffer, nbytes);
    return nbytes;
}

//
// TCP peer reader with quirk for iViewer
//
static int readIVPeer(int fd, char* buffer, int max)
{
     //DEBUG2("[DS]: readIVPeer()");
    int nbytes = read(fd, buffer, max);

    
    if (nbytes < 0)  { // Read error
        ERROR2("[DS]: readIVPeer() error %d",errno);
        errnoDebug("[DS]: readIVPeer() read ",errno);  // testing debug
    } else if (nbytes == 0) {
        DEBUG2("[DS]: readIVPeer() EOF");
    } else {
        boolean_t heartbeat = BOOL_NO;
        char *hptr = buffer;
        //DEBUG2("[DS]: readIVPeer() got %s", buffer);
        
        //while ((hptr = strstr(hptr, "h=0"))) {
        if (strstr(hptr, "h=0")) {
            // do not erase it from read data, because it used as disconnect timer in cfg.files
            /*
            *hptr = '\r';
            hptr++;
            *hptr = '\r';
            hptr++;
            *hptr = '\r';
            */

            heartbeat = BOOL_YES;
        }

        hptr = buffer;
        while ((hptr = strchr(hptr, '\3'))) {  // end-of-text marker in CommandFusion
            // replace \3 separator to \r
            *hptr = '\r';
        }

        if (heartbeat) {
            DEBUG2("[DS]: readIVPeer() send iViever heartbeat");
            writeIViewerHeartbeat(fd);
        }
    }

    //DEBUG2("[DS]: readIVPeer() got >%s< %d", buffer, nbytes);
    return nbytes;
}

//
// CLIENT_NOAT peer reader
// in IR communication a lot of 'empty' chars (= -1) could be added
//
static int readIRPeer(int fd, char* buffer, int max)
{
    int nbytes = read(fd, buffer, max);
    if (nbytes < 0)  { // Read error
        ERROR2("[DS]: readIRPeer() error %d",errno);
        errnoDebug("[DS]: readIRPeer() read ",errno);  // testing debug
    } else if (nbytes == 0) {
        DEBUG2("[DS]: readIRPeer() EOF");
    } else {
        buffer[nbytes] = '\0';
        char * k2 = buffer;
        while (*k2 != '\0') {
            if (*k2 == -1) {
                *k2 = '\r';
            }
            k2++;
        }
    }

    //DEBUG2("[DS]: readIRPeer() got >%s< %d", buffer, nbytes);
    return nbytes;
}

//
// BT peer reader with quirk for Bemused
// must read VOLM command with an additional value byte
// SHFL/REPT _can_ follow with an additional byte - so don not bother about them
//
static int readBmPeer(int fd, char* buffer, int max)
{
    int nbytes = read(fd, buffer, max);
    if (nbytes < 0)  { // Read error
        ERROR2("[DS]: readBmPeer() error %d",errno);
        errnoDebug("[DS]: readBmPeer() read ",errno);  // testing debug
    } else if (nbytes == 0) {
        DEBUG2("[DS]: readBmPeer() EOF");
    } else {
        boolean_t heartbeat = BOOL_NO;
        char *hptr = NULL;
        while ((hptr = strstr(buffer, "CHCK"))) {
            // erase it from read data
            *hptr = '\r';
            hptr++;
            *hptr = '\r';
            hptr++;
            *hptr = '\r';
            hptr++;
            *hptr = '\r';

            heartbeat = BOOL_YES;
        }

        if (heartbeat) {
            DEBUG2("[DS]: readBmPeer() send Bemused heartbeat");
            writeBemusedHeartbeat(fd);
        }

        // hack to make it work correctly with Bemused clients
        char c = '\r';
        if (nbytes >=4 &&
                (strncmp((buffer+nbytes-4), "VOLM", 4) == 0 //|| // read only VOLM without value to set
                 /*strncmp((buffer+nbytes-4), "SHFL", 4) == 0 ||
                 strncmp((buffer+nbytes-4), "REPT", 4) == 0*/)) {

            int ret;
            if ((ret = bt_readchar(fd,&c,500000)) < 0) {
                DEBUG2("[DS]: parseCommand: Bemused hack: read < 0");
            } else {
                sprintf(tmp, "[DS]: parseCommand: Bemused hack: read >%c<", c);
                logger(L_DBG,tmp);
                buffer[nbytes] = c;
                nbytes++;
            }
        }

    }

    //DEBUG2("[DS]: readBmPeer() got >%s< %d", buffer, nbytes);
    return nbytes;
}

static int readPeersInternal(ConnectInfo* peer, string_t* buffer)
{
    //INFO2("[DS]: readPeersInternal peer mode %d", peer->mode);
    char buf[MAXCMDLEN];

    int len = 0;
    //printf("read_command\n");

    if (peer->mode == SERVER_STDIN) {

        len = readStdin(buf,MAXCMDLEN);  // returns EOF in case of error and EOF
        if (len > 0 && len != EOF) {
            DEBUG2("[DS]: read_command SERVER_STDIN %d %s\n", len, buf);

            stringAppend(buffer,"\r");  // separator
            stringAppend(buffer,buf);
        }
    } else if (peer->mode == SERVER_WEB || peer->mode == SERVER_CMXML) {

        len = checkWebPort(buf, MAXCMDLEN);  // returns EOF in case of error and EOF
        if (len > 0 && len != EOF) {
            stringAppend(buffer,"\r");  // separator
            stringAppend(buffer,buf);
        }

    } else {
        
        /* use select() for others
        int fd = _peerHandlers[peer->mode].descriptor(peer);
        if (peer->state == PEER_CONNECTED && fd >= 0) {
        
            char* p = buf;

            while (len < MAXCMDLEN) {
                char c;
                int ch = bt_readchar(fd,&c,100);
                
                INFO2("[DS]: readPeersInternal read from peer with mode %d got %d", peer->mode, ch);
                
                if (ch == EOF) {
                    INFO2("[DS]: readPeersInternal EOF peer with mode %d", peer->mode);
                    buf[0] = 0;
                    peer->state = PEER_DISCONNECTED;
                    break;
                } else if (ch == EOF-1) {
                    break;
                } else if (ch >= 0 && (c == '\r' || c == ';')) {
                    break;
                } else  if (ch >= 0 && c != '\r' && c != ';') {
                    *p++ = c;
                    len++;
                }
            }

            buf[len] = '\0';
            stringAppend(buffer,"\r");  // separator
            stringAppend(buffer,buf);
        }
        */
        
    }
    return len;
}

static int doReadPeer(ConnectInfo* peer, string_t* buffer, int fd)
{
    char buf[MAXCMDLEN];
    
    //DEBUG2("[DS]: doReadPeer() ready to read from fd %d (peer mode %d)", fd, peer->mode);

    int rc = 0;
    
    if (peer->mode == SERVER_TCP && getIViewer()) {
        rc = readIVPeer(fd, buf, MAXCMDLEN);
    } else if (peer->mode == SERVER_BT && getBemused()) {
        rc = readBmPeer(fd, buf, MAXCMDLEN);
    } else if (peer->mode == CLIENT_NOAT) {
        rc = readIRPeer(fd, buf, MAXCMDLEN);
    } else if (peer->mode == CLIENT_RFCOMM || peer->mode == CLIENT_AT) {
        rc = atRead(peer, buf, MAXCMDLEN);
    } else {
        rc = readPeer(fd, buf, MAXCMDLEN);
    }

    if (rc <= 0) {  // EOF or error
        DEBUG2("[DS]: doReadPeer() EOF or error fd %d (peer mode %d), close connection", fd, peer->mode);
        _peerHandlers[peer->mode].closeConnection(peer, 0);
        return EOF;
    } else {
        stringAppend(buffer,"\r");  // separator
        stringAppendLen(buffer,buf,rc);
    }

    return 1;
}

int readPeers(string_t* buffer)
{
    //DEBUG2("[DS]: readPeers()");
    
    fd_set read_fds;
    FD_ZERO(&read_fds);

    struct timeval tv;
    tv.tv_sec  = 0;
    tv.tv_usec = 100;

    int max_fd = -1;

    SingleList* list = _connections;
    while (list) {
        ConnectInfo* peer = (ConnectInfo*) list->data;

        if ((peer->state == PEER_CONNECTED && 
             !(peer->mode == SERVER_STDIN || peer->mode == SERVER_WEB || peer->mode == SERVER_CMXML)) ||
            peer->state == PEER_WAIT_ACCEPT) {

            int fd = _peerHandlers[peer->mode].descriptor(peer);
            //DEBUG2("[DS]: readPeers() peer with mode %d connected %s state (fd=%d)", peer->mode, 
            //             (peer->state == PEER_CONNECTED ? "CONNECTED" : "WAIT_ACCEPT"),fd);
                         
            if (fd >= 0) {
                FD_SET(fd, &read_fds);
            //} else {  web, xml can have that 
            //    DEBUG2("[DS]: readPeers() peer with mode %d improper descriptor", peer->mode); 
            }
            if (max_fd < fd) {
                max_fd = fd;
            }
        }
        list = listSingleNext(list);
    }

    if (max_fd >= 0) {  // have some peers

        int rc = select(max_fd + 1, &read_fds, NULL, NULL, &tv);
        
        //if (rc != 0) {
        //    DEBUG2("[DS]: readPeers() select()=%d, maxfd=%d",rc,max_fd);
        //}
        
        if (rc == 0) {          // timeout
            //DEBUG2("[DS]: readPeers() no data from select()");
        } else if (rc < 0) {    // some error
            //DEBUG2("[DS]: readPeers() error %d", errno);
            errnoDebug("[DS]: readPeers() select ",errno);  // testing debug
        } else {
            //DEBUG2("[DS]: readPeers() got data from select()");
            
            list = _connections;
            while (list) {
             
                ConnectInfo* peer = (ConnectInfo*) list->data;
                
                int fd = _peerHandlers[peer->mode].descriptor(peer);
                if (fd >= 0) {
                
                    //DEBUG2("[DS]: readPeers() %d descriptor on peer with mode %d", fd, peer->mode);

                    if (FD_ISSET(fd, &read_fds)) {

                        if (peer->state == PEER_CONNECTED) {
                            //DEBUG2("[DS]: readPeers() peer with mode %d readable", peer->mode);
                            int rr = doReadPeer(peer, buffer, fd);
                            if (rr == EOF) {
                                int cnum = countConnections(0);

                                // no more connections
                                if (cnum == 0) { 
                                     DEBUG2("[DS]: readPeers() disconnect notify");
                                     return EOF;
                                }
                            }
                        } else if (peer->state == PEER_WAIT_ACCEPT) {

                            DEBUG2("[DS]: readPeers() accept connection peer with mode %d", peer->mode);
                            if (_peerHandlers[peer->mode].acceptConnection(peer) == EXIT_OK) {
                                int cnum = countConnections(1);

                                // do this only on first connect
                                if (cnum == 1) {
                                    DEBUG2("[DS]: readPeers() connectNotify");
                                    connectNotify();
                                } else {
                                    freeCachedData();
                                    syncPeer(peer);
                                }
                            }
                        }
                    //} else {
                    //    DEBUG2("[DS]: readPeers() no FD_ISSET on peer with mode %d", peer->mode);
                    }
                //} else {
                //   DEBUG2("[DS]: readPeers() no descriptor on peer with mode %d", peer->mode);
                }
                list = listSingleNext(list);
            }
        }
    //} else {
    //    DEBUG2("[DS]: readPeers() no peers to read");
    }

    //DEBUG2("[DS]: readPeers() special cases");
    // special cases
    list = _connections;
    while (list) {
        ConnectInfo* peer = (ConnectInfo*) list->data;
        if (peer->state == PEER_CONNECTED) {
            int rc = readPeersInternal(peer, buffer);
            if (rc < 0 || rc == EOF) {  // EOF or error
                DEBUG2("[DS]: readPeers() EOF or error special case (peer mode %d), close connection", peer->mode);
                _peerHandlers[peer->mode].closeConnection(peer, 0);
                return EOF;
            }
        // Only stdin, web and cmxml goes here. None of them can have state PEER_WAIT_ACCEPT
        //} else if (peer->state == PEER_WAIT_ACCEPT) {
        //    //DEBUG2("[DS]: readPeers() TODO aa1");
        }
        list = listSingleNext(list);
    }

    //DEBUG2("[DS]: readPeers() return %s (%ld)", buffer->str, buffer->len);
    return buffer->len;
}

int writePeers(dMessage* dm)
{
    int ret = EXIT_NOK;
    
    if (needStoreState()) {
        updateState(dm);
    }

    SingleList* list = _connections;
    while (list) {
        ConnectInfo* peer = (ConnectInfo*) list->data;

        if (peer->state == PEER_CONNECTED) {
            INFO2("[DS]: write to peer %d", peer->mode);

            int ret1 = (_peerHandlers[peer->mode].writeConnection != NULL ?
                        _peerHandlers[peer->mode].writeConnection(peer, dm) : EXIT_NOK);

            if (ret1 != EXIT_NOK) {
                ret = ret1;
            }
        }
        list = listSingleNext(list);
    }

    //INFO2("[DS]: write result %s", (ret == EXIT_NOK? "NOK": "OK"));
    return ret;
}

static int writeByteInternal(int fd, int byte)
{
    unsigned char byte2write[2];
    byte2write[0] = (unsigned char) byte;
    byte2write[1] = '\0';

    if (write(fd, byte2write, 1) < 0) {
        logger(L_ERR, "error writing bytes");
        return EXIT_NOK;
    }
    return EXIT_OK;
}

static int peerWriteBytes(ConnectInfo* peer, const char* command)
{
    int fd = _peerHandlers[peer->mode].descriptor(peer);
    if (fd < 0) {
        logger(L_DBG,"[DS]: peerWriteBytes() no connection data");
        return EXIT_NOK;
    }

    // send command
    if (fd >= 0 && command && command[0]) {

        char byteStr[MAXCKPDLEN];
        memset(byteStr,0,MAXCKPDLEN);

        strncpy(byteStr,command,MAXCKPDLEN-1);

        DEBUG2("[DS]: peerWriteBytes >%s<", byteStr); 

        char* bStr = strtok(byteStr,",");
        while (bStr != NULL) {

        //DEBUG2("[DS]: Next byte is >%s<", bStr); 

            char bStripped[4];

            while (*bStr == ' ') {
                bStr++;
            }
            int i = 0;
            while (*bStr != ' ' && i < 3) {  // 0 < ... < 256
                bStripped[i] = *bStr;
                bStr++;
                i++;
            }
            bStripped[i] = '\0';

            //DEBUG2("[DS]: Next byte is >%s<", bStripped); 

            if (writeByteInternal(fd, atoi(bStripped)) != EXIT_OK) {
                logger(L_DBG,"[DS]: Fails in peerWriteBytes()");
                return EXIT_NOK;
            }

            bStr = strtok(NULL,",");
        }
    }
    //logger(L_DBG, "peerWriteBytes EXIT");
    return EXIT_OK;
}

int writeBytesPeers(char* command)
{
    int ret = EXIT_NOK;
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* peer = (ConnectInfo*) list->data;
        if (peer && 
            peer->state == PEER_CONNECTED &&
            _peerHandlers[peer->mode].writeBytesConnection != NULL) {

            INFO2("[DS]: write bytes to peer %d", peer->mode);
            int ret1 = _peerHandlers[peer->mode].writeBytesConnection(peer, command);

            if (ret1 != EXIT_NOK) {
                ret = ret1;
            }
        }
        list = listSingleNext(list);
    }
    return ret;
}

//
// In case of WEB/CMXML it is enougn to send name of file, not full content
//
int writeFilePeers(dMessage* dm)
{
    INFO2("[DS]: DM_SETFILE");
    
    if (needStoreState()) {
        updateState(dm);
    }

    int  size = 0;
    char* buf = NULL;
    dMessage* dm2 = NULL;
    boolean_t evaluated = BOOL_NO;

    SingleList* list = _connections;
    while (list) {

        ConnectInfo* peer = (ConnectInfo*) list->data;

        if (peer->state == PEER_CONNECTED) {
            INFO2("[DS]: write file to peer %d", peer->mode);

            if (peer->mode == SERVER_WEB || peer->mode == SERVER_CMXML) {

                // nothing, state stored by updateState(...) above

            } else {

                if (evaluated == BOOL_NO) {

                    evaluated = BOOL_YES;

                    // read data from file
                    buf = readFromFile(dm->value, dm->file, &size);

                    INFO2("[DS]: got from file %d bytes", size);

                    if (buf && size > 0) {

                        dm2 = (dMessage*) malloc(sizeof(dMessage));

                        dm2->type  = DM_SET;
                        dm2->value = buf;
                        dm2->file  = NULL;
                        dm2->size  = size;
                    }

                }

                if (dm2) {
                    if (_peerHandlers[peer->mode].writeConnection != NULL) {
                        _peerHandlers[peer->mode].writeConnection(peer, dm2);
                    }
                }
            }
        }
        list = listSingleNext(list);
    }
    freeDMessage(dm2);
    return EXIT_OK;
}

int writeCKPD(dMessage* dm)
{
    logger(L_DBG, "[DS]: Send CKPD");

    SingleList* list = _connections;
    while (list) {

        ConnectInfo* peer = (ConnectInfo*) list->data;

        if (peer->state == PEER_CONNECTED &&
            (peer->mode == CLIENT_RFCOMM || peer->mode == CLIENT_AT)) {

            int fd = _peerHandlers[peer->mode].descriptor(peer);
            if (fd >= 0) {
                sendSeq(fd, (char*) dm->value);
            }
        }
        list = listSingleNext(list);
    }
    return EXIT_OK;
}

int writeCMER(dMessage* dm)
{
    logger(L_DBG, "[DS]: Send CMER");

    SingleList* list = _connections;
    while (list) {

        ConnectInfo* peer = (ConnectInfo*) list->data;

        if (peer->state == PEER_CONNECTED &&
            (peer->mode == CLIENT_RFCOMM || peer->mode == CLIENT_AT)) {

            int fd = _peerHandlers[peer->mode].descriptor(peer);
            if (fd >= 0) {
                sendCMER(fd, dm->size);
            }
        }
        list = listSingleNext(list);
    }
    return EXIT_OK;
}

//
// returns EXIT_OK if at least one connection exists
//
int connected()
{
    SingleList* list = _connections;
    while (list) {

        ConnectInfo* peer = (ConnectInfo*) list->data;
        if (peer->state == PEER_CONNECTED) {
            return EXIT_OK;
        }
        list = listSingleNext(list);
    }
    return EXIT_NOK;
}

//
// Send heartbeat message to all TCP connection.
// Suppose we have only one connection to iViewer
//
static int socketWriteByte(int fd, int byte)
{
    unsigned char byte2write[2];
    byte2write[0] = (unsigned char) byte;
    byte2write[1] = '\0';

    if (write(fd, byte2write, 1) < 0) {
        logger(L_ERR, "error writing byte to socket");
        return EXIT_NOK;
    }
    return EXIT_OK;
}

static void writeIViewerHeartbeat(int fd)
{
    // reply message is h=1\03
    socketWriteByte(fd, 104); // h
    socketWriteByte(fd, 61);  // =
    socketWriteByte(fd, 49);  // 1
    socketWriteByte(fd, 3);   // \03
}

static void writeBemusedHeartbeat(int fd)
{
    // reply message to "CHCK" is "Y"
    writePeer(fd, "Y", 1);
}

void writeHeartbeat(ConnectInfo* peer)
{
    int fd = _peerHandlers[peer->mode].descriptor(peer);
    if (fd >= 0) {
        writeIViewerHeartbeat(fd);
    }
}

void sendIViewerHeartbeat(void)
{
    SingleList* list = _connections;
    while (list) {

        ConnectInfo* peer = (ConnectInfo*) list->data;
        if (peer->state == PEER_CONNECTED && 
            peer->mode == SERVER_TCP) {  // iViewer connection can be TCP only
            
            writeHeartbeat(peer);
        }
        list = listSingleNext(list);
    }
}

//
// returns EXIT_OK if there are only client peers and all peers unconnected
//
int needExit()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->state == PEER_CONNECTED) {
            return EXIT_NOK;
        }

        if (!(cn->mode == CLIENT_RFCOMM ||
              cn->mode == CLIENT_AT     ||
              cn->mode == CLIENT_ILIRC)) {
            return EXIT_NOK;
        }
        list = listSingleNext(list);
    }
    return EXIT_OK;
}

//
// returns EXIT_OK if there are web/cmxml peer exists
//
int needFinalizer()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == SERVER_WEB || cn->mode == SERVER_CMXML) {
            return EXIT_OK;
        }
        list = listSingleNext(list);
    }
    return EXIT_NOK;
}

//
// returns EXIT_OK if there are AT peer exists
//
int needAtMainMenuReturn()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == CLIENT_RFCOMM || cn->mode == CLIENT_AT) {
            return EXIT_OK;
        }
        list = listSingleNext(list);
    }
    return EXIT_NOK;
}

//
// returns EXIT_OK if there are server-mode peer exists
//
int isServerMode()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == SERVER_BT    ||
                cn->mode == SERVER_TCP   ||
                cn->mode == SERVER_WEB   ||
                cn->mode == SERVER_CMXML ||
                cn->mode == SERVER_UX    ||
                cn->mode == CLIENT_NOAT  ||
                cn->mode == SERVER_L2CAP) {
            return EXIT_OK;
        }
        list = listSingleNext(list);
    }
    return EXIT_NOK;
}

//
// returns EXIT_OK if there are server-mode (no WEB/CMXML) peer exists
//
int isServerModeNoWeb()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == SERVER_BT    ||
            cn->mode == SERVER_TCP   ||
            cn->mode == SERVER_UX    ||
            cn->mode == CLIENT_NOAT  ||
            cn->mode == SERVER_L2CAP) {
            return EXIT_OK;
        }
        list = listSingleNext(list);
    }
    return EXIT_NOK;
}

int isWebServer()
{
    return needFinalizer();
}

//
// returns EXIT_OK if there are at-mode peer exists
//
int isAtMode()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == CLIENT_RFCOMM ||
            cn->mode == CLIENT_AT     ||
            cn->mode == CLIENT_ILIRC) {
            return EXIT_OK;
        }
        list = listSingleNext(list);
    }
    return EXIT_NOK;
}

//
// returns EXIT_OK if there are CLIENT_RFCOMM/CLIENT_AT peer exists
//
int isAtModeDuplex()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == CLIENT_RFCOMM ||
            cn->mode == CLIENT_AT) {
            return EXIT_OK;
        }
        list = listSingleNext(list);
    }
    return EXIT_NOK;
}

//
// In iViewer mode returns TCP port (search for the first TCP connection)
//
int getIViewerTcpPort(void)
{
    int port = -1;
    
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == SERVER_TCP && cn->port >= 0) {  // SERVER_TCP handle also Unix (file) sockets
            return cn->port;
        }
        list = listSingleNext(list);
    }
    
    return port;
}

boolean_t checkActiveCall()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == CLIENT_RFCOMM) {
            if (rfcommCheckActiveCall(cn)) {
                return BOOL_YES;
            }
        } else if (cn->mode == CLIENT_AT) { 
            if (serialCheckActiveCall(cn)) {
                return BOOL_YES;
            }
        }
        list = listSingleNext(list);
    }
    return BOOL_NO;
}

boolean_t hasActiveCall()
{
    SingleList* list = _connections;
    while (list) {
        ConnectInfo* cn = (ConnectInfo*) list->data;

        if (cn->mode == CLIENT_RFCOMM) {
            if (rfcommHasActiveCall(cn)) {
                return BOOL_YES;
            }
        } else if (cn->mode == CLIENT_AT) { 
            if (serialHasActiveCall(cn)) {
                return BOOL_YES;
            }
        }
        list = listSingleNext(list);
    }
    return BOOL_NO;
}
