% make_csvs_from_colorcet.m
% 1. Requires latest colorcet.m from peterkovesi.com in same directory
% 2. Copy `mapnames` cell array to below from colorcet.m.
% 3. After running this in octave or MATLAB, follow instructions in CET_merge.py
%
% Generates CET_updates.py with `new_aliases` and `new_cetnames` for CSVs newly-generated by this script
% (aliases and cetnames for existing CSVs not created.)

% Copy `mapnames` from colorcet.m here:
mapnames = {};

aliases = containers.Map();
cetnames = containers.Map();

existing_csvs = arrayfun(@(f) f.name, dir('CET/*.csv'), 'UniformOutput', false);
for mni = 1:numel(mapnames)
    % first string is the CET- name. The rest are some of the aliases
    mapname = mapnames{mni}{1};
    CETname = ['CET-', mapname];
    othernames = mapnames{mni}(2:end);
    % don't keep names like D04
    othernames = othernames(cellfun(@isempty, regexp(othernames, '^[A-Z][0-9]+')));
    for ni = [1, 2]  % second time is s25
        if ni == 2 && isempty(regexp(mapname, '^C\d+')), break, end  % only do s25 for cyclic
        % Get the map. If cyclic, the second time get a map shifted by 0.25.
        switch ni
            case 1
                [map, descriptorname, description] = colorcet(mapname);
            case 2
                [map, descriptorname, description] = colorcet(mapname, 'shift', 0.25);
        end
        dnames = strsplit(descriptorname, ' ');  % Some descriptorname entries have multiple verions
        descriptorname = dnames{end};  % last one is the algorithmic name
        othernames = [othernames, dnames(1:end-1)];  % Others are aliases
        if ni == 2
            descriptorname = strrep(descriptorname, '_n256', '_n256_s25');
        end
        mapcsvname = [descriptorname, '.csv'];

        % If the map CSV file exists, go to the next map
        if ismember(mapcsvname, existing_csvs), continue, end;

        % Create the map
        csvwrite(fullfile('CET', mapcsvname), map, 'precision', '%0.5g');

        % Get the non-hyphenated, non-'_n256' name and update cetnames and aliases
        cleanname = strrep(strrep(descriptorname, '-', '_'), '_n256', '');
        if ni == 2
            CETname = [CETname, 's'];
        end
        cetnames(CETname) = cleanname;
        if ~isempty(othernames) && ni == 1
            aliases(cleanname) = othernames;
        end
    end
end

% Create CET_updates.py to use in CET_merge.py
fupdates = fopen('CET_updates.py', 'w');
fprintf(fupdates, 'new_aliases = dict(\n');
keys = aliases.keys();
for keyi = 1:numel(keys)
    key = keys{keyi};
    als = strrep(aliases(key), '-', '_');  % can't have hyphens in aliases
    if numel(als) == 0, continue, end
    fprintf(fupdates, '    %s = [', strrep(key, '-', '_'));
    for ali = 1:numel(als)
        fprintf(fupdates, '''%s'',', lower(als{ali}));
    end
    fprintf(fupdates, '],\n');
end
fprintf(fupdates, ')\n');
fprintf(fupdates, '\n');

fprintf(fupdates, 'new_cetnames = {\n')
keys = cetnames.keys();
for keyi = 1:numel(keys)
    key = keys{keyi};
    fprintf(fupdates, '    ''%s'': ''%s'',\n', key, cetnames(key));
end
fprintf(fupdates, '}\n');
fprintf(fupdates, '\n');

fclose(fupdates);

