/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

using namespace std;

#include <stdlib.h>
#include <stddef.h>
#include <inttypes.h>
#include <vector>
#include "wchar.h"

#include "Heap.h"
#include "VMState.h"
#include "VM.h"
#include "stdfuns.h"
#include "unpickler.h"

Value* unmarshal(void* vmptr,wchar_t* x, kint id)
{
    VMState* vm = (VMState*)vmptr;
    vector<Value*> done;
    kint i = read_int(vm,x);
    if (i!=id) {
	vm->kaya_rtsError(INVALID_MARSHALLING_ID);
    }
    kint h = read_int(vm,x);
    if (h!=getFnHash()) {
/*	cout << "marshal hash: " << h << endl;
	cout << "ftable hash: " << getFnHash() << endl; */
	vm->kaya_rtsError(INVALID_FUNCTION_TABLE_HASH);
    }

    Value* v = unpickle(vm, done, x, id);
    return v;
}

Value* unpickle(VMState* vm, vector<Value*>& done, wchar_t*& x, kint id)
{
    Value *v;

    switch(x[0]) {
    case 'I':
	v=un_int(vm,done,++x);
	return v;
    case 'S':
	v=un_string(vm,done,++x);
	return v;
    case 'U':
	v=un_union(vm,done,++x,id);
	return v;
    case 'F':
	v=un_closure(vm,done,++x,id);
	return v;
    case 'A':
	v=un_array(vm,done,++x,id);
	return v;
    case 'R':
	v=un_real(vm,done,++x);
	return v;
    case 'C':
	v=un_circle(vm,done,++x);
	return v;
    default:
	vm->kaya_rtsError(INVALID_VALUE);
	return MKVAL(NULL,KVT_NULL);
    }
}

kint read_int(VMState* vm, wchar_t* &x) {
    wchar_t buf[30]; // = (wchar_t*)GC_MALLOC_ATOMIC(sizeof(wchar_t)*20);
    kint i=-1;
    
    if (*x!='[') {
	vm->kaya_rtsError(INVALID_VALUE);
    }
    ++x; // Move past the initial [
    while(*x!=']' && i<30) {
	buf[++i]=*x;
	++x;
    }
    if (i == 50) {
	vm->kaya_rtsError(INVALID_VALUE);
    }
    buf[++i]='\0';
#ifdef KAYA_SHORT_INTS
    kint val = wcstoimax(buf,NULL,10);
#else
    kint val = wcstol(buf,NULL,10);
#endif
    //      = atoi(wctostr(buf));

    ++x;
    return val;
}

Value* un_int(VMState* vm,vector<Value*>& done,wchar_t* &x)
{
    // Expecting [num]
    return mkint2(vm,(void*)read_int(vm,x));
}

Value* un_real(VMState* vm,vector<Value*>& done,wchar_t* &x)
{
    // Expecting [num]
    wchar_t buf[100]; // = (wchar_t*)GC_MALLOC_ATOMIC(sizeof(wchar_t)*100); // Yeah, horrible, I know.
    kint i=-1;
    
    if (*x!='[') {
	vm->kaya_rtsError(INVALID_VALUE);
    }
    ++x; // Move past the initial [
    while(*x!=']' && i<100) {
	buf[++i]=*x;
	++x;
    }
    buf[++i]='\0';
    double val = wcstod(buf,NULL);

    ++x;
    return MKREAL(val);
}

Value* un_circle(VMState* vm,vector<Value*>& done,wchar_t* &x)
{
    if (*x!='[') vm->kaya_rtsError(INVALID_CIRCULAR_STRUCTURE);
    kint circid = read_int(vm,x);
//    cout << circid << "," << done.size() << endl;
    if (done.size()<(ukint)circid) 
	vm->kaya_rtsError(INVALID_CIRCULAR_STRUCTURE);
    // clone or we get some very odd effects - CIM
    return done[circid]->clone();
}

Value* un_string(VMState* vm,vector<Value*>& done,wchar_t* &x)
{
    // Expecting [num]string
//    cout << "Doing string, x is " << x << endl;

    if (*x!='[') vm->kaya_rtsError(INVALID_VALUE);
    ++x;
    kint len = read_int(vm,x);
    kint alloc = read_int(vm,x);
    if (alloc < len) {
      alloc = len;
    }
    
    wchar_t buf[len+1]; // = (wchar_t*)GC_MALLOC_ATOMIC(sizeof(wchar_t)*len+1);
    kint i=0;

    while(i<len) {
	buf[i++]=*x;
	++x;
    }
    buf[i]='\0';
    ++x;

//    cout << "Got string of length " << len << "," << buf << endl;
//    cout << "x is " << x << endl;
    Value* str = createString(vm,alloc);
    str->getString()->append(buf);
    return str;
}

Value* un_array(VMState* vm,vector<Value*>& done,wchar_t* &x, kint id)
{
//    cout << "Doing array " << endl;

    if (*x!='[') vm->kaya_rtsError(INVALID_VALUE);
    ++x;
    kint circid = read_int(vm,x);
    kint sz = read_int(vm,x); // advisory only, but it makes memory usage much more efficient if it's right...
    if (sz == 0) { // unless it's zero, in which case explictly set a small size
      sz = 5;
    }
    // sz-1 because of taking the *next* power of two
    // means that unmarshal(marshal()) == identity()
    Array* ar = new(vm->getAPool()) Array(sz-1);
    Value* thisval = MKPVAL(vm->getVPool(),ar,KVT_ARRAY);

    // Remember the value in case we get a circular structure
    done.push_back(thisval);
    if ((ukint)circid!=done.size()-1) {
	vm->kaya_internalError(252);
    }
    
    while(x[0]!=']')
    {
	Value* v = unpickle(vm,done,x,id);
	ar->push_back(v);
    }
    ++x;
//    thisval->setPtr(MKVAL(ar,arraytable));
    return thisval;
}

Value* un_union(VMState* vm,vector<Value*>& done,wchar_t* &x, kint id)
{
//    cout << "Doing union " << endl;

    if (*x!='[') vm->kaya_rtsError(INVALID_VALUE);
    ++x;

    kint circid = read_int(vm,x);
    kint tag = read_int(vm,x);
    kint arity = read_int(vm,x);

    Value* thisval;
    if (arity > 0) {
      Union* u;
      u = new(vm->getUPool()) Union(NULL,arity,false);
      thisval = MKPVAL(vm->getVPool(),(void*)u,KVT_UNION+tag);

    // Remember the value in case we get a circular structure
      done.push_back(thisval);
      if ((ukint)circid!=done.size()-1) {
	vm->kaya_internalError(252);	
      }

//    cout << "Adding " << circid << "," << done.size() << endl;

      kint i=0;
      while(i<arity)
      {
	Value* v = unpickle(vm,done,x,id);
	u->args[i]=v;
	++i;
      }
    } else {
      thisval = MKPVAL(vm->getVPool(),(void*)tag,KVT_CUNION);
      // annoyingly necessary this next bit, for now.
      // could change the marshal format so CUNIONs didn't have a
      // circid, perhaps
      done.push_back(thisval);
      if ((ukint)circid!=done.size()-1) {
	vm->kaya_internalError(252);	
      }
    }

    ++x;

    return thisval;
}

Value* un_closure(VMState* vm,vector<Value*>& done,wchar_t* &x, kint id)
{
//    cout << "Doing union " << endl;

    if (*x!='[') vm->kaya_rtsError(INVALID_VALUE);
    ++x;

    kint circid = read_int(vm,x);
    kint fnid = read_int(vm,x);
    kint arity = read_int(vm,x);

    Closure* c = new Closure(NULL,getFn(fnid),arity,arity,false,false);
    Value* thisval = MKPVAL(vm->getVPool(),(void*)c,KVT_FUNC);

    // Remember the value in case we get a circular structure
    done.push_back(thisval);
    if ((ukint)circid!=done.size()-1) {
	vm->kaya_internalError(252);	
    }

//    cout << "Adding " << circid << "," << done.size() << endl;

    kint i=0;
    while(i<arity)
    {
	Value* v = unpickle(vm,done,x,id);
	c->setArg(i,v);
	++i;
    }
    ++x;

    return thisval;
}



